/* matrix/gdl_matrix_ulong.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_MATRIX_ULONG_H__
#define __GDL_MATRIX_ULONG_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_vector.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size1;
  size_t size2;
  size_t tda;
  unsigned long * data;
  gdl_block_ulong * block;
  int owner;
} gdl_matrix_ulong;

typedef struct
{
  gdl_matrix_ulong matrix;
} _gdl_matrix_ulong_view;

typedef _gdl_matrix_ulong_view gdl_matrix_ulong_view;

typedef struct
{
  gdl_matrix_ulong matrix;
} _gdl_matrix_ulong_const_view;

typedef const _gdl_matrix_ulong_const_view gdl_matrix_ulong_const_view;

/* Allocation */

gdl_matrix_ulong * 
gdl_matrix_ulong_alloc (const size_t n1, const size_t n2);

gdl_matrix_ulong * 
gdl_matrix_ulong_calloc (const size_t n1, const size_t n2);

gdl_matrix_ulong * 
gdl_matrix_ulong_alloc_from_block (gdl_block_ulong * b, 
                                   const size_t offset, 
                                   const size_t n1, 
                                   const size_t n2, 
                                   const size_t d2);

gdl_matrix_ulong * 
gdl_matrix_ulong_alloc_from_matrix (gdl_matrix_ulong * m,
                                    const size_t k1, 
                                    const size_t k2,
                                    const size_t n1, 
                                    const size_t n2);

gdl_vector_ulong * 
gdl_vector_ulong_alloc_row_from_matrix (gdl_matrix_ulong * m,
                                        const size_t i);

gdl_vector_ulong * 
gdl_vector_ulong_alloc_col_from_matrix (gdl_matrix_ulong * m,
                                        const size_t j);

void gdl_matrix_ulong_free (gdl_matrix_ulong * m);

/* Views */

_gdl_matrix_ulong_view 
gdl_matrix_ulong_submatrix (gdl_matrix_ulong * m, 
                            const size_t i, const size_t j, 
                            const size_t n1, const size_t n2);

_gdl_vector_ulong_view 
gdl_matrix_ulong_row (gdl_matrix_ulong * m, const size_t i);

_gdl_vector_ulong_view 
gdl_matrix_ulong_column (gdl_matrix_ulong * m, const size_t j);

_gdl_vector_ulong_view 
gdl_matrix_ulong_diagonal (gdl_matrix_ulong * m);

_gdl_vector_ulong_view 
gdl_matrix_ulong_subdiagonal (gdl_matrix_ulong * m, const size_t k);

_gdl_vector_ulong_view 
gdl_matrix_ulong_superdiagonal (gdl_matrix_ulong * m, const size_t k);

_gdl_matrix_ulong_view
gdl_matrix_ulong_view_array (unsigned long * base,
                             const size_t n1, 
                             const size_t n2);

_gdl_matrix_ulong_view
gdl_matrix_ulong_view_array_with_tda (unsigned long * base, 
                                      const size_t n1, 
                                      const size_t n2,
                                      const size_t tda);


_gdl_matrix_ulong_view
gdl_matrix_ulong_view_vector (gdl_vector_ulong * v,
                              const size_t n1, 
                              const size_t n2);

_gdl_matrix_ulong_view
gdl_matrix_ulong_view_vector_with_tda (gdl_vector_ulong * v,
                                       const size_t n1, 
                                       const size_t n2,
                                       const size_t tda);


_gdl_matrix_ulong_const_view 
gdl_matrix_ulong_const_submatrix (const gdl_matrix_ulong * m, 
                                  const size_t i, const size_t j, 
                                  const size_t n1, const size_t n2);

_gdl_vector_ulong_const_view 
gdl_matrix_ulong_const_row (const gdl_matrix_ulong * m, 
                            const size_t i);

_gdl_vector_ulong_const_view 
gdl_matrix_ulong_const_column (const gdl_matrix_ulong * m, 
                               const size_t j);

_gdl_vector_ulong_const_view
gdl_matrix_ulong_const_diagonal (const gdl_matrix_ulong * m);

_gdl_vector_ulong_const_view 
gdl_matrix_ulong_const_subdiagonal (const gdl_matrix_ulong * m, 
                                    const size_t k);

_gdl_vector_ulong_const_view 
gdl_matrix_ulong_const_superdiagonal (const gdl_matrix_ulong * m, 
                                      const size_t k);

_gdl_matrix_ulong_const_view
gdl_matrix_ulong_const_view_array (const unsigned long * base,
                                   const size_t n1, 
                                   const size_t n2);

_gdl_matrix_ulong_const_view
gdl_matrix_ulong_const_view_array_with_tda (const unsigned long * base, 
                                            const size_t n1, 
                                            const size_t n2,
                                            const size_t tda);

_gdl_matrix_ulong_const_view
gdl_matrix_ulong_const_view_vector (const gdl_vector_ulong * v,
                                    const size_t n1, 
                                    const size_t n2);

_gdl_matrix_ulong_const_view
gdl_matrix_ulong_const_view_vector_with_tda (const gdl_vector_ulong * v,
                                             const size_t n1, 
                                             const size_t n2,
                                             const size_t tda);

/* Operations */

unsigned long   gdl_matrix_ulong_get(const gdl_matrix_ulong * m, const size_t i, const size_t j);
void    gdl_matrix_ulong_set(gdl_matrix_ulong * m, const size_t i, const size_t j, const unsigned long x);

unsigned long * gdl_matrix_ulong_ptr(gdl_matrix_ulong * m, const size_t i, const size_t j);
const unsigned long * gdl_matrix_ulong_const_ptr(const gdl_matrix_ulong * m, const size_t i, const size_t j);

void gdl_matrix_ulong_set_zero (gdl_matrix_ulong * m);
void gdl_matrix_ulong_set_identity (gdl_matrix_ulong * m);
void gdl_matrix_ulong_set_all (gdl_matrix_ulong * m, unsigned long x);

gdl_matrix_ulong * gdl_matrix_ulong_fread (FILE * stream) ;
int gdl_matrix_ulong_fwrite (FILE * stream, const gdl_matrix_ulong * m) ;
int gdl_matrix_ulong_fscanf (FILE * stream, gdl_matrix_ulong * m);
int gdl_matrix_ulong_fprintf (FILE * stream, const gdl_matrix_ulong * m, const char * format);
 
int gdl_matrix_ulong_memcpy(gdl_matrix_ulong * dest, const gdl_matrix_ulong * src);
int gdl_matrix_ulong_swap(gdl_matrix_ulong * m1, gdl_matrix_ulong * m2);

int gdl_matrix_ulong_swap_rows(gdl_matrix_ulong * m, const size_t i, const size_t j);
int gdl_matrix_ulong_swap_columns(gdl_matrix_ulong * m, const size_t i, const size_t j);
int gdl_matrix_ulong_swap_rowcol(gdl_matrix_ulong * m, const size_t i, const size_t j);
int gdl_matrix_ulong_transpose (gdl_matrix_ulong * m);
int gdl_matrix_ulong_transpose_memcpy (gdl_matrix_ulong * dest, const gdl_matrix_ulong * src);

unsigned long gdl_matrix_ulong_max (const gdl_matrix_ulong * m);
unsigned long gdl_matrix_ulong_min (const gdl_matrix_ulong * m);
void gdl_matrix_ulong_minmax (const gdl_matrix_ulong * m, unsigned long * min_out, unsigned long * max_out);

void gdl_matrix_ulong_max_index (const gdl_matrix_ulong * m, size_t * imax, size_t *jmax);
void gdl_matrix_ulong_min_index (const gdl_matrix_ulong * m, size_t * imin, size_t *jmin);
void gdl_matrix_ulong_minmax_index (const gdl_matrix_ulong * m, size_t * imin, size_t * jmin, size_t * imax, size_t * jmax);

int gdl_matrix_ulong_isnull (const gdl_matrix_ulong * m);

int gdl_matrix_ulong_add (gdl_matrix_ulong * a, const gdl_matrix_ulong * b);
int gdl_matrix_ulong_sub (gdl_matrix_ulong * a, const gdl_matrix_ulong * b);
int gdl_matrix_ulong_mul_elements (gdl_matrix_ulong * a, const gdl_matrix_ulong * b);
int gdl_matrix_ulong_div_elements (gdl_matrix_ulong * a, const gdl_matrix_ulong * b);
int gdl_matrix_ulong_scale (gdl_matrix_ulong * a, const double x);
int gdl_matrix_ulong_add_constant (gdl_matrix_ulong * a, const double x);
int gdl_matrix_ulong_add_diagonal (gdl_matrix_ulong * a, const double x);

/***********************************************************************/
/* The functions below are obsolete                                    */
/***********************************************************************/
int gdl_matrix_ulong_get_row(gdl_vector_ulong * v, const gdl_matrix_ulong * m, const size_t i);
int gdl_matrix_ulong_get_col(gdl_vector_ulong * v, const gdl_matrix_ulong * m, const size_t j);
int gdl_matrix_ulong_set_row(gdl_matrix_ulong * m, const size_t i, const gdl_vector_ulong * v);
int gdl_matrix_ulong_set_col(gdl_matrix_ulong * m, const size_t j, const gdl_vector_ulong * v);

/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline 
unsigned long
gdl_matrix_ulong_get(const gdl_matrix_ulong * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VAL("first index out of range", GDL_EINVAL, 0) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VAL("second index out of range", GDL_EINVAL, 0) ;
    }
#endif
  return m->data[i * m->tda + j] ;
} 

extern inline 
void
gdl_matrix_ulong_set(gdl_matrix_ulong * m, const size_t i, const size_t j, const unsigned long x)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VOID("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VOID("second index out of range", GDL_EINVAL) ;
    }
#endif
  m->data[i * m->tda + j] = x ;
}

extern inline 
unsigned long *
gdl_matrix_ulong_ptr(gdl_matrix_ulong * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (unsigned long *) (m->data + (i * m->tda + j)) ;
} 

extern inline 
const unsigned long *
gdl_matrix_ulong_const_ptr(const gdl_matrix_ulong * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (const unsigned long *) (m->data + (i * m->tda + j)) ;
} 

#endif

__END_DECLS

#endif /* __GDL_MATRIX_ULONG_H__ */
