/* matrix/oper_source.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

int 
FUNCTION(gdl_matrix, add) (TYPE(gdl_matrix) * a, const TYPE(gdl_matrix) * b)
{
  const size_t M = a->size1;
  const size_t N = a->size2;

  if (b->size1 != M || b->size2 != N)
    {
      GDL_ERROR ("matrices must have same dimensions", GDL_EBADLEN);
    }
  else 
    {
      const size_t tda_a = a->tda;
      const size_t tda_b = b->tda;

      size_t i, j;

      for (i = 0; i < M; i++)
        {
          for (j = 0; j < N; j++)
            {
              a->data[i * tda_a + j] += b->data[i * tda_b + j];
            }
        }
      
      return GDL_SUCCESS;
    }
}

int 
FUNCTION(gdl_matrix, sub) (TYPE(gdl_matrix) * a, const TYPE(gdl_matrix) * b)
{
  const size_t M = a->size1;
  const size_t N = a->size2;

  if (b->size1 != M || b->size2 != N)
    {
      GDL_ERROR ("matrices must have same dimensions", GDL_EBADLEN);
    }
  else 
    {
      const size_t tda_a = a->tda;
      const size_t tda_b = b->tda;

      size_t i, j;

      for (i = 0; i < M; i++)
        {
          for (j = 0; j < N; j++)
            {
              a->data[i * tda_a + j] -= b->data[i * tda_b + j];
            }
        }
      
      return GDL_SUCCESS;
    }
}

int 
FUNCTION(gdl_matrix, mul_elements) (TYPE(gdl_matrix) * a, const TYPE(gdl_matrix) * b)
{
  const size_t M = a->size1;
  const size_t N = a->size2;

  if (b->size1 != M || b->size2 != N)
    {
      GDL_ERROR ("matrices must have same dimensions", GDL_EBADLEN);
    }
  else 
    {
      const size_t tda_a = a->tda;
      const size_t tda_b = b->tda;

      size_t i, j;

      for (i = 0; i < M; i++)
        {
          for (j = 0; j < N; j++)
            {
              a->data[i * tda_a + j] *= b->data[i * tda_b + j];
            }
        }
      
      return GDL_SUCCESS;
    }
}

int 
FUNCTION(gdl_matrix, div_elements) (TYPE(gdl_matrix) * a, const TYPE(gdl_matrix) * b)
{
  const size_t M = a->size1;
  const size_t N = a->size2;

  if (b->size1 != M || b->size2 != N)
    {
      GDL_ERROR ("matrices must have same dimensions", GDL_EBADLEN);
    }
  else 
    {
      const size_t tda_a = a->tda;
      const size_t tda_b = b->tda;

      size_t i, j;

      for (i = 0; i < M; i++)
        {
          for (j = 0; j < N; j++)
            {
              a->data[i * tda_a + j] /= b->data[i * tda_b + j];
            }
        }
      
      return GDL_SUCCESS;
    }
}

int 
FUNCTION(gdl_matrix, scale) (TYPE(gdl_matrix) * a, const double x)
{
  const size_t M = a->size1;
  const size_t N = a->size2;
  const size_t tda = a->tda;
  
  size_t i, j;
  
  for (i = 0; i < M; i++)
    {
      for (j = 0; j < N; j++)
        {
          a->data[i * tda + j] *= x;
        }
    }
  
  return GDL_SUCCESS;
}

int 
FUNCTION(gdl_matrix, add_constant) (TYPE(gdl_matrix) * a, const double x)
{
  const size_t M = a->size1;
  const size_t N = a->size2;
  const size_t tda = a->tda;

  size_t i, j;

  for (i = 0; i < M; i++)
    {
      for (j = 0; j < N; j++)
        {
          a->data[i * tda + j] += x;
        }
    }
  
  return GDL_SUCCESS;
}


int 
FUNCTION(gdl_matrix, add_diagonal) (TYPE(gdl_matrix) * a, const double x)
{
  const size_t M = a->size1;
  const size_t N = a->size2;
  const size_t tda = a->tda;
  const size_t loop_lim = ( M < N ? M : N );
  size_t i;
  for (i = 0; i < loop_lim; i++)
  {
    a->data[i * tda + i] += x;
  }

  return GDL_SUCCESS;
}

ATOMIC
FUNCTION(gdl_matrix, sum_all) (const TYPE(gdl_matrix) * a)
{
  const size_t M = a->size1;
  const size_t N = a->size2;
  size_t i;
  double s = 0;

  for (i = 0; i < M*N; i++)
  {
  	   s += a->data[i];
  }

  return s;
}
