/*  
 * 	meta/factory.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_meta.h>

gdl_meta_tag *
gdl_meta_tag_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		gdl_meta_tag * t = gdl_meta_tag_alloc ();
		
		t->name  = gdl_string_fread (stream);
		GDL_FREAD_STATUS (t->name != NULL, 1);
		t->value = gdl_string_fread (stream);
		GDL_FREAD_STATUS (t->value != NULL, 1);
		
		return t;	
	}	
}


int
gdl_meta_tag_fwrite (FILE * stream, const gdl_meta_tag * t)
{
	if (stream == 0 || t == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		
		status = gdl_string_fwrite (stream, t->name );
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		status = gdl_string_fwrite (stream, t->value);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;	
	}	
}


gdl_meta *
gdl_meta_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t i, size, status;
		gdl_meta * m = gdl_meta_alloc ();
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		for (i = 0; i < size; i++)
		{
			gdl_meta_tag * t = gdl_meta_tag_fread (stream);
			GDL_FREAD_STATUS (t != NULL, 1);
			gdl_meta_add (m, t);
		}
		
		return m;
	}
}


int
gdl_meta_fwrite (FILE * stream, const gdl_meta * m)
{
	if (stream == 0 || m == 0)
		return GDL_EINVAL;
	else
	{
		size_t i, status, size;
		
		size = gdl_list_size (m->tags);
		
		status = fwrite (&size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (size)
		{
			gdl_list_itr * itr
				 = gdl_list_iterator_front (m->tags);
			do
			{
				gdl_meta_tag * t = 
					(gdl_meta_tag *) gdl_list_iterator_value (itr);
				status = gdl_meta_tag_fwrite (stream, t);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
			while (gdl_list_iterator_next (itr));
			
			gdl_list_iterator_free (itr);
		}		
		
		return GDL_SUCCESS;
	}
}

gdl_meta *
gdl_meta_fscanf (FILE * stream)
{
#define TEST_EOF { \
		if (ch == EOF) \
		{ \
			GDL_FREE (tmp); \
			gdl_meta_free (m); \
			return NULL; \
		}};
	
	if (stream == 0)
	{
		return NULL;
	}
	else
	{
		int i, ch;
		int size;
		char * tmp;
		gdl_list_itr * itr;
		gdl_meta * m = gdl_meta_alloc();
		
		tmp = GDL_CALLOC (char, GDL_MAXLINE);
		// size
		ch   = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
		TEST_EOF
		size = atoi(tmp);
		for (i = 0; i < size; i++)
		{
			gdl_meta_tag  * tag;
			ch = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
			TEST_EOF
			tag       = gdl_meta_tag_alloc ();
			tag->name = gdl_string_clone (tmp);
			gdl_meta_add (m, tag);
		}
		// size
		ch   = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
		TEST_EOF
		size = atoi(tmp);
		
		itr = gdl_list_iterator_front (m->tags);
		
		for (i = 0; i < size; i++)
		{
			gdl_meta_tag * tag = (gdl_meta_tag *) gdl_list_iterator_value (itr);
			
			ch = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
			TEST_EOF
			tag->value = gdl_string_clone (tmp);
			
			gdl_list_iterator_next (itr);
		}
		
		gdl_list_iterator_free (itr);
		
		GDL_FREE (tmp);
		
		return m;
	}

#undef TEST_EOF
	
}

int
gdl_meta_name_fprintf (FILE * stream, gdl_meta * m)
{
	if (stream == 0 || m == 0)
	{
		return GDL_EINVAL;
	}
	else
	{
		gdl_meta_tag * tag;
		gdl_list * tags = m->tags;
		gdl_list_itr * itr;
		
		if (gdl_list_size (tags))
		{
			// Write tag names
			itr = gdl_list_iterator_front (tags);
			do
			{
				tag = (gdl_meta_tag *) gdl_list_iterator_value (itr);
				fprintf (stream, "\t%s", tag->name);			
			} while (gdl_list_iterator_next (itr));
			
			gdl_list_iterator_free (itr);
			
			fprintf (stream, "\n");			
		}
	}
	
	return GDL_SUCCESS;
}

int
gdl_meta_value_fprintf (FILE * stream, gdl_meta * m)
{
	if (stream == 0 || m == 0)
	{
		return GDL_EINVAL;
	}
	else
	{
		gdl_meta_tag * tag;
		gdl_list * tags = m->tags;
		gdl_list_itr * itr;
		
		if (gdl_list_size (tags))
		{
			// Write tag values
			itr = gdl_list_iterator_front (tags);
			do
			{
				tag = (gdl_meta_tag *) gdl_list_iterator_value (itr);
				fprintf (stream, "\t%s", tag->value);			
			} while (gdl_list_iterator_next (itr));
			
			gdl_list_iterator_free (itr);
			
			fprintf (stream, "\n");			
		}
	}
	
	return GDL_SUCCESS;
}

int
gdl_meta_fprintf (FILE * stream, gdl_meta * m)
{
	if (stream == 0 || m == 0)
	{
		return GDL_EINVAL;
	}
	else
	{
		gdl_list * tags = m->tags;
		
		fprintf (stream, "%d", gdl_list_size (tags));
		gdl_meta_name_fprintf (stream, m);
		fprintf (stream, "%d", gdl_list_size (tags));
		gdl_meta_value_fprintf (stream, m);
			
	}
	
	return GDL_SUCCESS;
}
