/*  
 *  mosaic/cluster.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_frequency_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_mosaic.h>

gdl_mosaic_cluster *
gdl_mosaic_cluster_alloc (const size_t K, const size_t L, const size_t * NA)
{
	gdl_mosaic_cluster * c;
	
	c = GDL_CALLOC (gdl_mosaic_cluster, 1);
	
	c->K     = K;
	c->L     = L;
	c->NA    = NA;
	c->theta = gdl_frequencies_block_alloc (K, L, NA);
	c->update_theta = gdl_frequencies_block_alloc (K, L, NA);
	c->haplo = gdl_allele_block_alloc (K, L, 1);
	
	return c;
}

void
gdl_mosaic_cluster_free (gdl_mosaic_cluster * c)
{
	if (c)
	{
		gdl_frequencies_block_free (c->theta);
		gdl_frequencies_block_free (c->update_theta);
		gdl_allele_block_free (c->haplo);
		GDL_FREE (c);
	}
}

void
gdl_mosaic_cluster_ran_init (gdl_mosaic_cluster * c, const gdl_rng * rng)
{
	size_t i, j, k;
	double * alpha, * theta;
	
	for (k = 0; k < c->K; k++)
	{
		for (i = 0; i < c->L; i++)
		{
			alpha = GDL_MALLOC (double, c->NA[i]);
			theta = GDL_MALLOC (double, c->NA[i]);
			for (j = 0; j < c->NA[i]; j++)
			{
				alpha[j] = 1;
			}
			gdl_ran_dirichlet (rng, c->NA[i], alpha, theta);
			for (j = 0; j < c->NA[i]; j++)
			{
				gdl_frequencies_block_set (c->theta, k, i, j, theta[j]);
				gdl_frequencies_block_set (c->update_theta, k, i, j, 0);
			}
			GDL_FREE (alpha);
			GDL_FREE (theta);
		}
	}
}

void
gdl_mosaic_cluster_haplo_init (gdl_mosaic_cluster * c, double * mu)
{
	size_t i, j, k, maxj;
	double x, max;
	
	for (k = 0; k < c->K; k++)
	{
		//printf("HAPLO %d", k);
		for (i = 0; i < c->L; i++)
		{
			max=0;
			for (maxj = j = 0; j < c->NA[i]; j++)
			{
				x = gdl_frequencies_block_get (c->theta, k, i, j);
				if (x > max)
				{
					max=x;
					maxj=j;
				}
			}
			gdl_allele_block_set (c->haplo, k, i, 0, maxj);
			mu[i] += (1-max)/(c->K);
			//printf (" %d", gdl_allele_block_get (c->haplo, k, i, 0));
		}
		//printf ("\n");
	}
}

void
gdl_mosaic_cluster_update (const gdl_mosaic_cluster * c, double * abs, double * sq)
{
	size_t l, a, k, n;
	double x, y, s, e, e1=0, e2=0;
	
	for (n = k = 0; k < c->K; k++)
	{
		//printf ("THETA[%d]", k);
		for (l = 0; l < c->L; l++)
		{
			for (s = 0, a = 0; a < c->NA[l]; a++)
			{
				s += gdl_frequencies_block_get (c->update_theta, k, l, a);
			}
			if (s)
			{
				for (a = 0; a < c->NA[l]; a++)
				{
					x = gdl_frequencies_block_get (c->update_theta, k, l, a);
					x /= s;
					if (x < 1.e-10) x = 1.e-10;
					y = gdl_frequencies_block_get (c->theta, k, l, a);
					gdl_frequencies_block_set (c->theta, k, l, a, x);
					gdl_frequencies_block_set (c->update_theta, k, l, a, 0);
					e = fabs(x-y);
					e1 += e;
					e2  += e*e;
					n++;
					//if (a==0) printf (" %1.2f", gdl_frequencies_block_get (c->theta, k, l, a));
				}				
			}
		}
		//printf("\n");
	}
	
	*abs = e1/n;
	*sq  = e2/n;
}
