/*  
 *  mosaic/forward.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_mosaic.h>

void
gdl_mosaic_hmm_forward (gdl_mosaic * hmm, size_t c, size_t p, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p))
{
	size_t j;
	double pr;
	
	pr = gdl_mosaic_hmm_forward_init (hmm, c, p, obs_proba);
	for (j = 1; j < hmm->data->L; j++)
	{
		gdl_mosaic_hmm_forward_induc (hmm, c, p, j, &pr, obs_proba);
	}
}

double
gdl_mosaic_hmm_forward_init (gdl_mosaic * hmm, size_t c, size_t p, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p))
{
	size_t k;
	double x, y;
	
	hmm->util->norm[0] = 0.;
	
	for(k = 0; k < hmm->clust->K; k++)
	{
		x  = (*obs_proba)(hmm, k, c, p, 0);
		y  = hmm->param->f[0][k];
		hmm->util->alpha[0][k] = x * y;
		hmm->util->norm[0] += hmm->util->alpha[0][k];
	}
	for(k = 0; k < hmm->clust->K; k++)
	{
		hmm->util->alpha[0][k] /= hmm->util->norm[0];
	}	
	hmm->util->log = log(hmm->util->norm[0]);
	
	return hmm->util->norm[0];
}

void
gdl_mosaic_hmm_forward_induc (gdl_mosaic * hmm, size_t c, size_t p, size_t l, double * pr, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p))
{
	size_t k, kk;
	double x, t, u, v, w, s;
	
	t = gdl_mosaic_hmm_norec_proba (hmm, l-1, hmm->data->D[l-1]);
	
	s = *pr;
	
	hmm->util->norm[l] = 0.;
	
	for (k = 0; k < hmm->clust->K; k++)
	{
		u = hmm->param->f[l][k];
		x = hmm->util->alpha[l-1][k];
		x *= t;
		x += (1.0-t)*s*u;
		u = (*obs_proba)(hmm, k, c, p, l);
		x *= u;
		hmm->util->alpha[l][k] = x;
		hmm->util->norm[l] += hmm->util->alpha[l][k];
	}
	for (k = 0; k < hmm->clust->K; k++)
	{
		hmm->util->alpha[l][k] /= hmm->util->norm[l];
	}
	
	hmm->util->log += log(hmm->util->norm[l]);
	
	*pr = hmm->util->norm[l];
}

void
gdl_mosaic_hmm_store_fb (gdl_mosaic * hmm, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p))
{
	size_t i, j, k, l;
	
	gdl_mosaic_util_alloc_fb_buffer (hmm->util, hmm->data->NC, hmm->data->P);
	
	for (i = 0; i < hmm->data->NC; i++)
	{
		for (j = 0; j < hmm->data->P; j++)
		{
			gdl_mosaic_hmm_forward (hmm, i, j, obs_proba);
			gdl_mosaic_hmm_backward (hmm, i, j, obs_proba);
			for (l = 0; l < hmm->data->L; l++)
			{
				for (k = 0; k < hmm->clust->K; k++)
				{
					gdl_loci_block_set (hmm->util->alpha_buffer, k, i, l, j, hmm->util->alpha[l][k]);	
					gdl_loci_block_set (hmm->util->beta_buffer, k, i, l, j, hmm->util->beta[l][k]);
				}	
			}
		}	
	}
}
