/*  
 *  mosaic/gdl_mosaic.h
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#ifndef __GDL_MOSAIC_H__
#define __GDL_MOSAIC_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_frequency_block.h>
#include <gdl/gdl_locus_block.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gview_wrapper.h>

__BEGIN_DECLS

// data interface
typedef struct
{
	size_t N;
	size_t NC;
	size_t L;
	size_t P;
	size_t * NA;
	double * D;
	double * W;
	const gdl_chromosome   * chrom;   // the chromosome
	gdl_gview_wrapper      * data;    // genotypic view of the sub data set
	gdl_allele_block * seq;           // the observed haplotypes (nc * l)
	gdl_gvalues_get  * gbuf;
} gdl_mosaic_data;

gdl_mosaic_data * gdl_mosaic_data_alloc (const gdl_chromosome * chrom, gdl_gview_wrapper * data);
void gdl_mosaic_data_free (gdl_mosaic_data * d);
gdl_boolean gdl_mosaic_data_is_missing (const gdl_mosaic_data * d, size_t c, size_t l);
size_t gdl_mosaic_data_get_allele (const gdl_mosaic_data * d, size_t c, size_t l, size_t p);
double gdl_mosaic_data_distance (const gdl_mosaic_data * d, size_t i1, size_t i2);

// cluster interface
typedef struct
{
	size_t K;
	size_t L;
	const size_t * NA;
	gdl_allele_block      * haplo;
	gdl_frequencies_block * theta;
	gdl_frequencies_block * update_theta;
} gdl_mosaic_cluster;

gdl_mosaic_cluster * gdl_mosaic_cluster_alloc (const size_t K, const size_t L, const size_t * NA);
void gdl_mosaic_cluster_free (gdl_mosaic_cluster * d);
void gdl_mosaic_cluster_ran_init (gdl_mosaic_cluster * c, const gdl_rng * rng);
void gdl_mosaic_cluster_haplo_init (gdl_mosaic_cluster * c, double * mu);
void gdl_mosaic_cluster_update (const gdl_mosaic_cluster * c, double * abs, double * sq);
// parameter interface
typedef struct
{
	size_t K;
	size_t L;
	const double * D;
	double * rho;
	double * mu;
	double ** f;
	double * update_rho;
	double * update_mu;
	double ** update_f;
	double * rho_count;
} gdl_mosaic_params;

gdl_mosaic_params * gdl_mosaic_params_alloc (const size_t K, const size_t L, const double * D);
void gdl_mosaic_params_free (gdl_mosaic_params * p);
void gdl_mosaic_params_ran_init (gdl_mosaic_params * p, const gdl_rng * rng);
void gdl_mosaic_params_update (const gdl_mosaic_params * c, gdl_boolean rho_cst, gdl_boolean freq_cst, gdl_boolean mu_cst, double * abs, double * sq);

typedef struct
{
	size_t L;
	size_t K;
	// forward-backward buffer
	double log;
	double * norm;
	double ** alpha;
	double ** beta;
	// viterbi buffer
	double * d;
	double * m;
	size_t ** phi;
	gdl_loci_block * alpha_buffer;
	gdl_loci_block * beta_buffer;
} gdl_mosaic_util;

gdl_mosaic_util * gdl_mosaic_util_alloc (const size_t K, const size_t L);
void gdl_mosaic_util_free (gdl_mosaic_util * u);
void gdl_mosaic_util_alloc_fb_buffer (gdl_mosaic_util * u, const size_t N, const size_t P);

typedef struct
{
	double log;
	double abs;
	double sq;
	gdl_mosaic_data    * data;
	gdl_mosaic_cluster * clust;
	gdl_mosaic_params  * param;
	gdl_mosaic_util    * util;
	const gdl_rng      * rng;
	FILE * logger;
} gdl_mosaic;

gdl_mosaic * gdl_mosaic_alloc (const size_t K, const gdl_chromosome * chrom, gdl_gview_wrapper * data, const gdl_rng * rng);
void gdl_mosaic_free (gdl_mosaic * hmm);

void gdl_mosaic_ran_init (gdl_mosaic * m);
int gdl_mosaic_iterate_theta (gdl_mosaic * m, gdl_boolean rho_cst, gdl_boolean freq_cst);
int gdl_mosaic_iterate_haplo (gdl_mosaic * m, gdl_boolean rho_cst, gdl_boolean freq_cst, gdl_boolean mu_cst);
void gdl_mosaic_hmm_update_theta (gdl_mosaic * m, const size_t c, const size_t p);
void gdl_mosaic_hmm_update_haplo (gdl_mosaic * m, const size_t c, const size_t p);
int gdl_mosaic_perform (gdl_mosaic * m, gdl_boolean rho_cst, gdl_boolean freq_cst, gdl_boolean mu_cst, const double eps, const size_t max);
int gdl_mosaic_imputation_iterate (gdl_mosaic * m, double * abs_res, double * sq_res, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
int gdl_mosaic_imputation (gdl_mosaic * m, const double eps, const size_t max, size_t mode);
/*
 * HMM interface
 */
/*
 * Compute the loglikelihood of the data
 */
void gdl_mosaic_hmm_store_fb (gdl_mosaic * hmm, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
// forward
void   gdl_mosaic_hmm_forward (gdl_mosaic * hmm, size_t c, size_t p, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
double gdl_mosaic_hmm_forward_init (gdl_mosaic * hmm, size_t c, size_t p, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
void   gdl_mosaic_hmm_forward_induc (gdl_mosaic * hmm, size_t c, size_t p, size_t l, double * pr, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
// bacward
void   gdl_mosaic_hmm_backward (gdl_mosaic * hmm, size_t c, size_t p, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
double gdl_mosaic_hmm_backward_init (gdl_mosaic * hmm, size_t c, size_t p, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
void   gdl_mosaic_hmm_backward_induc (gdl_mosaic * hmm, size_t c, size_t p, size_t l, double * pr, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
// viterbi
gdl_vector_uint * gdl_mosaic_hmm_viterbi (gdl_mosaic * hmm, size_t c, size_t p, double (*obs_proba)(gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p));
// proba
double gdl_mosaic_hmm_obs_proba_theta (gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p);
double gdl_mosaic_hmm_obs_proba_haplo (gdl_mosaic * hmm, size_t k, size_t c, size_t l, size_t p);
double gdl_mosaic_hmm_norec_proba (gdl_mosaic * hmm, size_t l, double d);
double gdl_mosaic_hmm_trans_proba (gdl_mosaic * hmm, size_t l, size_t k1, size_t k2, double d);
gdl_vector * gdl_mosaic_interval_cluster_proba (gdl_mosaic * hmm, size_t i, size_t l, size_t p, double position);

__END_DECLS

#endif /* __GDL_MOSAIC_HMM__ */
