/*  
 *  mosaic/param.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_nan.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_mosaic.h>

static void
gdl_mosaic_hmm_missing_site_update_haplo (const gdl_mosaic * hmm,
                                    size_t im,
                                    size_t p,
                                    size_t l,
                                    size_t j,
                                    double * abs_res,
                                    double * sq_res,
                                    size_t * nres)
{
	size_t k, m, anc;
	double tmp, tot, u, v, mu, * nt;
	gdl_gvalues * t;
	
	t = gdl_gview_wrapper_missing_hget (hmm->data->data, im, p, j);
	nt = GDL_CALLOC (double, t->size);
	mu = hmm->param->mu[l];
	
	for (tot = k = 0; k < hmm->clust->K; k++)
	{
		anc = gdl_allele_block_get (hmm->clust->haplo, k, l, 0);
		u   = hmm->util->alpha[l][k]*hmm->util->beta[l][k];
		for (m = 0; m < t->size; m++)
		{
			if (t->values[m]->idx == anc)
			{
				v = u*(1.0-mu);
			}
			else
			{
				v = u*mu/((double)hmm->data->NA[l]-1);
			}
			nt[m]  += v;
			tot    += v;
		}
	}
	for (k = 0; k < t->size; k++)
	{
		u   = nt[k]/tot;
		tmp = (t->values[k]->value-u);
		(*abs_res) += fabs(tmp);
		(*sq_res)  += tmp*tmp;
		(*nres)++;
		t->values[k]->value = u;
	}
	GDL_FREE (nt);
}

static void
gdl_mosaic_hmm_missing_site_update_theta (const gdl_mosaic * hmm,
                                    size_t im,
                                    size_t p,
                                    size_t l,
                                    size_t j,
                                    double * abs_res,
                                    double * sq_res,
                                    size_t * nres)
{
	size_t k, m;
	double tmp, tot, u, v, * nt;
	gdl_gvalues * t;
	
	t  = gdl_gview_wrapper_missing_hget (hmm->data->data, im, p, j);
	nt = GDL_CALLOC (double, t->size);
	
	for (tot = k = 0; k < hmm->clust->K; k++)
	{
		u   = hmm->util->alpha[l][k]*hmm->util->beta[l][k];
		for (m = 0; m < t->size; m++)
		{
			v = u*gdl_frequencies_block_get (hmm->clust->theta, k, l, t->values[m]->idx);
			nt[m]  += v;
			tot    += v;
		}
	}
	for (k = 0; k < t->size; k++)
	{
		u   = nt[k]/tot;
		tmp = (t->values[k]->value-u);
		(*abs_res) += fabs(tmp);
		(*sq_res)  += tmp*tmp;
		(*nres)++;
		t->values[k]->value = u;
	}
	GDL_FREE (nt);
}

static int 
gdl_mosaic_hmm_missing_individual_update (gdl_mosaic * hmm, 
                                          size_t im,
                                          size_t p,
                                          double * abs_res,
                                          double * sq_res,
                                          double (*proba) (gdl_mosaic *, size_t , size_t , size_t , size_t ),
                                          void (*missing) (const gdl_mosaic *, size_t , size_t , size_t , size_t , double * abs_res, double * sq_res, size_t *))
{
	size_t ic, j, k, l, m, nml, nres=0;
	const size_t * lidx;	
	
	ic   = gdl_gview_wrapper_missing_accession_idx_c (hmm->data->data, im);
	nml  = gdl_gview_wrapper_missing_hlocus_size (hmm->data->data, im, p);
	lidx = gdl_gview_wrapper_missing_hlocus_idx (hmm->data->data, im, p);
	
	gdl_mosaic_hmm_forward (hmm, ic, p, proba);
	gdl_mosaic_hmm_backward (hmm, ic, p, proba);
	
	*abs_res = *sq_res = 0;
	
	for (j = 0; j < nml; j++)
	{
		(*missing)(hmm, im, p, lidx[j], j, abs_res, sq_res, &nres);
	}
	
	*abs_res /= nres;
	*sq_res  /= nres;
	
	return GDL_SUCCESS;
} 
 
int
gdl_mosaic_imputation_iterate (gdl_mosaic * hmm, double * abs_res, double * sq_res, double (*proba) (gdl_mosaic *, size_t , size_t , size_t , size_t ))
{
	double abs=0, sq=0;
	
	if (gdl_gview_wrapper_missing_size (hmm->data->data))
	{
		size_t i, j, k, l, nm, hna;
		
		nm = gdl_gview_wrapper_missing_accession_size (hmm->data->data);
		for (k = i = 0; i < nm; i++)
		{
			for (j = 0; j < hmm->data->P; j++)
			{
				if (proba == &gdl_mosaic_hmm_obs_proba_theta)
				{
					gdl_mosaic_hmm_missing_individual_update (hmm, i, j, &abs, &sq, proba, &gdl_mosaic_hmm_missing_site_update_theta);	
				}
				else
				{
					gdl_mosaic_hmm_missing_individual_update (hmm, i, j, &abs, &sq, proba, &gdl_mosaic_hmm_missing_site_update_haplo);
				}
				
			}
		}
	}
	*abs_res = abs;
	*sq_res  = sq;
}
