/*  
 *  mosaic/param.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_nan.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_mosaic.h>

gdl_mosaic_params *
gdl_mosaic_params_alloc (const size_t K, const size_t L, const double * D)
{
	size_t i;
	gdl_mosaic_params * p;
	
	p = GDL_MALLOC (gdl_mosaic_params, 1);
	
	p->K   = K;
	p->L   = L;
	p->D   = D;
	p->rho = GDL_CALLOC (double, L-1);
	p->mu  = GDL_CALLOC (double, L);
	p->f   = GDL_CALLOC (double * , L);
	p->update_rho = GDL_CALLOC (double, L-1);
	p->rho_count  = GDL_CALLOC (double, L-1);
	p->update_mu  = GDL_CALLOC (double, L);
	p->update_f   = GDL_CALLOC (double * , L);
	for (i = 0; i < L; i++)
	{
		p->f[i] = GDL_CALLOC (double, K);
		p->update_f[i] = GDL_CALLOC (double, K);
	}
	
	return p;
} 

void
gdl_mosaic_params_free (gdl_mosaic_params * p)
{
	if (p)
	{
	 	size_t i;
	 	for (i = 0; i < p->L; i++)
	 	{
	 		GDL_FREE (p->f[i]);
	 		GDL_FREE (p->update_f[i]);
	 	}
	 	GDL_FREE (p->f);
	 	GDL_FREE (p->rho);
	 	GDL_FREE (p->mu);
	 	GDL_FREE (p->update_f);
	 	GDL_FREE (p->update_rho);
	 	GDL_FREE (p->rho_count);
	 	GDL_FREE (p->update_mu);
	 	GDL_FREE (p);
	} 	
}

void
gdl_mosaic_params_ran_init (gdl_mosaic_params * p, const gdl_rng * rng)
{
	size_t i, j, k;
	double * alpha;
	
	alpha = GDL_MALLOC (double, p->K);
	for (k = 0; k < p->K; k++)
	{
		alpha[k]=1;
	}
	for (i = 0; i < p->L; i++)
	{
		if (i) p->rho[i-1] = 1.e-5 + gdl_ran_gaussian (rng, 1.e-10);
		p->mu[i] = 1.e-3 + gdl_ran_gaussian (rng, 1.e-6);
		//gdl_ran_dirichlet (rng, p->K, alpha, p->f[i]);
		for (k = 0; k < p->K; k++)
		{
			p->f[i][k]=1.0/p->K;
		}
	}
	GDL_FREE (alpha);
}

void
gdl_mosaic_params_update (const gdl_mosaic_params * p, gdl_boolean rho_cst, gdl_boolean freq_cst, gdl_boolean mu_cst, double * abs, double * sq)
{
	size_t l, k;
	double x, y, s, e, e1=0, e2=0, rho=0, tmu=0, mu=0, * fcst;
	
	if (freq_cst)
	{
		fcst = GDL_CALLOC (double, p->K);
	}
	
	for (l = 0; l < p->L; l++)
	{
		if (l)
		{
			x = (p->K > 1) ? p->update_rho[l-1]/p->rho_count[l-1] : 0;
			x = -log (1-x);
			if (p->D[l-1]) x /= p->D[l-1];
			if (rho_cst)
			{
				rho += p->rho[l-1];
			}
			else
			{
				e = fabs(x-p->rho[l-1]);
				p->rho[l-1] = x;
				e1 += e/p->L;
				e2 += e*e/p->L;
			}
			p->update_rho[l-1] = 0;
			p->rho_count[l-1]  = 0;
		}
		for (s = k = 0; k < p->K; k++)
		{
			s += p->update_f[l][k];
		}
		if (s)
		{
			for (k = 0; k < p->K; k++)
			{
				x = p->update_f[l][k]/s;
				if (freq_cst)
				{
					fcst[k] += x;
				}
				else
				{
					if (x < 1.e-10) x = 1.e-10;
					e = fabs(x-p->f[l][k]);
					p->f[l][k] = x;
					e1 += e/(p->K*p->L);
					e2 += e*e/(p->K*p->L);
				}
				p->update_f[l][k] = 0;
			}
			x = p->update_mu[l]/s;
			if (mu_cst)
			{
				mu  += x;
				tmu += s;	
			}
			else
			{
				e = fabs(x-p->mu[l]);
				p->mu[l] = x; // minus missing sites 
				e1 += e/p->L;
				e2 += e*e/p->L;
			}
			p->update_mu[l] = 0;	
		}
	}
	if (rho_cst || mu_cst || freq_cst)
	{
		for (l = 0; l < p->L; l++)
		{
			if (l && rho_cst) 
			{
				if (l==1)
				{
					e = fabs(rho/(p->L-1)-p->rho[l-1]);
					e1 += e;
					e2 += e*e;
				}
				p->rho[l-1] = rho / (p->L-1);
			}
			if (mu_cst) 
			{
				if (!l)
				{
					e = fabs(mu/tmu-p->mu[l]);
					e1 += e;
					e2 += e*e;
				}
				p->mu[l] = mu / tmu;
			}
			if (freq_cst)
			{
				for (k = 0; k < p->K; k++)
				{
					if (!l)
					{
						e = fabs(fcst[k]/p->L-p->f[l][k]);
						e1 += e/p->K;
						e2 += e*e/p->K;
					}
					p->f[l][k] = fcst[k]/p->L;					
				}
				if (l == p->L-1)
				{
					GDL_FREE (fcst);
				}
			}
		}
	}
	
	*abs += e1;
	*sq  += e2;
}
