/*  
 *  mosaic/plot.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <math.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_glabels.h>
#include <gdl/gdl_graphics.h>
#include <gdl/gdl_plot.h>
#include <gdl/gdl_mosaic.h>
#include <gdl/gdl_mosaic_result.h>

static const int GDL_MOSAIC_PLOT_GENOME              = 0;
static const int GDL_MOSAIC_PLOT_CHROMOSOME          = 0;
static const double GDL_MOSAIC_PLOT_LOCUS_HEIGHT     = 10;
static const double GDL_MOSAIC_PLOT_LOCUS_WIDTH      = 5;
static const double GDL_MOSAIC_PLOT_CHROM_GAP        = 25;
static const int GDL_MOSAIC_PLOT_PHYSIC_DISTANCE     = gdl_gdistance_base;
static const int GDL_MOSAIC_PLOT_GENETIC_DISTANCE    = gdl_gdistance_centi_morgan;
static const double GDL_MOSAIC_PLOT_PHYSIC_SCALE     = 100;
static const double GDL_MOSAIC_PLOT_GENETIC_SCALE    = 1;
static const double GDL_MOSAIC_PLOT_VERTICAL_SPACE   = 25;
static const double GDL_MOSAIC_PLOT_CHART_HEIGHT     = 100;
static const double GDL_MOSAIC_PLOT_LABEL_WIDTH      = 100;
static const double GDL_MOSAIC_PLOT_ANCESTRAL_CEX    = 5;
static const double GDL_MOSAIC_PLOT_ACCESSION_CEX    = 1;
static const int GDL_MOSAIC_PLOT_ACCESSION_LABEL     = 1;
static const double GDL_MOSAIC_PLOT_BLOCK_WIDTH      = 25;
static const int GDL_MOSAIC_PLOT_WITH_MUTATION       = 1;
static const int GDL_MOSAIC_PLOT_BEST_BLOCK_PATH     = 1;
static const int GDL_MOSAIC_PLOT_ACCESSION_FAST      = 0;
static const double GDL_MOSAIC_PLOT_ACCESSION_SPACE  = 10;
static const int GDL_MOSAIC_PLOT_SHOW_MISSING        = 0;

typedef struct
{
	double height;
	double width;
	double * xlocus;
	double xwidth;
	gdl_vector * rho;
	gdl_vector * mut;
	gdl_gdistance_type * tlocus;
	const gdl_mosaic_result * result;	
} gdl_mosaic_plot_t;

static void
gdl_mosaic_plot_default_parameters (gdl_plot_parameters * params)
{
	gdl_plot_parameter * par;
	
	par = gdl_plot_parameters_get (params, "genome");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_GENOME);
		gdl_plot_parameters_set (params, "genome", par);
	}
	par = gdl_plot_parameters_get (params, "chromosome");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_CHROMOSOME);
		gdl_plot_parameters_set (params, "chromosome", par);
	}
	par = gdl_plot_parameters_get (params, "locus-height");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_LOCUS_HEIGHT);
		gdl_plot_parameters_set (params, "locus-height", par);
	}
	par = gdl_plot_parameters_get (params, "locus-width");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_LOCUS_WIDTH);
		gdl_plot_parameters_set (params, "locus-width", par);
	}
	par = gdl_plot_parameters_get (params, "locus-color");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, 1);
		gdl_plot_parameter_set (par, 0, gdl_rgb_color_get_black ());
		gdl_plot_parameters_set (params, "locus-color", par);
	}
	par = gdl_plot_parameters_get (params, "chrom-gap");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_CHROM_GAP);
		gdl_plot_parameters_set (params, "chrom-gap", par);
	}
	par = gdl_plot_parameters_get (params, "physic-distance");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_PHYSIC_DISTANCE);
		gdl_plot_parameters_set (params, "physic-distance", par);
	}
	par = gdl_plot_parameters_get (params, "genetic-distance");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_PHYSIC_DISTANCE);
		gdl_plot_parameters_set (params, "genetic-distance", par);
	}
	par = gdl_plot_parameters_get (params, "physic-scale");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_PHYSIC_SCALE);
		gdl_plot_parameters_set (params, "physic-scale", par);
	}
	par = gdl_plot_parameters_get (params, "genetic-scale");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_GENETIC_SCALE);
		gdl_plot_parameters_set (params, "genetic-scale", par);
	}
	par = gdl_plot_parameters_get (params, "vertical-space");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_VERTICAL_SPACE);
		gdl_plot_parameters_set (params, "vertical-space", par);
	}
	par = gdl_plot_parameters_get (params, "accession-space");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_ACCESSION_SPACE);
		gdl_plot_parameters_set (params, "accession-space", par);
	}
	par = gdl_plot_parameters_get (params, "chart-height");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_CHART_HEIGHT);
		gdl_plot_parameters_set (params, "chart-height", par);
	}
	par = gdl_plot_parameters_get (params, "label-width");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_LABEL_WIDTH);
		gdl_plot_parameters_set (params, "label-width", par);
	}
	par = gdl_plot_parameters_get (params, "ancestral-cex");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_ANCESTRAL_CEX);
		gdl_plot_parameters_set (params, "ancestral-cex", par);
	}
	par = gdl_plot_parameters_get (params, "accession-cex");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_ACCESSION_CEX);
		gdl_plot_parameters_set (params, "accession-cex", par);
	}
	par = gdl_plot_parameters_get (params, "accession-font");
	if (!par)
	{
		gdl_font * font = gdl_font_default ();
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_font, 1);
		gdl_plot_parameter_set (par, 0, font);
		gdl_plot_parameters_set (params, "accession-font", par);
	}
	par = gdl_plot_parameters_get (params, "accession-label");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_ACCESSION_LABEL);
		gdl_plot_parameters_set (params, "accession-label", par);
	}
	par = gdl_plot_parameters_get (params, "block-width");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_BLOCK_WIDTH);
		gdl_plot_parameters_set (params, "block-width", par);
	}
	par = gdl_plot_parameters_get (params, "with-mutation");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_WITH_MUTATION);
		gdl_plot_parameters_set (params, "with-mutation", par);
	}
	par = gdl_plot_parameters_get (params, "mutation-color");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, 1);
		gdl_plot_parameter_set (par, 0, gdl_rgb_color_get_black ());
		gdl_plot_parameters_set (params, "mutation-color", par);
	}
	par = gdl_plot_parameters_get (params, "best-block-path");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_BEST_BLOCK_PATH);
		gdl_plot_parameters_set (params, "best-block-path", par);
	}
	par = gdl_plot_parameters_get (params, "accession-fast");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_ACCESSION_FAST);
		gdl_plot_parameters_set (params, "accession-fast", par);
	}
	par = gdl_plot_parameters_get (params, "ignore-missing");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_int, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_MOSAIC_PLOT_SHOW_MISSING);
		gdl_plot_parameters_set (params, "ignore-missing", par);
	}
//	par = gdl_plot_parameters_get (params, "missing-color");
//	if (!par)
//	{
//		par = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, 1);
//		gdl_plot_parameter_set (par, 0, gdl_rgb_color_get_white ());
//		gdl_plot_parameters_set (params, "missing-color", par);
//	}
}

static void
gdl_mosaic_plot_create_chromosome (gdl_mosaic_plot_t * p, gdl_plot_parameters * par)
{
	size_t i, j, l, b, np, nb, nl;
	int * pd, * gd;
	double x, * lh, * lw, * cg, * ps, * gs, * bw;
	const gdl_mosaic_result * presult;
	gdl_gdistance * distance;
	
	lh  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	lw  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	bw  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "block-width"), 0);
	cg  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "chrom-gap"), 0);
	ps  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "physic-scale"), 0);
	gs  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "genetic-scale"), 0);
	pd  = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "physic-distance"), 0);
	gd  = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "genetic-distance"), 0);
	
	p->xlocus[0] = p->width;
	for (j = 1; j < p->result->L; j++)
	{
		p->xlocus[j] = p->xlocus[j-1] + *lw + p->result->D[j-1]*(*ps);
	}
	j--;
	p->xwidth  = p->xlocus[j]-p->xlocus[0]+*lw;
	p->width  += p->xlocus[j]+*lw;
	p->height += *lh;
}

static void
gdl_mosaic_plot_create_recomb (gdl_mosaic_plot_t * p, gdl_plot_parameters * par, size_t mode)
{
	size_t i, j, l, np, nl, nlt;
	int * pd, * gd;
	double x, max, cmax, * r, * lh, * lw, * ch;
	gdl_vector * rho;
	
	lh  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	lw  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	ch  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "chart-height"), 0);
	
	rho  = gdl_vector_calloc (p->result->L-1);
	for (i = 0; i < p->result->L-1; i++)
	{
		gdl_vector_set (rho, l, p->result->rho[i]);
	}
	max   = gdl_vector_max (rho);
	cmax  = max*1.1; 
	gdl_vector_scale (rho, -(*ch)/cmax);
	gdl_vector_add_constant (rho, (*ch));
	
	p->rho  = rho;
	
	p->height += *ch;
}

static void
gdl_mosaic_plot_create_mutation (gdl_mosaic_plot_t * p, gdl_plot_parameters * par)
{
	size_t i, j, l, np, nl, nlt;
	int * pd, * gd;
	double x, max, cmax, * r, * lh, * lw, * ch;
	gdl_vector * mut;
	
	lh  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	lw  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	ch  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "chart-height"), 0);
	
	mut  = gdl_vector_alloc (p->result->L);
	
	for (i = 0; i < p->result->L; i++)
	{
		gdl_vector_set (mut, l, p->result->mu[i]);
	}
	
	max   = gdl_vector_max (mut);
	cmax  = max*1.1; 
	gdl_vector_scale (mut, -(*ch)/cmax);
	gdl_vector_add_constant (mut, (*ch));
	
	p->mut = mut;
	
	p->height += *ch;
}

static void
gdl_mosaic_plot_create_ancestral (gdl_mosaic_plot_t * p, gdl_plot_parameters * par)
{
	size_t i, b, j, k, nb, np, nl, na, nk, maxa, maxk;
	double * lh, * sp, * ac;
	gdl_plot_parameter                 * acolors;
	gdl_plot_parameter                 * hcolors;
	gdl_rgb_color_vector               * palette;
	
	lh      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	ac      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "ancestral-cex"), 0);
	sp      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "vertical-space"), 0);
	acolors = gdl_plot_parameters_get (par, "allele-color");
	hcolors = gdl_plot_parameters_get (par, "ancestral-color");
	
	for (maxa = i = 0; i < p->result->L; i++)
	{
		for (k = 0; k < p->result->K; k++)
		{
			size_t allele = gdl_allele_block_get (p->result->haplo, k, i, 0);
			if (allele+1 > maxa)
			{
				maxa = allele+1;
			}
		}
	}
	
	palette = gdl_rgb_color_palette (maxa+p->result->K);
	
	j = 0;
	
	if (hcolors==NULL || gdl_plot_parameter_size (hcolors) < p->result->K)
	{
		hcolors = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, p->result->K);
		for (i = 0; i < p->result->K; i++, j++)
		{
			gdl_plot_parameter_set_copy (hcolors, i, palette->colors[i]);
		}
		gdl_plot_parameters_set (par, "ancestral-color", hcolors);
	}
	if (acolors==NULL || gdl_plot_parameter_size (acolors) < maxa)
	{
		acolors = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, maxa);
		for (i = 0; i < maxa; i++)
		{
			gdl_plot_parameter_set_copy (acolors, i, palette->colors[j++]);
		}
		gdl_plot_parameters_set (par, "allele-color", acolors);
	}
	
	gdl_rgb_color_vector_free (palette);
	
	p->height += 2*p->result->K*((*lh)*(*ac))+(p->result->K-1)*(*sp)/4;
}

static void
gdl_mosaic_plot_create_accession (gdl_mosaic_plot_t * p, gdl_plot_parameters * par)
{
	size_t i, j, np, na, pl;
	int * wm;
	double * lh, * sp, * ac;
	gdl_plot_parameter * alist;
	
	lh      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	sp      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "accession-space"), 0);
	ac      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "accession-cex"), 0);
	wm      = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "with-mutation"), 0);
	
	pl   = p->result->P;
	
	alist = gdl_plot_parameters_get (par, "accession-list");
	
	if (!alist)
	{
		size_t * index;
		
		na = p->result->N;
		
		index = gdl_mosaic_result_sort_accession (p->result);
		
		alist = gdl_plot_parameter_alloc (gdl_plot_parameter_string, na);
		
		for (i = 0; i < na; i++)
		{
			const gdl_string * name = gdl_glabels_accession (p->result->labels, index[i]);
			//printf ("ACCESSION %d => %d (%s) [%d]\n", i, index[i], name, na);
			gdl_plot_parameter_set_copy (alist, i, name);
		}
		
		gdl_plot_parameters_set (par, "accession-list", alist);
		
		GDL_FREE (index);
	}
	
	na = gdl_plot_parameter_size (alist);
	
	if (*wm)
	{
		p->height += (pl*na*(*lh*2)+(pl*na)*(*sp)/6 + na*(*sp)/4)*(*ac);
	}
	else
	{
		p->height += (pl*na*(*lh)+(pl*na)*(*sp)/6 + na*(*sp)/4)*(*ac);
	}
	
}

static int
gdl_mosaic_plot_create (gdl_mosaic_plot_t * p, gdl_plot_parameters * par)
{
	double * sp, * la;
	
	la = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "label-width"), 0);
	sp = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "vertical-space"), 0);
	
	p->height = 0;
	
	p->width  = *la;
	
	gdl_mosaic_plot_create_chromosome (p, par);
	
	p->height += *sp;
	
	gdl_mosaic_plot_create_recomb (p, par, 0);
	
	p->height += *sp;
	
	gdl_mosaic_plot_create_mutation (p, par);
	
	p->height += *sp;
	
	gdl_mosaic_plot_create_ancestral (p, par);
	
	p->height += *sp;
	
	gdl_mosaic_plot_create_accession (p, par);
	
	//printf ("CREATE [ %g, %g ]\n", p->height, p->width);
}

static int
gdl_mosaic_plot_alloc (void * plot, gdl_plot_parameters * par, const void * data, double * width, double * height)
{
	int * g, * c;
	gdl_mosaic_plot_t * p;
	
	p = (gdl_mosaic_plot_t *) plot;
	
	p->result = (gdl_mosaic_result *) data;
	
	gdl_mosaic_plot_default_parameters (par);
	
	g = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "genome"), 0);
	c = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "chromosome"), 0);
	
	p->xlocus  = GDL_MALLOC (double, p->result->L);
	p->tlocus  = GDL_MALLOC (gdl_gdistance_type, p->result->L);
	
	gdl_mosaic_plot_create (p, par);
	
	*width  = p->width;
	*height = p->height;
	
	return GDL_SUCCESS;
}

static int
gdl_mosaic_plot_free (void * plot)
{
	if (plot)
	{
		gdl_mosaic_plot_t * p = (gdl_mosaic_plot_t *) plot;
		GDL_FREE (p->xlocus);
		GDL_FREE (p->tlocus);
	}
}

static double
gdl_mosaic_plot_draw_chromosome (gdl_mosaic_plot_t * p, const gdl_plot_parameters * par, gdl_graphics2D * graph, double y)
{
	size_t i, j, l, b, np, nl, nb;
	double x, * lh, * lw;
	gdl_rgb_color * color;
	
	lh    = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	lw    = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	color = (gdl_rgb_color *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-color"), 0);
	
	gdl_graphics2D_set_color (graph, color);
	gdl_graphics2D_set_paint (graph, color);
	
	for (i = 0; i < p->result->L; i++)
	{
		gdl_graphics2D_fill_rect (graph, p->xlocus[i], y, *lw, *lh);
	}
	
	return y+(*lw);
}

static double
gdl_mosaic_plot_draw_recomb (gdl_mosaic_plot_t * p, const gdl_plot_parameters * par, gdl_graphics2D * graph, double y, size_t mode)
{
	size_t i, nl;
	double x1, y1, x2, y2, * ch, * lw;
	gdl_rgb_color * color;
	const gdl_vector * rho;
	
	ch    = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "chart-height"), 0);
	lw    = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	
	rho   = p->rho;
	
	for (i = 0; i < p->result->L - 1; i++)
	{
		if (i)
		{
			x1 = x2 = p->xlocus[i]+*lw;
			y1 = y+gdl_vector_get (rho, i-1);
			y2 = y+gdl_vector_get (rho, i);
		}
		else
		{
			x1 = x2 = p->xlocus[i];
			y1 = y+*ch;
			y2 = y+gdl_vector_get (rho, i);
		}
		gdl_graphics2D_draw_line (graph, x1, y1, x2, y2);
		x2 = p->xlocus[i+1]+*lw;
		y1 = y2;
		gdl_graphics2D_draw_line (graph, x1, y1, x2, y2);		 
	}
	gdl_graphics2D_draw_line (graph, x2, y2, x2, y+*ch);
	gdl_graphics2D_draw_rect (graph, p->xlocus[0], y, p->xwidth, *ch);
	
	return y+(*ch);
}

static double
gdl_mosaic_plot_draw_mutation (gdl_mosaic_plot_t * p, const gdl_plot_parameters * par, gdl_graphics2D * graph, double y)
{
	size_t i, nl;
	double x1, y1, x2, y2, * ch, * lw;
	gdl_rgb_color * color;
	const gdl_vector * mut;
	
	ch    = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "chart-height"), 0);
	lw    = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	
	mut   = p->mut;
	
	for (i = 0; i < p->result->L; i++)
	{
		x1 = x2 = p->xlocus[i];
		y1 = y+*ch;
		y2 = y+gdl_vector_get (mut, i);
		gdl_graphics2D_draw_line (graph, x1, y1, x2, y2);
		x2 += *lw;
		y1 = y2;
		gdl_graphics2D_draw_line (graph, x1, y1, x2, y2);
		x1 = x2;
		y2 = y+*ch;
		gdl_graphics2D_draw_line (graph, x1, y1, x2, y2);
	}
	gdl_graphics2D_draw_rect (graph, p->xlocus[0], y, p->xwidth, *ch);
	
	return y+(*ch);
}

static double
gdl_mosaic_plot_draw_ancestral (gdl_mosaic_plot_t * p, const gdl_plot_parameters * par, gdl_graphics2D * graph, double y)
{
	size_t i, j, b, k, l, nb, np, nl, nk, allele;
	double x1, x2, y1, y2, * y3, * y4, w, h, pr, * lh, * lw, * sp, * ac, * bw;
	gdl_plot_parameter * acolors, * hcolors;
	
	lh      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	lw      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	bw      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "block-width"), 0);
	ac      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "ancestral-cex"), 0);
	sp      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "vertical-space"), 0);
	acolors = gdl_plot_parameters_get (par, "allele-color");
	hcolors = gdl_plot_parameters_get (par, "ancestral-color");
	
	gdl_graphics2D_set_stroke (graph, 2, 0, NULL);
	
	y3      = GDL_CALLOC (double, p->result->K);
	y4      = GDL_CALLOC (double, p->result->K);
	
	for (i = 0; i < p->result->L; i++)
	{
		for (k = 0; k < p->result->K; k++)
		{
			allele = gdl_allele_block_get (p->result->haplo, k, i, 0);
			pr     = p->result->f[i][k];
			w      = sqrt(pr)*(*lw);
			h      = sqrt(pr)*((*lh)*(*ac));
			x1     = p->xlocus[i] + 0.5*(*lw) - 0.5*w;
			y1     = y + 2*k*((*lh)*(*ac))+k*(*sp)/4+((*lh)*(*ac))-0.5*h;
			gdl_graphics2D_set_paint (graph, (gdl_rgb_color *)gdl_plot_parameter_get (acolors, allele));
			gdl_graphics2D_fill_rect (graph, x1, y1, w, h);
			gdl_graphics2D_set_color (graph, (gdl_rgb_color *)gdl_plot_parameter_get (hcolors, k));
			x1 = p->xlocus[i];
			if (j && y4[k])
			{
				x2 = x1;
				gdl_graphics2D_draw_line (graph, x1, y3[k], x2, y1);
				gdl_graphics2D_draw_line (graph, x1, y4[k], x2, y1+h);
			}
			if (i < p->result->L-1)
			{
				x2 = p->xlocus[i+1];
			}
			else
			{
				x2 = p->xlocus[i]+*lw;
			}
			y3[k] = y2 = y1;
			//y1 = y1*1.1;//y + 2*k*((*lh)*(*ac)) + k*(*sp)/4;
			gdl_graphics2D_draw_line (graph, x1, y1, x2, y2);
			y1 += h;
			y4[k] = y2 = y1; //+ 2*(k+1)*((*lh)*(*ac)) + k*(*sp)/4;
			gdl_graphics2D_draw_line (graph, x1, y1, x2, y2);
		}
	}
	
	GDL_FREE (y3);
	GDL_FREE (y4);
		
	return y+2*((*lh)*(*ac))*p->result->K+(p->result->K-1)*(*sp)/4;
}

static double
gdl_mosaic_plot_draw_accession (gdl_mosaic_plot_t * p, const gdl_plot_parameters * par, gdl_graphics2D * graph, double y)
{
	size_t m, i, j, l, k, ka, np, na, nl, pl, tnl, accession;
	int * al, * wm, * bbp, * af, * sm, pka;
	double x1, x2, y1, y2, w, h, pr, * lh, * lw, * sp, * ac;
	gdl_rgb_color * mutcol, * miscol;
	gdl_vector_uint * path = NULL;
	gdl_string * name;
	const gdl_plot_parameter           * hcolors;
	const gdl_plot_parameter           * alist;
	
	lh      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-height"), 0);
	lw      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "locus-width"), 0);
	sp      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "accession-space"), 0);
	ac      = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "accession-cex"), 0);
	al      = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "accession-label"), 0);
	af      = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "accession-fast"), 0);
	wm      = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "with-mutation"), 0);
	bbp     = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "best-block-path"), 0);
	sm      = (int *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "ignore-missing"), 0);
	mutcol  = (gdl_rgb_color *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "mutation-color"), 0);
	miscol  = (gdl_rgb_color *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "mutation-color"), 0);
	hcolors = gdl_plot_parameters_get (par, "ancestral-color");
	alist   = gdl_plot_parameters_get (par, "accession-list");
	
	gdl_graphics2D_set_color (graph, gdl_rgb_color_get_black ());
	gdl_graphics2D_set_font  (graph, (gdl_font *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "accession-font"), 0));
	gdl_graphics2D_set_stroke (graph, 0.25, 0, NULL);
	
	pl   = p->result->P;
	tnl  = p->result->L;
	
#define HAPLO_Y {  if (*wm) y1 = y + ((j*pl+k)*(2*(*lh)+(*sp)/6) + j*(*sp)/4 + *lh)*(*ac); \
						 else y1 = y + ((j*pl+k)*((*lh)+(*sp)/6) + j*(*sp)/4)*(*ac);}
	
	
	na = gdl_plot_parameter_size (alist);
	
	for (j = 0; j < na; j++)
	{
		name = (gdl_string *) gdl_plot_parameter_get (alist, j);
		accession = gdl_glabels_search_accession (p->result->labels, name);
		if (accession>=0)
		{	
			if (*al)
			{
				if (*wm)
				{
					gdl_graphics2D_draw_text (graph, 0, y + (j*pl*((*lh)*2+(*sp)/6) + j*(*sp)/4 + (*lh))*(*ac), name);
				}
				else
				{
					gdl_graphics2D_draw_text (graph, 0, y + (j*pl*((*lh)+(*sp)/6) + j*(*sp)/4)*(*ac), name);
				}
			}
			for (m = 0; m < p->result->L; m++)
			{
				for (k = 0; k < p->result->P; k++)
				{
					
					HAPLO_Y
					
					w = *lw;
					
					if ((*af && m == 0) || !(*af))
					{
						x1 = p->xlocus[m];
					}
						
					ka = gdl_allele_block_get (p->result->viterbi, gdl_clustering_cluster (p->result->clust, accession), m, k);

					if (*af && (ka != pka && m > 0))
					{
						// draw the previous contiguous color rectangle....
						w  = p->xlocus[m] - x1;
						gdl_graphics2D_set_paint (graph, (gdl_rgb_color *)gdl_plot_parameter_get (hcolors, pka));
						gdl_graphics2D_fill_rect (graph, x1, y1, w, (*lh)*(*ac));
						x1  = p->xlocus[m];
						pka = ka;
					}
					else if (*af && m == 0)
					{
						pka = ka;
					}
					else if (!(*af))
					{
						gdl_graphics2D_set_paint (graph, (gdl_rgb_color *)gdl_plot_parameter_get (hcolors, ka));
						gdl_graphics2D_fill_rect (graph, x1, y1, w, (*lh)*(*ac));
					}
				}
			}
		}
		else
		{
			fprintf (stderr, "WARNING : Unable to find accession [ %s ]\n", (gdl_string *) gdl_plot_parameter_get (alist, j));	
		}
	}
	
	if (*wm)
	{
		return y+(pl*na*(*lh*2)+(pl*na)*(*sp)/6 + na*(*sp)/4)*(*ac);
	}
	else
	{
		return y+(pl*na*(*lh)+(pl*na)*(*sp)/6 + na*(*sp)/4)*(*ac);	
	}

#undef HAPLO_Y	

}

static int
gdl_mosaic_plot_draw (void * plot, const gdl_plot_parameters * par, gdl_graphics2D * graph)
{
	double y = 0, * sp;
	gdl_rgb_color * c0;
	gdl_font      * f0;
	gdl_mosaic_plot_t * p       = (gdl_mosaic_plot_t *) plot;
	
	c0 = gdl_graphics2D_get_color (graph);
	f0 = gdl_graphics2D_get_font (graph);
	
	sp = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "vertical-space"), 0);
	
	y = gdl_mosaic_plot_draw_chromosome (p, par, graph, y);
	y+= *sp;
	y = gdl_mosaic_plot_draw_recomb (p, par, graph, y, 0);
	y+= *sp;
	y = gdl_mosaic_plot_draw_mutation (p, par, graph, y);
	y+= *sp;
	y = gdl_mosaic_plot_draw_ancestral (p, par, graph, y);
	y+= *sp;
	y = gdl_mosaic_plot_draw_accession (p, par, graph, y);
	
	gdl_graphics2D_set_color (graph, c0);
	gdl_graphics2D_set_font (graph, f0);
	
	return GDL_SUCCESS;
}

static const gdl_plot_type _gdl_mosaic_result_plot =
{
	"gdl_mosaic_result",
	sizeof (gdl_mosaic_plot_t),
	&gdl_mosaic_plot_alloc,
	&gdl_mosaic_plot_free,
	&gdl_mosaic_plot_draw,
	NULL
};

const gdl_plot_type * gdl_mosaic_result_plot = &_gdl_mosaic_result_plot;

