/* multifit/fsolver.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <gdl/gdl_common.h>
#include <stdlib.h>
#include <string.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_multifit_nlin.h>

gdl_multifit_fsolver *
gdl_multifit_fsolver_alloc (const gdl_multifit_fsolver_type * T, 
                            size_t n, size_t p) 
{
  int status;

  gdl_multifit_fsolver * s;

  if (n < p)
    {
      GDL_ERROR_VAL ("insufficient data points, n < p", GDL_EINVAL, 0);
    }

  s = (gdl_multifit_fsolver *) malloc (sizeof (gdl_multifit_fsolver));

  if (s == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for multifit solver struct",
                        GDL_ENOMEM, 0);
    }

  s->x = gdl_vector_calloc (p);

  if (s->x == 0) 
    {
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for x", GDL_ENOMEM, 0);
    }

  s->f = gdl_vector_calloc (n);

  if (s->f == 0) 
    {
      gdl_vector_free (s->x);
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for f", GDL_ENOMEM, 0);
    }

  s->dx = gdl_vector_calloc (p);

  if (s->dx == 0) 
    {
      gdl_vector_free (s->x);
      gdl_vector_free (s->f);
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for dx", GDL_ENOMEM, 0);
    }

  s->state = malloc (T->size);

  if (s->state == 0)
    {
      gdl_vector_free (s->dx);
      gdl_vector_free (s->x);
      gdl_vector_free (s->f);
      free (s);         /* exception in constructor, avoid memory leak */
      
      GDL_ERROR_VAL ("failed to allocate space for multifit solver state",
                        GDL_ENOMEM, 0);
    }

  s->type = T ;

  status = (s->type->alloc)(s->state, n, p);

  if (status != GDL_SUCCESS)
    {
      (s->type->free)(s->state);
      free (s->state);
      gdl_vector_free (s->dx);
      gdl_vector_free (s->x);
      gdl_vector_free (s->f);
      free (s);         /* exception in constructor, avoid memory leak */
      
      GDL_ERROR_VAL ("failed to set solver", status, 0);
    }
  
  s->function = NULL;

  return s;
}

int
gdl_multifit_fsolver_set (gdl_multifit_fsolver * s, 
                          gdl_multifit_function * f, 
                          gdl_vector * x)
{
  if (s->f->size != f->n)
    {
      GDL_ERROR ("function size does not match solver", GDL_EBADLEN);
    }

  if (s->x->size != x->size)
    {
      GDL_ERROR ("vector length does not match solver", GDL_EBADLEN);
    }  
  
  s->function = f;
  gdl_vector_memcpy(s->x,x);
  
  return (s->type->set) (s->state, s->function, s->x, s->f, s->dx);
}

int
gdl_multifit_fsolver_iterate (gdl_multifit_fsolver * s)
{
  return (s->type->iterate) (s->state, s->function, s->x, s->f, s->dx);
}

void
gdl_multifit_fsolver_free (gdl_multifit_fsolver * s)
{
  (s->type->free) (s->state);
  free (s->state);
  gdl_vector_free (s->dx);
  gdl_vector_free (s->x);
  gdl_vector_free (s->f);
  free (s);
}

const char *
gdl_multifit_fsolver_name (const gdl_multifit_fsolver * s)
{
  return s->type->name;
}

gdl_vector *
gdl_multifit_fsolver_position (const gdl_multifit_fsolver * s)
{
  return s->x;
}
