/* multifit/work.c
 * 
 * Copyright (C) 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_multifit.h>

gdl_multifit_linear_workspace *
gdl_multifit_linear_alloc (size_t n, size_t p)
{
  gdl_multifit_linear_workspace *w;

  w = (gdl_multifit_linear_workspace *)
    malloc (sizeof (gdl_multifit_linear_workspace));

  if (w == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for multifit_linear struct",
                     GDL_ENOMEM, 0);
    }

  w->n = n;                     /* number of observations */
  w->p = p;                     /* number of parameters */

  w->A = gdl_matrix_alloc (n, p);

  if (w->A == 0)
    {
      free (w);
      GDL_ERROR_VAL ("failed to allocate space for A", GDL_ENOMEM, 0);
    }

  w->Q = gdl_matrix_alloc (p, p);

  if (w->Q == 0)
    {
      gdl_matrix_free (w->A);
      free (w);
      GDL_ERROR_VAL ("failed to allocate space for Q", GDL_ENOMEM, 0);
    }

  w->QSI = gdl_matrix_alloc (p, p);

  if (w->QSI == 0)
    {
      gdl_matrix_free (w->Q);
      gdl_matrix_free (w->A);
      free (w);
      GDL_ERROR_VAL ("failed to allocate space for QSI", GDL_ENOMEM, 0);
    }

  w->S = gdl_vector_alloc (p);

  if (w->S == 0)
    {
      gdl_matrix_free (w->QSI);
      gdl_matrix_free (w->Q);
      gdl_matrix_free (w->A);
      free (w);
      GDL_ERROR_VAL ("failed to allocate space for S", GDL_ENOMEM, 0);
    }

  w->t = gdl_vector_alloc (n);

  if (w->t == 0)
    {
      gdl_vector_free (w->S);
      gdl_matrix_free (w->QSI);
      gdl_matrix_free (w->Q);
      gdl_matrix_free (w->A);
      free (w);
      GDL_ERROR_VAL ("failed to allocate space for t", GDL_ENOMEM, 0);
    }

  w->xt = gdl_vector_calloc (p);

  if (w->xt == 0)
    {
      gdl_vector_free (w->t);
      gdl_vector_free (w->S);
      gdl_matrix_free (w->QSI);
      gdl_matrix_free (w->Q);
      gdl_matrix_free (w->A);
      free (w);
      GDL_ERROR_VAL ("failed to allocate space for xt", GDL_ENOMEM, 0);
    }

  w->D = gdl_vector_calloc (p);

  if (w->D == 0)
    {
      gdl_vector_free (w->D);
      gdl_vector_free (w->t);
      gdl_vector_free (w->S);
      gdl_matrix_free (w->QSI);
      gdl_matrix_free (w->Q);
      gdl_matrix_free (w->A);
      free (w);
      GDL_ERROR_VAL ("failed to allocate space for xt", GDL_ENOMEM, 0);
    }

  return w;
}

void
gdl_multifit_linear_free (gdl_multifit_linear_workspace * work)
{
  gdl_matrix_free (work->A);
  gdl_matrix_free (work->Q);
  gdl_matrix_free (work->QSI);
  gdl_vector_free (work->S);
  gdl_vector_free (work->t);
  gdl_vector_free (work->xt);
  gdl_vector_free (work->D);
  free (work);
}

