/* multimin/fminimizer.c
 * 
 * Copyright (C) 2002 Tuomo Keskitalo, Ivo Alxneit
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <config.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_multimin.h>

gdl_multimin_fminimizer *
gdl_multimin_fminimizer_alloc (const gdl_multimin_fminimizer_type * T,
                               size_t n)
{
  int status;

  gdl_multimin_fminimizer *s =
    (gdl_multimin_fminimizer *) malloc (sizeof (gdl_multimin_fminimizer));

  if (s == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for minimizer struct",
                     GDL_ENOMEM, 0);
    }

  s->type = T;

  s->x = gdl_vector_calloc (n);

  if (s->x == 0) 
    {
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for x", GDL_ENOMEM, 0);
    }

  s->state = malloc (T->size);

  if (s->state == 0)
    {
      gdl_vector_free (s->x);
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for minimizer state",
                     GDL_ENOMEM, 0);
    }

  status = (T->alloc) (s->state, n);

  if (status != GDL_SUCCESS)
    {
      free (s->state);
      gdl_vector_free (s->x);
      free (s);

      GDL_ERROR_VAL ("failed to initialize minimizer state", GDL_ENOMEM, 0);
    }

  return s;
}

int
gdl_multimin_fminimizer_set (gdl_multimin_fminimizer * s,
                             gdl_multimin_function * f,
                             const gdl_vector * x,
                             const gdl_vector * step_size)
{
  if (s->x->size != f->n)
    {
      GDL_ERROR ("function incompatible with solver size", GDL_EBADLEN);
    }
  
  if (x->size != f->n || step_size->size != f->n) 
    {
      GDL_ERROR ("vector length not compatible with function", GDL_EBADLEN);
    }  
    
  s->f = f;

  gdl_vector_memcpy (s->x,x);

  return (s->type->set) (s->state, s->f, s->x, &(s->size), step_size);
}

void
gdl_multimin_fminimizer_free (gdl_multimin_fminimizer * s)
{
  (s->type->free) (s->state);
  free (s->state);
  gdl_vector_free (s->x);
  free (s);
}

int
gdl_multimin_fminimizer_iterate (gdl_multimin_fminimizer * s)
{
  return (s->type->iterate) (s->state, s->f, s->x, &(s->size), &(s->fval));
}

const char * 
gdl_multimin_fminimizer_name (const gdl_multimin_fminimizer * s)
{
  return s->type->name;
}


gdl_vector * 
gdl_multimin_fminimizer_x (const gdl_multimin_fminimizer * s)
{
  return s->x;
}

double 
gdl_multimin_fminimizer_minimum (const gdl_multimin_fminimizer * s)
{
  return s->fval;
}

double
gdl_multimin_fminimizer_size (const gdl_multimin_fminimizer * s)
{
  return s->size;
}
