/* multimin/gdl_multimin.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Fabrice Rossi
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/* Modified by Tuomo Keskitalo to include fminimizer and 
   Nelder Mead related lines */

#ifndef __GDL_MULTIMIN_H__
#define __GDL_MULTIMIN_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_min.h>

__BEGIN_DECLS

/* Definition of an arbitrary real-valued function with gdl_vector input and */
/* parameters */
struct gdl_multimin_function_struct 
{
  double (* f) (const gdl_vector * x, void * params);
  size_t n;
  void * params;
};

typedef struct gdl_multimin_function_struct gdl_multimin_function;

#define GDL_MULTIMIN_FN_EVAL(F,x) (*((F)->f))(x,(F)->params)

/* Definition of an arbitrary differentiable real-valued function */
/* with gdl_vector input and parameters */
struct gdl_multimin_function_fdf_struct 
{
  double (* f) (const gdl_vector  * x, void * params);
  void (* df) (const gdl_vector * x, void * params,gdl_vector * df);
  void (* fdf) (const gdl_vector * x, void * params,double *f,gdl_vector * df);
  size_t n;
  void * params;
};

typedef struct gdl_multimin_function_fdf_struct gdl_multimin_function_fdf;

#define GDL_MULTIMIN_FN_EVAL_F(F,x) (*((F)->f))(x,(F)->params)
#define GDL_MULTIMIN_FN_EVAL_DF(F,x,g) (*((F)->df))(x,(F)->params,(g))
#define GDL_MULTIMIN_FN_EVAL_F_DF(F,x,y,g) (*((F)->fdf))(x,(F)->params,(y),(g))

int gdl_multimin_diff (const gdl_multimin_function * f,
                       const gdl_vector * x, gdl_vector * g);

/* minimization of non-differentiable functions */

typedef struct 
{
  const char *name;
  size_t size;
  int (*alloc) (void *state, size_t n);
  int (*set) (void *state, gdl_multimin_function * f,
              const gdl_vector * x, 
              double * size,
              const gdl_vector * step_size);
  int (*iterate) (void *state, gdl_multimin_function * f, 
                  gdl_vector * x, 
                  double * size,
                  double * fval);
  void (*free) (void *state);
}
gdl_multimin_fminimizer_type;

typedef struct 
{
  /* multi dimensional part */
  const gdl_multimin_fminimizer_type *type;
  gdl_multimin_function *f;

  double fval;
  gdl_vector * x;
  
  double size;

  void *state;
}
gdl_multimin_fminimizer;

gdl_multimin_fminimizer *
gdl_multimin_fminimizer_alloc(const gdl_multimin_fminimizer_type *T,
                              size_t n);

int 
gdl_multimin_fminimizer_set (gdl_multimin_fminimizer * s,
                             gdl_multimin_function * f,
                             const gdl_vector * x,
                             const gdl_vector * step_size);

void
gdl_multimin_fminimizer_free(gdl_multimin_fminimizer *s);

const char * 
gdl_multimin_fminimizer_name (const gdl_multimin_fminimizer * s);

int
gdl_multimin_fminimizer_iterate(gdl_multimin_fminimizer *s);

gdl_vector * 
gdl_multimin_fminimizer_x (const gdl_multimin_fminimizer * s);

double 
gdl_multimin_fminimizer_minimum (const gdl_multimin_fminimizer * s);

double
gdl_multimin_fminimizer_size (const gdl_multimin_fminimizer * s);

/* Convergence test functions */

int
gdl_multimin_test_gradient(const gdl_vector * g,double epsabs);

int
gdl_multimin_test_size(const double size ,double epsabs);

/* minimisation of differentiable functions */

typedef struct 
{
  const char *name;
  size_t size;
  int (*alloc) (void *state, size_t n);
  int (*set) (void *state, gdl_multimin_function_fdf * fdf,
              const gdl_vector * x, double * f, 
              gdl_vector * gradient, double step_size, double tol);
  int (*iterate) (void *state,gdl_multimin_function_fdf * fdf, 
                  gdl_vector * x, double * f, 
                  gdl_vector * gradient, gdl_vector * dx);
  int (*restart) (void *state);
  void (*free) (void *state);
}
gdl_multimin_fdfminimizer_type;

typedef struct 
{
  /* multi dimensional part */
  const gdl_multimin_fdfminimizer_type *type;
  gdl_multimin_function_fdf *fdf;

  double f;
  gdl_vector * x;
  gdl_vector * gradient;
  gdl_vector * dx;

  void *state;
}
gdl_multimin_fdfminimizer;

gdl_multimin_fdfminimizer *
gdl_multimin_fdfminimizer_alloc(const gdl_multimin_fdfminimizer_type *T,
                                size_t n);

int 
gdl_multimin_fdfminimizer_set (gdl_multimin_fdfminimizer * s,
                               gdl_multimin_function_fdf *fdf,
                               const gdl_vector * x,
                               double step_size, double tol);

void
gdl_multimin_fdfminimizer_free(gdl_multimin_fdfminimizer *s);

const char * 
gdl_multimin_fdfminimizer_name (const gdl_multimin_fdfminimizer * s);

int
gdl_multimin_fdfminimizer_iterate(gdl_multimin_fdfminimizer *s);

int
gdl_multimin_fdfminimizer_restart(gdl_multimin_fdfminimizer *s);

gdl_vector * 
gdl_multimin_fdfminimizer_x (gdl_multimin_fdfminimizer * s);

gdl_vector * 
gdl_multimin_fdfminimizer_dx (gdl_multimin_fdfminimizer * s);

gdl_vector * 
gdl_multimin_fdfminimizer_gradient (gdl_multimin_fdfminimizer * s);

double 
gdl_multimin_fdfminimizer_minimum (gdl_multimin_fdfminimizer * s);

GDL_VAR const gdl_multimin_fdfminimizer_type *gdl_multimin_fdfminimizer_steepest_descent;
GDL_VAR const gdl_multimin_fdfminimizer_type *gdl_multimin_fdfminimizer_conjugate_pr;
GDL_VAR const gdl_multimin_fdfminimizer_type *gdl_multimin_fdfminimizer_conjugate_fr;
GDL_VAR const gdl_multimin_fdfminimizer_type *gdl_multimin_fdfminimizer_vector_bfgs;
GDL_VAR const gdl_multimin_fdfminimizer_type *gdl_multimin_fdfminimizer_vector_bfgs2;
GDL_VAR const gdl_multimin_fminimizer_type *gdl_multimin_fminimizer_nmsimplex;

__END_DECLS

#endif /* __GDL_MULTIMIN_H__ */
