/* multimin/vector_bfgs.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Fabrice Rossi
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/* vector_bfgs.c -- Limited memory Broyden-Fletcher-Goldfarb-Shanno method */

/* Modified by Brian Gough to use single iteration structure */

#include <config.h>
#include <gdl/gdl_multimin.h>
#include <gdl/gdl_blas.h>

#include "directional_minimize.c"

typedef struct
{
  int iter;
  double step;
  double max_step;
  double tol;
  gdl_vector *x1;
  gdl_vector *dx1;
  gdl_vector *x2;
  double g0norm;
  double pnorm;
  gdl_vector *p;
  gdl_vector *x0;
  gdl_vector *g0;
  gdl_vector *dx0;
  gdl_vector *dg0;
}
vector_bfgs_state_t;

static int
vector_bfgs_alloc (void *vstate, size_t n)
{
  vector_bfgs_state_t *state = (vector_bfgs_state_t *) vstate;

  state->x1 = gdl_vector_calloc (n);

  if (state->x1 == 0)
    {
      GDL_ERROR ("failed to allocate space for x1", GDL_ENOMEM);
    }

  state->dx1 = gdl_vector_calloc (n);

  if (state->dx1 == 0)
    {
      gdl_vector_free (state->x1);
      GDL_ERROR ("failed to allocate space for dx1", GDL_ENOMEM);
    }

  state->x2 = gdl_vector_calloc (n);

  if (state->x2 == 0)
    {
      gdl_vector_free (state->dx1);
      gdl_vector_free (state->x1);
      GDL_ERROR ("failed to allocate space for x2", GDL_ENOMEM);
    }

  state->p = gdl_vector_calloc (n);

  if (state->p == 0)
    {
      gdl_vector_free (state->x2);
      gdl_vector_free (state->dx1);
      gdl_vector_free (state->x1);
      GDL_ERROR ("failed to allocate space for p", GDL_ENOMEM);
    }

  state->x0 = gdl_vector_calloc (n);

  if (state->x0 == 0)
    {
      gdl_vector_free (state->p);
      gdl_vector_free (state->x2);
      gdl_vector_free (state->dx1);
      gdl_vector_free (state->x1);
      GDL_ERROR ("failed to allocate space for g0", GDL_ENOMEM);
    }

  state->g0 = gdl_vector_calloc (n);

  if (state->g0 == 0)
    {
      gdl_vector_free (state->x0);
      gdl_vector_free (state->p);
      gdl_vector_free (state->x2);
      gdl_vector_free (state->dx1);
      gdl_vector_free (state->x1);
      GDL_ERROR ("failed to allocate space for g0", GDL_ENOMEM);
    }

  state->dx0 = gdl_vector_calloc (n);

  if (state->dx0 == 0)
    {
      gdl_vector_free (state->g0);
      gdl_vector_free (state->x0);
      gdl_vector_free (state->p);
      gdl_vector_free (state->x2);
      gdl_vector_free (state->dx1);
      gdl_vector_free (state->x1);
      GDL_ERROR ("failed to allocate space for g0", GDL_ENOMEM);
    }

  state->dg0 = gdl_vector_calloc (n);

  if (state->dg0 == 0)
    {
      gdl_vector_free (state->dx0);
      gdl_vector_free (state->g0);
      gdl_vector_free (state->x0);
      gdl_vector_free (state->p);
      gdl_vector_free (state->x2);
      gdl_vector_free (state->dx1);
      gdl_vector_free (state->x1);
      GDL_ERROR ("failed to allocate space for g0", GDL_ENOMEM);
    }

  return GDL_SUCCESS;
}

static int
vector_bfgs_set (void *vstate, gdl_multimin_function_fdf * fdf,
                 const gdl_vector * x, double *f, gdl_vector * gradient,
                 double step_size, double tol)
{
  vector_bfgs_state_t *state = (vector_bfgs_state_t *) vstate;

  state->iter = 0;
  state->step = step_size;
  state->max_step = step_size;
  state->tol = tol;

  GDL_MULTIMIN_FN_EVAL_F_DF (fdf, x, f, gradient);

  /* Use the gradient as the initial direction */

  gdl_vector_memcpy (state->x0, x);
  gdl_vector_memcpy (state->p, gradient);
  gdl_vector_memcpy (state->g0, gradient);

  {
    double gnorm = gdl_blas_dnrm2 (gradient);
    state->pnorm = gnorm;
    state->g0norm = gnorm;
  }

  return GDL_SUCCESS;
}

static void
vector_bfgs_free (void *vstate)
{
  vector_bfgs_state_t *state = (vector_bfgs_state_t *) vstate;

  gdl_vector_free (state->dg0);
  gdl_vector_free (state->dx0);
  gdl_vector_free (state->g0);
  gdl_vector_free (state->x0);
  gdl_vector_free (state->p);
  gdl_vector_free (state->x2);
  gdl_vector_free (state->dx1);
  gdl_vector_free (state->x1);
}

static int
vector_bfgs_restart (void *vstate)
{
  vector_bfgs_state_t *state = (vector_bfgs_state_t *) vstate;

  state->iter = 0;
  return GDL_SUCCESS;
}

static int
vector_bfgs_iterate (void *vstate, gdl_multimin_function_fdf * fdf,
                     gdl_vector * x, double *f,
                     gdl_vector * gradient, gdl_vector * dx)
{
  vector_bfgs_state_t *state = (vector_bfgs_state_t *) vstate;

  gdl_vector *x1 = state->x1;
  gdl_vector *dx1 = state->dx1;
  gdl_vector *x2 = state->x2;
  gdl_vector *p = state->p;
  gdl_vector *g0 = state->g0;
  gdl_vector *x0 = state->x0;

  double pnorm = state->pnorm;
  double g0norm = state->g0norm;

  double fa = *f, fb, fc;
  double dir;
  double stepa = 0.0, stepb, stepc = state->step, tol = state->tol;

  double g1norm;
  double pg;

  if (pnorm == 0.0 || g0norm == 0.0)
    {
      gdl_vector_set_zero (dx);
      return GDL_ENOPROG;
    }

  /* Determine which direction is downhill, +p or -p */

  gdl_blas_ddot (p, gradient, &pg);

  dir = (pg >= 0.0) ? +1.0 : -1.0;

  /* Compute new trial point at x_c= x - step * p, where p is the
     current direction */

  take_step (x, p, stepc, dir / pnorm, x1, dx);

  /* Evaluate function and gradient at new point xc */

  fc = GDL_MULTIMIN_FN_EVAL_F (fdf, x1);

  if (fc < fa)
    {
      /* Success, reduced the function value */
      state->step = stepc * 2.0;
      *f = fc;
      gdl_vector_memcpy (x, x1);
      GDL_MULTIMIN_FN_EVAL_DF (fdf, x1, gradient);
      return GDL_SUCCESS;
    }

#ifdef DEBUG
  printf ("got stepc = %g fc = %g\n", stepc, fc);
#endif

  /* Do a line minimisation in the region (xa,fa) (xc,fc) to find an
     intermediate (xb,fb) satisifying fa > fb < fc.  Choose an initial
     xb based on parabolic interpolation */

  intermediate_point (fdf, x, p, dir / pnorm, pg,
                      stepa, stepc, fa, fc, x1, dx1, gradient, &stepb, &fb);

  if (stepb == 0.0)
    {
      return GDL_ENOPROG;
    }

  minimize (fdf, x, p, dir / pnorm,
            stepa, stepb, stepc, fa, fb, fc, tol,
            x1, dx1, x2, dx, gradient, &(state->step), f, &g1norm);

  gdl_vector_memcpy (x, x2);

  /* Choose a new direction for the next step */

  state->iter = (state->iter + 1) % x->size;

  if (state->iter == 0)
    {
      gdl_vector_memcpy (p, gradient);
      state->pnorm = g1norm;
    }
  else
    {
      /* This is the BFGS update: */
      /* p' = g1 - A dx - B dg */
      /* A = - (1+ dg.dg/dx.dg) B + dg.g/dx.dg */
      /* B = dx.g/dx.dg */

      gdl_vector *dx0 = state->dx0;
      gdl_vector *dg0 = state->dg0;

      double dxg, dgg, dxdg, dgnorm, A, B;

      /* dx0 = x - x0 */
      gdl_vector_memcpy (dx0, x);
      gdl_blas_daxpy (-1.0, x0, dx0);

      /* dg0 = g - g0 */
      gdl_vector_memcpy (dg0, gradient);
      gdl_blas_daxpy (-1.0, g0, dg0);

      gdl_blas_ddot (dx0, gradient, &dxg);
      gdl_blas_ddot (dg0, gradient, &dgg);
      gdl_blas_ddot (dx0, dg0, &dxdg);

      dgnorm = gdl_blas_dnrm2 (dg0);

      if (dxdg != 0) 
        {
          B = dxg / dxdg;
          A = -(1.0 + dgnorm * dgnorm / dxdg) * B + dgg / dxdg;
        }
      else
        {
          B = 0;
          A = 0; 
        }

      gdl_vector_memcpy (p, gradient);
      gdl_blas_daxpy (-A, dx0, p);
      gdl_blas_daxpy (-B, dg0, p);

      state->pnorm = gdl_blas_dnrm2 (p);
    }

  gdl_vector_memcpy (g0, gradient);
  gdl_vector_memcpy (x0, x);
  state->g0norm = gdl_blas_dnrm2 (g0);

#ifdef DEBUG
  printf ("updated directions\n");
  printf ("p: ");
  gdl_vector_fprintf (stdout, p, "%g");
  printf ("g: ");
  gdl_vector_fprintf (stdout, gradient, "%g");
#endif

  return GDL_SUCCESS;
}

static const gdl_multimin_fdfminimizer_type vector_bfgs_type = {
  "vector_bfgs",                /* name */
  sizeof (vector_bfgs_state_t),
  &vector_bfgs_alloc,
  &vector_bfgs_set,
  &vector_bfgs_iterate,
  &vector_bfgs_restart,
  &vector_bfgs_free
};

const gdl_multimin_fdfminimizer_type
  * gdl_multimin_fdfminimizer_vector_bfgs = &vector_bfgs_type;
