/* All the subroutines in this file come from

  Numerical Recipes in C, The Art of Scientific Computing
  by W. H. Press, B. P. Flannery, S. A. Teukolsky and W. T. Vetterling.
  1990, Cambridge University Press, Cambridge, England
  ISBN 0-521-35465-X  (the book)
  
  They have been modified ony slightly by Chris J. Basten, on
  19 January 1994.  The modifications should not effect the behavior
  of the programs in any way.  The source has been reformatted to
  the liking of C.J. Basten, and the function definitions have been
  moved to header file. All 'float' declarations were changed to 'double'.
  Finally, the uniform random number generator
  ranf() was substituted for ran1().  ranf() is an implementation of
  of a FORTRAN subroutine written by John Monohan at North Carolina
  State University.
  
  New functions and slight modification were added by J.B. Veyrieras
  on June 2004.
*/
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_nrec.h>

double gammln(double xx)
{
  double x, tmp, ser;
  static double cof[6] = {76.18009173, -86.50532033, 24.01409822,
  -1.231739516, 0.120858003e-2, -0.536382e-5};
  int j;
  x = xx - 1.0;
  tmp = x + 5.5;
  tmp -= (x + 0.5) * log(tmp);
  ser = 1.0;
  for (j = 0; j <= 5; j++) {
    x += 1.0;
    ser += cof[j] / x;
  }
  return -tmp + log(2.50662827465 * ser);
}

double gammp(double a,double  x)
{
  double gamser, gammcf, gln;

  if (x < 0.0 || a <= 0.0)
    GDL_ERROR_VAL ("Invalid arguments in routine GAMMP", GDL_EINVAL, 0);
  if (x < (a + 1.0)) {
    gser(&gamser, a, x, &gln);
    return gamser;
  }
  else {
    gcf(&gammcf, a, x, &gln);
    return 1.0 - gammcf;
  }
}

void gcf(double *gammcf,double  a,double  x,double  *gln)
{
  int n;
  double gold = 0.0, g, fac = 1.0, b1 = 1.0;
  double b0 = 0.0, anf, ana, an, a1, a0 = 1.0;

  *gln = gammln(a);
  a1 = x;
  for (n = 1; n <= GDL_NREC_ITMAX; n++) {
    an = (double) n;
    ana = an - a;
    a0 = (a1 + a0 * ana) * fac;
    b0 = (b1 + b0 * ana) * fac;
    anf = an * fac;
    a1 = x * a0 + anf * a1;
    b1 = x * b0 + anf * b1;
    if (a1) {
      fac = 1.0 / a1;
      g = b1 * fac;
      if (fabs((g - gold) / g) < GDL_NREC_EPS) {
	*gammcf = exp(-x + a * log(x) - (*gln)) * g;
	return;
      }
      gold = g;
    }
  }
  GDL_ERROR_VOID ("a too large, ITMAX too small in routine GCF", GDL_FAILURE);
}



void gser(double *gamser,double a,double x,double *gln)
{
  int n;
  double sum, del, ap;

  *gln = gammln(a);
  if (x <= 0.0) {
    if (x < 0.0)
      GDL_ERROR_VOID ("x less than 0 in routine GSER", GDL_EINVAL);
    *gamser = 0.0;
    return;
  }
  else {
    ap = a;
    del = sum = 1.0 / a;
    for (n = 1; n <= GDL_NREC_ITMAX; n++) {
      ap += 1.0;
      del *= x / ap;
      sum += del;
      if (fabs(del) < fabs(sum) * GDL_NREC_EPS) {
	*gamser = sum * exp(-x + a * log(x) - (*gln));
	return;
      }
    }
    GDL_ERROR_VOID ("a too large, ITMAX too small in routine GSER", GDL_EINVAL);
    return;
  }
}

void iindexx(int n, int arrin[],int indx[])
{
  int l, j, ir, indxt, i;
  int q;

  for (j = 1; j <= n; j++)
    indx[j] = j;
  l = (n >> 1) + 1;
  ir = n;
  for (;;) {
    if (l > 1)
      q = arrin[(indxt = indx[--l])];
    else {
      q = arrin[(indxt = indx[ir])];
      indx[ir] = indx[1];
      if (--ir == 1) {
	indx[1] = indxt;
	return;
      }
    }
    i = l;
    j = l << 1;
    while (j <= ir) {
      if (j < ir && arrin[indx[j]] < arrin[indx[j + 1]])
	j++;
      if (q < arrin[indx[j]]) {
	indx[i] = indx[j];
	j += (i = j);
      }
      else
	j = ir + 1;
    }
    indx[i] = indxt;
  }
}

void indexx(int n,double arrin[],int indx[])
{
  int l, j, ir, indxt, i;
  double q;

  for (j = 1; j <= n; j++)
    indx[j] = j;
  l = (n >> 1) + 1;
  ir = n;
  for (;;) {
    if (l > 1)
      q = arrin[(indxt = indx[--l])];
    else {
      q = arrin[(indxt = indx[ir])];
      indx[ir] = indx[1];
      if (--ir == 1) {
	indx[1] = indxt;
	return;
      }
    }
    i = l;
    j = l << 1;
    while (j <= ir) {
      if (j < ir && arrin[indx[j]] < arrin[indx[j + 1]])
	j++;
      if (q < arrin[indx[j]]) {
	indx[i] = indx[j];
	j += (i = j);
      }
      else
	j = ir + 1;
    }
    indx[i] = indxt;
  }
}

void moment(double data[],int n,double *ave,double *adev,double *sdev,double *svar,double *skew,double *curt)
{
  int j;
  double s, p;


  if (n <= 1)
    GDL_ERROR_VOID ("n must be at least 2 in MOMENT", GDL_EINVAL);
  s = 0.0;
  for (j = 1; j <= n; j++)
    s += data[j];
  *ave = s / n;
  *adev = (*svar) = (*skew) = (*curt) = 0.0;
  for (j = 1; j <= n; j++) {
    *adev += fabs(s = data[j] - (*ave));
    *svar += (p = s * s);
    *skew += (p *= s);
    *curt += (p *= s);
  }
  *adev /= n;
  *svar /= (n - 1);
  *sdev = sqrt(*svar);
  if (*svar) {
    *skew /= (n * (*svar) * (*sdev));
    *curt = (*curt) / (n * (*svar) * (*svar)) - 3.0;
  }
  else
    GDL_ERROR_VOID ("No skew/kurtosis when variance = 0 (in MOMENT)", GDL_FAILURE);
}


double beta(double z,double  w)
{
  return  exp(gammln(z) + gammln(w) - gammln(z + w)) ;
}


double betacf(double a,double  b,double  x)
{
  double qap, qam, qab, em, tem, d;
  double bz, bm = 1.0, bp, bpp;
  double az = 1.0, am = 1.0, ap, app, aold;
  int m;

  qab = a + b;
  qap = a + 1.0;
  qam = a - 1.0;
  bz = 1.0 - qab * x / qap;
  for (m = 1; m <= GDL_NREC_ITMAX; m++) {
    em = (float) m;
    tem = em + em;
    d = em * (b - em) * x / ((qam + tem) * (a + tem));
    ap = az + d * am;
    bp = bz + d * bm;
    d = -(a + em) * (qab + em) * x / ((qap + tem) * (a + tem));
    app = ap + d * az;
    bpp = bp + d * bz;
    aold = az;
    am = ap / bpp;
    bm = bp / bpp;
    az = app / bpp;
    bz = 1.0;
    if (fabs(az - aold) < (GDL_NREC_EPS * fabs(az)))
      return(az);
  }
  GDL_ERROR_VAL ("a or b too big, or ITMAX too small in BETACF", GDL_EINVAL, 0);
  return(0.0);
}

double betai(double a,double  b,double  x)
{
  double bt;

  if (x < 0.0 || x > 1.0)
    GDL_ERROR_VAL ("Bad x in routine BETAI", GDL_EINVAL, 0); 
  if (x == 0.0 || x == 1.0)
    bt = 0.0;
  else
    bt = exp(gammln(a + b) - gammln(a) - gammln(b) + a * log(x) + b * log(1.0 - x));
  if (x < (a + 1.0) / (a + b + 2.0))
    return bt * betacf(a, b, x) / a;
  else
    return 1.0 - bt * betacf(b, a, 1.0 - x) / b;
}
			
double dpythag(double a, double b) {
   double absa, absb;
   absa = fabs(a);
   absb = fabs(b);
   if (absa > absb) return absa*sqrt(1.0+GDL_SQR_DBL(absb/absa));
   else return (absb == 0.0 ? 0.0 : absb*sqrt(1.0+GDL_SQR_DBL(absa/absb)));
}

float pythag(float a, float b) {
   float absa, absb;
   absa = fabs(a);
   absb = fabs(b);
   if (absa > absb) return absa*sqrt(1.0+GDL_SQR(absb/absa));
   else return (absb == 0.0 ? 0.0 : absb*sqrt(1.0+GDL_SQR(absa/absb)));
}

/**
* Computes the covariance matrix for adjusted data
* set which is represented by the double matrix
* data[1..nn][1..pp]. It assumes that cov[1..p][1..p]
* has been previously initialized.
*/
void cov_matrix(double **cov, double **data, int pp, int nn) {
	int ii, jj;
	for(ii=1; ii<=pp; ii++) {
		for(jj=ii; jj<=pp; jj++)
		   cov[jj][ii] = cov[ii][jj] = stats_cov_mcol(data, ii, jj, nn);
	}
}
/**
* Computes the correlation matrix for adjusted data
* set which is represented by the double matrix
* data[1..nn][1..pp]. It assumes that cov[1..p][1..p]
* has been previously initialized.
* Missing data in data must be coded using MISS_VAL.
*/
void cor_matrix(double **cor, double **data, int pp, int nn) {
	int ii, jj;
	cov_matrix(cor, data, pp, nn);
	for(ii=1; ii<=pp; ii++)
		for(jj=ii+1; jj<=pp; jj++) {
			if (cor[ii][jj] != 0.)
				cor[ii][jj] = cor[jj][ii] =
					cor[ii][jj]/sqrt(cor[ii][ii]*cor[jj][jj]);
		}
	for(ii=1; ii<=pp; ii++)
		cor[ii][ii] = 1.0;					
}
/**
 * Householder reduction of a real, symmetric matrix a[1..n][1..n]. 
 * On output, a is replaced by the orthogonal matrix Q effecting the
 * transformation. d[1..n] returns the diagonal elements of the tridiagonal
 * matrix, and e[1..n] the off-diagonal elements, with e[1]=0. 
 */
void tred2(double **a, int n, double *d, double *e) {
	int l, k, j, i;
	double scale, hh, h, g, f;
	
	for(i=n; i>=2; i--) {
		l=i-1;
		h=scale=0.0;
		if (l > 1) {
			for(k=1; k<=l; k++)
				scale += fabs(a[i][k]);
			if(scale == 0.0)
				e[i]=a[i][l];
			else {
				for(k=1; k<=l; k++) {
					a[i][k] /= scale;
					h += a[i][k]*a[i][k];
				}
				f=a[i][l];
				g=(f >= 0.0) ? -sqrt(h) : sqrt(h);
				e[i] = scale*g;
				h -= f*g;
				a[i][l]=f-g;
				f=0.0;
				for(j=1;j<=l;j++) {
					a[j][i]=a[i][j]/h;
					g=0.0;
					for(k=1;k<=j;k++)
						g += a[j][k]*a[i][k];
					for(k=j+1;k<=l;k++)
						g += a[k][j]*a[i][k];
					e[j] = g/h;
					f += e[j]*a[i][j];
				}
				hh=f/(h+h);
				for(j=1;j<=l;j++) {
				   f = a[i][j];
				   e[j] = e[j] - hh*f;
				   g = e[j];
				   for(k=1; k<=j; k++)
				   		a[j][k] -= (f*e[k]+g*a[i][k]);
				}
			}			
		} else
			e[i] = a[i][l];
		d[i]=h;
	}
	d[1]=0.0;
	e[1]=0.0;
	for(i=1; i<=n; i++) {
		l = i-1;
		if (d[i]) {
			for(j=1; j<=l; j++) {
				g=0.0;
				for(k=1;k<=l;k++)
					g += a[i][k]*a[k][j];
				for(k=1;k<=l;k++)
					a[k][j] -= g*a[k][i];
			}
		}
		d[i]=a[i][i];
		a[i][i]=1.0;
		for(j=1; j<=l; j++) a[j][i]=a[i][j]=0.0;
	}
}
/**
 * QL algorithm with implicit shifts, to determine the eigenvalues and
 * eigenvectors of a real, symetric, tridiagonal matrix, or of a real,
 * symetric matrix previously reduced by tred2().
 * On input, d[1..n] contains the diagonal elements of the tridiagonal matrix.
 * On output, it returns the eigenvalues. The vector e[1..n] inputs the 
 * subdiagonals elements of the tridiagonal matrix, with e[1] arbitrary.
 * On output e is destroyed. if the eigenvectors of a tridiagonal matrix
 * are desired, the matrix z[1..n][1..n] is input as the identity matrix. If the
 * eigenvectors of a matrix that has been reduced by tred2() are required, the z
 * is input as the matrix output by tred2. In either case, the kth column of z
 * returns the normalized eigenvector corresponding to d[k].
 */
void tlqi(double *d, double *e, int n, double **z) {
	int m, l, iter, i, k;
	double s, r, p, g, f, dd, c, b;	
	
	for(i=2; i<=n; i++) e[i-1] = e[i];
	e[n] = 0.0;
	for(l=1; l<=n; l++) {
		iter=0;
		do {
			for(m=l; m<=n-1; m++) {
				dd=fabs(d[m])+fabs(d[m+1]);
				if ((double)(fabs(e[m])+dd) == dd) break;
			}
			if (m != l) {
				if (iter++ == 30)
					GDL_ERROR_VOID ("Too many iterations in tlqi()", GDL_FAILURE);
				g=(d[l+1]-d[l])/(2.0*e[l]);
				r=dpythag(g, 1.0);
				g=d[m]-d[l]+e[l]/(g+GDL_NREC_SIGN(r,g));
				s=c=1.0;
				p=0.0;
				for(i=m-1;i>=l;i--) {
					f=s*e[i];
					b=c*e[i];
					e[i+1]=(r=dpythag(f,g));
					if(r == 0.0) {
						d[i+1] -= p;
						e[m] = 0.0;
						break;
					}
					s=f/r;
					c=g/r;
					g=d[i+1]-p;
					r=(d[i]-g)*s+2.0*c*b;
					d[i+1]=g+(p=s*r);
					g=c*r-b;
					for(k=1;k<=n;k++) {
						f=z[k][i+1];
						z[k][i+1]=s*z[k][i]+c*f;												
						z[k][i]=c*z[k][i]-s*f;
					}
				}
				if (r == 0.0 && i >= l) continue;
				d[l] -= p;
				e[l] = g;
				e[m] = 0.0;
			}
		} while(m != l);
	}	
}
/**
 * Given an array of data x[1..n] returns its mean
 * Missing value code is MISS_VAL
 */
double stats_mean(double *x, int n) {
	int i, tot;
	double mean;
	tot = 0;
	mean = 0.0;
	for(i=1; i<=n; i++) {
		if (!gdl_isnan(x[i])) {
			mean += x[i];
			tot++;
		}
	}
	if (tot != 0)
		mean /= (double)tot;
	else
		printf("Warning mean() : no data to compute the mean\n");
	return mean;
}
/**
 * Given an array of data x[1..n] returns its variance
 * Missing value code is DMISS
 */
double stats_variance(double *x, int n) {
	int i;
	int tot = 0;
	double m = stats_mean(x, n);
	double var = 0.0;
	for(i=1; i<=n; i++) {
		if (!gdl_isnan(x[i])) {
			var += (x[i]-m)*(x[i]-m);
			tot++;
		}
	}
	if (tot != 0)
		var /= (double)(tot-1);
	else
		printf("Warning var() : no data to compute the variance\n");
	return var;
}
/**
 * Given an array of data x[1..n] returns its variance
 * computed using the given mean.
 * Missing value code is DMISS
 */
double stats_variance_m(double *x, int n, double mean) {
	int i;
	int tot = 0;
	double var = 0.0;
	for(i=1; i<=n; i++) {
		if (!gdl_isnan(x[i])) {
			var += (x[i]-mean)*(x[i]-mean);
			tot++;
		}
	}
	if (tot != 0)
		var /= (double)(tot-1);
	else
		printf("Warning var_m() : no data to compute the variance\n");
	return var;	
}
/**
 * Given an array of data x[1..n] returns its mean
 * Missing value code is DMISS
 */
float stats_fmean(float *x, int n) {
	int i, tot;
	float mean;
	tot = 0;
	mean = 0.0;
	for(i=1; i<=n; i++) {
		if (gdl_isnan(x[i])) {
			mean += x[i];
			tot++;
		}
	}
	if (tot != 0)
		mean /= (float)tot;
	else
		printf("Warning mean() : no data to compute the mean\n");
	return mean;
}
/**
 * Given an array of data x[1..n] returns its variance
 * Missing value code is DMISS
 */
float stats_fvariance(float *x, int n) {
	int i;
	int tot = 0;
	float m = stats_fmean(x, n);
	float var = 0.0;
	for(i=1; i<=n; i++) {
		if (gdl_isnan(x[i])) {
			var += (x[i]-m)*(x[i]-m);
			tot++;
		}
	}
	if (tot != 0)
		var /= (float)(tot-1);
	else
		printf("Warning var() : no data to compute the variance\n");
	return var;
}
/**
 * Given an array of data x[1..n] returns its variance
 * computed using the given mean.
 * Missing value code is DMISS
 */
float stats_fvariance_m(float *x, int n, float mean) {
	int i;
	int tot = 0;
	float var = 0.0;
	for(i=1; i<=n; i++) {
		if (gdl_isnan(x[i])) {
			var += (x[i]-mean)*(x[i]-mean);
			tot++;
		}
	}
	if (tot != 0)
		var /= (float)(tot-1);
	else
		printf("Warning var_m() : no data to compute the variance\n");
	return var;	
}
/**
 * Computes the covariance between the iith and the jth column of the
 * data[1..n][..ii..jj..] matrix.
 * cov(x,y) = E(xy)-E(x)E(y)
 */
double stats_cov_mcol(double **data, int ii, int jj, int nn) {
	int kk, tot;
	double e_x, e_y, e_xy;
	e_x = e_y = e_xy = 0.0;	
	for(tot=0,kk=1; kk<=nn; kk++,tot++) {
		if (!gdl_isnan(data[kk][ii]) && !gdl_isnan(data[kk][jj])) {
				e_xy += data[kk][ii]*data[kk][jj];
				e_x += data[kk][ii];
				e_y += data[kk][jj];
			}
		else
			tot--;
	}
	if (tot != 0) {
		e_xy /= ((double)tot);
		e_x /= ((double)tot);
		e_y /= ((double)tot);
	}
	else
		printf("Warning stats_cov_mcol() : no data to compute covariance !\n");
	return e_xy - e_x*e_y;
}

/**
 * Given a n-dimensional point xold[1..n], the value of the function
 * and gradient there, fold and g[1..n], and a direction p[1..n], finds
 * a new point x[1..n] along the direction p from xold where the function
 * func has decreased "sufficiently". The new function value is returned
 * in f. stpmax is an input quantity that limits the length of the steps
 * so that you do not try to evaluate the function in regions where it is
 * undefined or subject to overflow. p is usually the Newton direction. The
 * output quantity check is false on a normal exit. It is true when x is too
 * close to xold. In a minimization algorithm, this usually signals convergence
 * and can be ignored. However, in a zero-finding algorithm the calling program
 * should check whether the convergence is spurious.
 */
void dlnsrch1(
			int n,
			double xold[],
			double fold,
			double g[],
			double p[],
			double x[],
			double *f,
			double stpmax,
			int *check,
			double (*func)(double [], void *param),
			void *param
			)
{
	int i;
	double a,alam,alam2=0.0,alamin,b,disc,f2=0.0,rhs1,rhs2,
	       slope,sum,temp,test,tmplam;
	
	*check=0;
	for(sum=0.0,i=1;i<=n;i++) sum += p[i]*p[i];
	sum=sqrt(sum);
	if(sum > stpmax)
		for(i=1;i<=n;i++) p[i] *= stpmax/sum;
	for(slope=0.0,i=1;i<=n;i++)
		slope+=g[i]*p[i];
	if(slope >= 0.0) {
		printf("Roundoff problem in lnsrch()\n");
	} else {
		test=0.0;
		for(i=1;i<=n;i++) {
			temp=fabs(p[i])/GDL_MAX_DBL(fabs(xold[i]), 1.0);
			if(temp > test) test=temp;
		}
		alamin=GDL_NREC_TOLX/test;
		alam=1.0;
		for(;;) {
			for(i=1;i<=n;i++)
				x[i]=xold[i]+alam*p[i];
			*f=(*func)(x, param);
			if (alam < alamin) {
				for(i=1;i<=n;i++) x[i]=xold[i];
				*check=1;
				return;
			} else if (*f <= fold+GDL_NREC_ALF*alam*slope) return;
			else {
				if (alam == 1.0)
					tmplam = -slope/(2.0*(*f-fold-slope));
				else {
					rhs1= *f-fold-alam*slope;
					rhs2=f2-fold-alam2*slope;
					a=(rhs1/(alam*alam)-rhs2/(alam2*alam2))/(alam-alam2);
					b=(-alam2*rhs1/(alam*alam)+alam*rhs2/(alam2*alam2))/(alam-alam2);
					if (a == 0.0) tmplam = -slope/(2.0*b);
					else {
						disc=b*b-3.0*a*slope;
						if (disc < 0.0) tmplam=0.5*alam;
						else if (b <= 0.0) tmplam=(-b+sqrt(disc))/(3.0*a);
						else tmplam=-slope/(b+sqrt(disc));
					}
					if (tmplam > 0.5*alam)
						tmplam=0.5*alam;
				}	
			}
			alam2=alam;
			f2 = *f;
			alam=GDL_MAX_DBL(tmplam,0.1*alam);
		}
	}
}
/**
 * Given a function f, and given a bracketing triplet of abscissas ax ,bx, cx
 * (such that bx is between ax and cx, and f(bx) is less than both f(ax) and
 * f(bx)), this routine isolates the minimum to a fractional precision of about
 * tol using Brent's method. The abscissa of the minimum is returned as xmi, and
 * the minimum function value is returned as brent, the returned function value.
 * 
 */
double brent(double ax, double bx, double cx, double (*f)(double), double tol, double *xmin)
{
	int iter;
	double a,b,d=0.0,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	double e=0.0;
	
	a=(ax < cx ? ax : cx);
	b=(ax > cx ? ax : cx);
	x=w=v=bx;
	fw=fv=fx=(*f)(x);
	for(iter=1;iter<=GDL_NREC_ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*fabs(x)+(double)GDL_NREC_ZEPS);
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (fabs(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0) p = -p;
			q=fabs(q);
			etemp=e;
			e=d;
			if(fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=(double)GDL_NREC_CGOLD*(e=(x >= xm ? a -x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=GDL_NREC_SIGN(tol1, xm-x);
			}
		} else {
			d=GDL_NREC_CGOLD*(e=(x >= xm ? a-x : b-x));	
		}
		u=(fabs(d) >= tol1 ? x+d : x+GDL_NREC_SIGN(tol1,d));
		fu=(*f)(u);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			GDL_NREC_SHFT(v,w,x,u);
			GDL_NREC_SHFT(fv,fw,fx,fu);	
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;	
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;	
			}
		}
	}
	printf("Too many iterations in brent\n");
	*xmin=x;
	return fx;
}
/**
 * Given a function f and its derivate function df, and given a bracketing
 * triplet of abscissas ax,bx,cx [such that bx is between ax and cx, and f(bx)
 * is less than both f(ax) and f(cx)], this routine isolates the minimum to a 
 * fractional percision of about tol using a modification of Brent's method
 * that uses derivates. The abscissa of the minimum is returned as xmin, and 
 * the minimum function value is returned as dbrent, the returned function
 * value.
 * 
 * It returns 0 if succeed, 1 otherwise.
 * 
 */
int dbrent(double ax, double bx, double cx, double (*f)(double), double (*df)(double), double tol, double *xmin, double *fret)
{
	int iter,ok1,ok2;
	double a,b,d=0.0,d1,d2,du,dv,dw,dx,e=0.0;
	double fu,fv,fw,fx,olde,tol1,tol2,u,u1,u2,v,w,x,xm;
	
	a=(ax < cx ? ax : cx);
	b=(ax > cx ? ax : cx);
	x=w=v=bx;
	
	fw=fv=fx=(*f)(x);
	dw=dv=dx=(*df)(x);
	
	for(iter=1;iter<=GDL_NREC_ITMAX;iter++) {
		xm=0.5*(a+b);
		tol1=tol*fabs(x)+GDL_NREC_ZEPS;
		tol2=2.0*tol1;
		if (fabs(x-xm)<=(tol2-0.5*(b-a))) {
			*xmin=x;
			*fret=fx;
			return 0;	
		}
		if (fabs(e) > tol1) {
			d1=2.0*(b-a);
			d2=d1;
			if (dw != dx) d1=(w-x)*dx/(dx-dw);
			if (dv != dx) d2=(v-x)*dx/(dx-dv);
			u1=x+d1;
			u2=x+d2;
			ok1 = (a-u1)*(u1-b) > 0.0 && dx*d1 <= 0.0;
			ok2 = (a-u2)*(u2-b) > 0.0 && dx*d2 <= 0.0;
			olde=e;
			e=d;
			if (ok1 || ok2) {
				if (ok1 && ok2)
					d=(fabs(d1) < fabs(d2) ? d1 : d2);
				else if (ok1)
					d=d1;
				else
					d=d2;
				if (fabs(d) <= fabs(0.5*olde)) {
					u=x+d;
					if (u-a < tol2 || b-u < tol2)
						d=GDL_NREC_SIGN(tol1, xm-x);
				} else {
					d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
				}
			} else {
				d=0.5*(e=(dx >= 0.0 ? a-x : b-x));	
			}
		} else {
			d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
		}
		if (fabs(d) >= tol1) {
			u=x+d;
			fu=(*f)(u);	
		} else {
			u=x+GDL_NREC_SIGN(tol1,d);
			fu=(*f)(u);
			if (fu > fx) {
				*xmin=x;
				*fret=fx;
				return 0;	
			}
		}
		du=(*df)(u);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			GDL_NREC_MOV3(v, fv, dv, w, fw, dw)
			GDL_NREC_MOV3(w, fw, dw, x, fx, dx)
			GDL_NREC_MOV3(x, fx, dx, u, fu, du)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				GDL_NREC_MOV3(v,fv,dv,w,fw,dw)
				GDL_NREC_MOV3(w,fw,dw,u,fu,du)	
			} else if (fu < fv || v ==x || v == w) {
				GDL_NREC_MOV3(v, fv,dv,u,fu,du)
			}
		}
	}
	fprintf(stderr, "Too many iterations in routine dbrent\n");
	return 1;
}
/**
 * Given a function func, and given distinct initial points ax and bx, this routine searches
 * in the downhill direction (defined by the function as evaluated at the initial points) and
 * returns new points ax, bx, cx that bracket a minimum of the function. Also returned are
 * the function values at the three points, fa, fb, anf fc.
 *  
 */
void mnbrak(double *ax, double *bx, double *cx, double *fa, double *fb, double *fc, double (*func)(double))
{
	double ulim,u,r,q,fu,dum;
	
	*fa=(*func)(*ax);
	*fb=(*func)(*bx);
	
	if (*fb > *fa) {
		GDL_NREC_SHFT(dum, *ax, *bx, dum);
		GDL_NREC_SHFT(dum, *fb, *fa, dum);	
	}
	
	*cx=(*bx)+GDL_NREC_GOLD*(*bx-*ax);
	*fc=(*func)(*cx);
	
	size_t iter = 0;
	
	while(*fb > *fc)
	{
		iter++;
		r=(*bx-*ax)*(*fb-*fc);
		q=(*bx-*cx)*(*fb-*fa);
		u=(*bx)-((*bx-*cx)*q-(*bx-*ax)*r)/(2.0*GDL_NREC_SIGN(GDL_MAX_DBL(fabs(q-r), GDL_NREC_TINY), q-r));
		ulim=(*bx)+GDL_NREC_GLIMIT*(*cx-*bx);
		
		if ((*bx-u)*(u-*cx) > 0.0) {
			fu=(*func)(u);
			if (fu < *fc) {
				*ax=(*bx);
				*bx=u;
				*fa=(*fb);
				*fb=fu;
				return;
			} else if (fu > *fb) {
				*cx=u;
				*fc=fu;
				return;
			}
			u=(*cx)+GDL_NREC_GOLD*(*cx-*bx);
			fu=(*func)(u);
		} else if ((*cx-u)*(u-ulim) > 0.0) {
			fu=(*func)(u);
			if (fu < *fc) {
				GDL_NREC_SHFT(*bx,*cx,u,*cx+GDL_NREC_GOLD*(*cx-*bx));
				GDL_NREC_SHFT(*fb,*fc,fu,(*func)(u));	
			}
		} else if ((u-ulim)*(ulim-*cx) >= 0.0) {
			u=ulim;
			fu=(*func)(u);
		} else {
			u=(*cx)+GDL_NREC_GOLD*(*cx-*bx);
			fu=(*func)(u);	
		}
		
		GDL_NREC_SHFT(*ax,*bx,*cx,u);
		GDL_NREC_SHFT(*fa,*fb,*fc,fu);
		
		if (iter > 25)
			break;
	}
}

/**
 * Given a n-dimensional point xold[1..n], the value of the function
 * and gradient there, fold and g[1..n], and a direction p[1..n], finds
 * a new point x[1..n] along the direction p from xold where the function
 * func has decreased "sufficiently". The new function value is returned
 * in f. stpmax is an input quantity that limits the length of the steps
 * so that you do not try to evaluate the function in regions where it is
 * undefined or subject to overflow. p is usually the Newton direction. The
 * output quantity check is false on a normal exit. It is true when x is too
 * close to xold. In a minimization algorithm, this usually signals convergence
 * and can be ignored. However, in a zero-finding algorithm the calling program
 * should check whether the convergence is spurious.
 */
void dlnsrch(
			int n,
			double xold[],
			double fold,
			double g[],
			double p[],
			double x[],
			double *f,
			double stpmax,
			int *check,
			double (*func)(double *x, void *param, int *check),
			void *param
			)
{
	int i, status;
	double a,alam,alam2=0.0,alamin,b,disc,f2=0.0,rhs1,rhs2,
	       slope,sum,temp,test,tmplam;
	
	*check=0;
	for(sum=0.0,i=1;i<=n;i++) sum += p[i]*p[i];
	sum=sqrt(sum);
	if(sum > stpmax)
		for(i=1;i<=n;i++) p[i] *= stpmax/sum;
	for(slope=0.0,i=1;i<=n;i++)
		slope+=g[i]*p[i];
	if(slope >= 0.0) {
		*check=1;
		printf("WARNING[NUMREC] : Roundoff problem in lnsrch()\n");
		return;
	} else {
		test=0.0;
		for(i=1;i<=n;i++) {
			temp=fabs(p[i])/GDL_MAX_DBL(fabs(xold[i]), 1.0);
			if(temp > test) test=temp;
		}
		alamin=GDL_NREC_TOLX/test;
		alam=1.0;
		for(;;) {
			for(i=1;i<=n;i++)
				x[i]=xold[i]+alam*p[i];
			*f=(*func)(x, param, &status);
			if (!status) {
				*check=1;
				return;					
			}
			if (alam < alamin) {
				for(i=1;i<=n;i++) x[i]=xold[i];
				*check=1;
				return;
			} else if (*f <= fold+GDL_NREC_ALF*alam*slope) return;
			else {
				if (alam == 1.0)
					tmplam = -slope/(2.0*(*f-fold-slope));
				else {
					rhs1= *f-fold-alam*slope;
					rhs2=f2-fold-alam2*slope;
					a=(rhs1/(alam*alam)-rhs2/(alam2*alam2))/(alam-alam2);
					b=(-alam2*rhs1/(alam*alam)+alam*rhs2/(alam2*alam2))/(alam-alam2);
					if (a == 0.0) tmplam = -slope/(2.0*b);
					else {
						disc=b*b-3.0*a*slope;
						if (disc < 0.0) tmplam=0.5*alam;
						else if (b <= 0.0) tmplam=(-b+sqrt(disc))/(3.0*a);
						else tmplam=-slope/(b+sqrt(disc));
					}
					if (tmplam > 0.5*alam)
						tmplam=0.5*alam;
				}	
			}
			alam2=alam;
			f2 = *f;
			alam=GDL_MAX_DBL(tmplam,0.1*alam);
		}
	}
}

/**
 * 
 */
int draw_ball(double cumul[], int nb, gdl_rng *rng)
{
    double val, test;
    int beg, end, mid;

    val = gdl_rng_uniform(rng);
    
    if (val <= cumul[1])
		return 1;
	if (val > cumul[nb])
		return nb;
   
    beg = 1;
    end = nb;
    
    while (end - beg > 1) {
		
		mid = (beg + end) / 2;
		test = cumul[mid];
		
		if (val < test)
		    end = mid;
		else if (val > test)
		    beg = mid;
		else
		    return mid;
		    
    }
    
    return end;
}
/**
 * 
 */
int draw_ball_tot(double cumul[], double tot, int nb, gdl_rng *rng)
{
    double val, test;
    int beg, end, mid;

    val = tot*gdl_rng_uniform(rng);
    
    if (val <= cumul[1])
		return 1;
	if (val > cumul[nb])
		return nb+1;
   
    beg = 1;
    end = nb;
    
    while (end - beg > 1) {
		
		mid = (beg + end) / 2;
		test = cumul[mid];
		
		if (val < test)
		    end = mid;
		else if (val > test)
		    beg = mid;
		else
		    return mid;
		    
    }
    
    return end;
}
/**
 * 
 */
int draw_ball_from_to(double cumul[], double from, double to, int nb, gdl_rng *rng)
{
    double val, test;
    int beg, end, mid;

    val = from + (to-from)*gdl_rng_uniform(rng);
    
    if (val <= cumul[1])
		return 1;
	if (val > cumul[nb])
		return nb+1;
   
    beg = 1;
    end = nb;
    
    while (end - beg > 1) {
		
		mid = (beg + end) / 2;
		test = cumul[mid];
		
		if (val < test)
		    end = mid;
		else if (val > test)
		    beg = mid;
		else
		    return mid;
		    
    }
    
    return end;
}
/**
 * This just balance the elements of the vector w[1..n]
 * by first getting the max value of the vector and then
 * computing a balancing threshold which is the input
 * tolerance tol multiplied by the max value.
 */
void dbalance_vector(double w[], int n, double tol) {
 	int i;
 	double wmax, thresh;
 	
 	wmax = 0.0;
	for(i=1;i<=n;i++)
		if (w[i] > wmax)
			wmax=w[i];
	 thresh = tol*wmax;
	 for(i=1;i<=n;i++) 
		if (w[i] < thresh)
			w[i] = 0.0;
} 
/**
 * Perform a Moore-Penrose inversion of the matrix m.
 * The pseudo-invers is output as im
 * 
 * s and v are buffer memory for SVD.
 */
//void moore_penrose_inversion(double **m, int n, int p, double **im, double *s, double **v)
//{
//	int i,j,l;
//	
//	if (p <= n) {
//		dsvdcmp(m, n, p, s, v);
//		/* Balance singular values */
//		dbalance_vector(s, p, GDL_NREC_TOL);
//	    for(i=1;i<=p;i++)
//			for(j=1;j<=n;j++)
//				for(im[i][j]=0.0,l=0;l<=p;l++)
//					if (s[l] != 0.0)
//						im[i][j] += v[i][l]*m[j][l]/s[l];
//	} else {
//		printf("WARNING: Cannot perform Moore-Penrose Inversion (matrix need to be transposed before !)\n");
//	}
//}
//
/**
 * Balance the columns of the matrix x[nh..nl][ph..pl]
 * using a threshold of eps.
 * @param x[nh..nl][ph..pl] the matrix to balance.
 * @param eps the balancing threshold (1.e-6 work well)
 * @param nh first row index
 * @param nl last row index
 * @param ph first col index
 * @param pl lest col index
 */
void dbalance_matrix_column(double **x, double eps, int nl, int nn, int pl, int pp)
{
  int ii, jj, kk;
  double emin, emax;
  
  if (eps > 1.0) eps = GDL_NREC_TOL;
  
  if (x != NULL) {
  	for(jj=pl;jj<pl+pp;jj++) {
  		for(emax=GDL_NEGINF,kk=nl;kk<nl+nn;kk++)
  			if (x[kk][jj] > emax) emax = x[kk][jj];
	  	emin = fabs(emax)*eps;
	  	for(ii=nl;ii<nl+nn;ii++)
	  		if (fabs(x[ii][jj]) < emin)	x[ii][jj] = 0.0;
    }
  }
}
/**
 * Balance the rows of the matrix x[nh..nl][ph..pl]
 * using a threshold of eps.
 * @param x[nh..nl][ph..pl] the matrix to balance.
 * @param eps the balancing threshold (1.e-6 work well)
 * @param nh first row index
 * @param nl last row index
 * @param ph first col index
 * @param pl lest col index
 */
void dbalance_matrix_row(double **x, double eps, int nl, int nn, int pl, int pp) {
  int ii, jj,kk;
  double emin, emax;
  
  if (eps > 1.0) eps = GDL_NREC_TOL;
  
  if (x != NULL) {
  	for(ii=nl;ii<nl+nn;ii++) {
  		for(emax=GDL_NEGINF,kk=pl;kk<pl+pp;kk++)
  			if (x[ii][kk] > emax) emax = x[ii][kk];
	  	emin = fabs(emax)*eps;
	  	for(jj=pl;jj<pl+pp;jj++)
	  		if (fabs(x[ii][jj]) < emin)	x[ii][jj] = 0.0;
    }
  }	
}
/**
 * Given a positive-definite symmetric matrix a[1..n][1..n], this routine
 * constructs its Cholesky decomposition, A=L.L'. On input, only the triangle
 * of a need be given; it is not modified. The Cholesky factor L is returned in
 * the lower traingle of a, except for its diagonal elements which are returned
 * in p[1..n].
 * It returns 1 if succeed, 0 otherwise.
 */
int dcholdc(double **a, int n, double p[])
{
	int i,j,k;
	double sum;
	
	for(i=1;i<=n;i++) {
		for(j=i;j<=n;j++) {
			for(sum=a[i][j],k=i-1;k>=1;k--) sum -= a[i][k]*a[j][k];
			if (i==j) {
				if (sum <= 0.0) 
					return 0;
				p[i]=sqrt(sum);
			} else a[j][i]=sum/p[i];
		}
	}
	return 1;
}
/**
 * 
 */
double dgolden(double ax, double bx, double cx, double (*f)(double), double tol, double *xmin)
{
#define R 0.61803399
#define C (1.0-R)
#define SHIFT2(a,b,c) (a)=(b);(b)=(c);
#define SHIFT3(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);

    size_t iter;
	double f1,f2,x0,x1,x2,x3;
	
	x0=ax;
	x3=cx;
	if (fabs(cx-bx) > fabs(bx-ax)) {
		x1=bx;
		x2=bx+C*(cx-bx);
	} else {
		x2=bx;
		x1=bx-C*(bx-ax);
	}
	f1=(*f)(x1);
	f2=(*f)(x2);
	iter=0;
	while(fabs(x3-x0) > tol*(fabs(x1)+fabs(x2))) {
		if (f2 < f1) 
		{
			SHIFT3(x0,x1,x2,R*x1+C*x3);    
			SHIFT2(f1,f2,(*f)(x2));
		}
		else
		{
			SHIFT3(x3,x2,x1,R*x2+C*x0);
			SHIFT2(f2,f1,(*f)(x1));
		} 
		iter++;
		if (iter > 1000)
		{
			break;			
		}
	}
	if (f1 < f2) {
		*xmin=x1;
		return f1;
	} else {
		*xmin=x2;
		return f2;	
	}
#undef SHIFT2
#undef SHIFT3
#undef R
#undef C
}

long
gdl_factorial (size_t n)
{
	if (n == 0 || n == 1)
	{
		return 1;	
	}
	else
	{
		return n*gdl_factorial (n-1);	
	}
}
