/*  
 * 	odb/header.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_odb.h>

static gdl_oheader *
oheader_fscanf (FILE * stream)
{
  int ch;
  char * tmp;
  gdl_oheader * header;
  
  header = GDL_MALLOC (gdl_oheader, 1);
  tmp  = GDL_CALLOC (char, GDL_MAXLINE);
  // id;
  ch = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
  header->id = atoi (tmp);
  // date;
  ch = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
  header->date = gdl_string_clone (tmp);
  // driver_name ;
  ch = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
  header->driver_name = gdl_string_clone (tmp);
  // driver_version ;
  ch = gdl_get_next_token (tmp, GDL_MAXLINE, stream);
  header->driver_version = gdl_string_clone (tmp);
  // meta;
  header->meta = gdl_meta_fscanf (stream);
  // positionate the stream at the begining of the result
  for (;ch != EOF && ch != '\n'; ch = fgetc (stream));
  ch = fgetc (stream); // shift '\n'
  
  GDL_FREE (tmp);
  
  return header;
}

static int
oheader_fprintf (FILE * stream, gdl_oheader * header, int gout)
{
  int ch;
  char * tmp;
  
  if (stream == 0 || header == 0)
  	return GDL_EINVAL;
  
  if (!gout)
  {
	  // id;
	  fprintf (stream, "%d", header->id);
	  // date
	  fprintf (stream, "\t%s", header->date);
	  // driver name;
	  fprintf (stream, "\t%s", header->driver_name);
	  // driver version;
	  fprintf (stream, "\t%s", header->driver_version);
	  // meta;
	  fprintf (stream, "\n");
	  gdl_meta_fprintf (stream, header->meta);
  }
  else
  {
  	  // id;
	  fprintf (stream, "%d\t", header->id);
	  // meta name
	  gdl_meta_name_fprintf (stream, header->meta);
	  // id;
	  fprintf (stream, "%d\t", header->id);
	  // meta name
	  gdl_meta_value_fprintf (stream, header->meta);
  }
  
}

static int
odb_rheader (gdl_odb * r)
{
	gdl_oheader * header = oheader_fscanf (r->_stream);
	
	if (header == 0)
	{
		return 1;
	}
	
	gdl_list_push_back (r->header, header, 1);
	
	return 0;
}

static int
odb_load_header (gdl_odb * r)
{
	int i, status;
	
	status = odb_ropen (r, 1);
	
	for (i = 0; i < r->size; i++)
	{
		status = odb_rnext (r);
		status = odb_rheader (r);
	}
		
	status = odb_close (r);
	
	return GDL_SUCCESS;
}

static int
odb_dump_header (gdl_odb * r, FILE * stream, int gout)
{
	gdl_oheader * header;
	gdl_list_itr * itr;
	
	itr = gdl_list_iterator_front (r->header);
	
	do 
	{
		header = (gdl_oheader *) gdl_list_iterator_value (itr);
		oheader_fprintf (stream, header, gout);
	}
	while (gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	return GDL_SUCCESS;	
}

int
gdl_odb_fsummary (gdl_odb * r, FILE * stream, int gout)
{
	int status = odb_load_header (r);
	if (status == GDL_SUCCESS)
	{
		status = odb_dump_header (r, stream, gout);
	}
	return status;
}
