/*  
 * 	phase/em_pl_hv.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <float.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gpartition.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_view.h>
#include <gdl/gdl_phase_em.h>
#include <gdl/gdl_phase_em_pl.h>

typedef struct
{
	size_t psize;
	size_t kbest;
	double pmt;
	double phcrm;
	const gdl_view * v;
	const gdl_gview * g;
	const gdl_mask * m;
	const gdl_rng   * r;
	const gdl_gligation_type * T;
	gdl_gpartition  * gpart;
	double res_abs;
	double res_sq;
} gdl_phase_em_pl_hview_t;

int
gdl_phase_em_pl_hview_alloc (void * vstate, const gdl_phase_em_pl_param * P)
{
	gdl_phase_em_pl_hview_t * state;
	
	state = (gdl_phase_em_pl_hview_t *) vstate;
	
	state->v = P->v;
	
	state->g = gdl_view_get_gview (P->v);
	
	if (state->g == 0)
	{
		return GDL_FAILURE;	
	}
	
	state->m = P->m;
	
	state->r = P->r;
	
	if (state->r == 0)
	{
		return GDL_FAILURE;	
	}
	
	state->pmt   = P->pmt;
	
	state->phcrm = P->phcrm;
	
	state->psize = P->psize;
	
	state->kbest = P->kbest;
	
	return GDL_SUCCESS;
}

static void
gdl_phase_em_pl_hview_state_free (void * data)
{
	gdl_phase_em_workspace * w = (gdl_phase_em_workspace *) data;
	gdl_hview * h = (gdl_hview *) gdl_phase_em_workspace_result (w);
	
	gdl_phase_em_workspace_free (w);
	gdl_hview_free (h);
}

int
gdl_phase_em_pl_hview_free (void * vstate)
{
	if (vstate)
	{
		gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
		
		gdl_gpartition_free (state->gpart, &gdl_phase_em_pl_hview_state_free);
				
		return GDL_SUCCESS;
	}
}

static gdl_gpartition_state **
gdl_phase_em_pl_hview_partition (const gdl_gview * g, const gdl_mask * m, size_t * ns, void * extra)
{
	gdl_phase_em_pl_hview_t * state = (gdl_phase_em_pl_hview_t *) extra;
	gdl_phase_em_param * param;
	gdl_gpartition_state ** pstates;
	gdl_list * pbuff;
	gdl_list_itr * itr;
	gdl_gview_gmask_itr * gitr;
	size_t i;
	
	pbuff = gdl_list_alloc (gdl_list_default);
	
	gitr = gdl_gview_gmask_iterator (state->g, state->m, gdl_gview_gmask_heterozygous, state->psize);
	
	do
	{
		gdl_list_push_back (pbuff, gdl_gview_gmask_iterator_value (gitr), 0);
	}
	while (gdl_gview_gmask_iterator_next (gitr));
	
	gdl_gview_gmask_iterator_free (gitr);
	
	*ns = gdl_list_size (pbuff);
	
	pstates = GDL_MALLOC (gdl_gpartition_state *, *ns);
	
	itr = gdl_list_iterator_front (pbuff);
	i = 0;
	do
	{
		gdl_phase_em_workspace * w;
		gdl_mask * pmask;
		
		pmask = (gdl_mask *) gdl_list_iterator_value (itr);
		
		param = gdl_phase_em_param_alloc (state->v, pmask, state->r, state->pmt);
		
		w = gdl_phase_em_workspace_alloc (gdl_phase_em_hview, param);
		
		gdl_phase_em_workspace_init_rng (w);
		
		pstates[i]       = GDL_MALLOC (gdl_gpartition_state, 1);
		pstates[i]->mask = pmask;
		pstates[i]->data = w;
		
		i++;
	}
	while (gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	gdl_list_free (pbuff);
	
	return pstates;
}

int
gdl_phase_em_pl_hview_init (void * vstate, const gdl_gligation_type * T)
{
	int status;
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	
	state->T = T;
	
	state->gpart = gdl_gpartition_alloc (state->g, state->m);
	
	return gdl_gpartition_init (state->gpart, &gdl_phase_em_pl_hview_partition, state);
}

size_t
gdl_phase_em_pl_hview_psize (void * vstate)
{
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	return gdl_gpartition_eligible_size (state->gpart, state->T);
}

int 
gdl_phase_em_pl_hview_iterate (void * vstate, size_t i)
{
	int status;
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	gdl_gpartition_state * pstate;
	gdl_phase_em_workspace * w;
	
	pstate = gdl_gpartition_get (state->gpart, i);
	
	w = (gdl_phase_em_workspace *) pstate->data;
	
	status = gdl_phase_em_workspace_iterate (w);
	
	state->res_abs = gdl_phase_em_workspace_residual_abs (w);
	state->res_sq  = gdl_phase_em_workspace_residual_sq (w);
	
	return status;
}

static int
gdl_phase_em_pl_hview_state_ligation (gdl_gpartition_state * cur,
                                      const gdl_gpartition_state * next,
                                      const gdl_gligation_type * T,  
                                      void * extra)
{
	gdl_phase_em_pl_hview_t * state = (gdl_phase_em_pl_hview_t *) extra;
	gdl_phase_em_workspace * w1, * w2;
	gdl_phase_em_param * param;
	gdl_hview * h1, * h2, * kh1, * kh2;
	gdl_mask * m;
	
	w1 = (gdl_phase_em_workspace *) cur->data;
	w2 = (gdl_phase_em_workspace *) next->data;
	
	h1 = (gdl_hview *) gdl_phase_em_workspace_result (w1);
	h2 = (gdl_hview *) gdl_phase_em_workspace_result (w2);
	
	kh1 = gdl_hview_kbest (h1, state->kbest);
	kh2 = gdl_hview_kbest (h2, state->kbest);
	
	if (state->phcrm > 0.0)
	{
		gdl_hview_hconfig_remove (kh1, state->phcrm);
		gdl_hview_hconfig_remove (kh2, state->phcrm);
	}
	
	m = gdl_hview_locus_ligation (kh1, kh2);
	
	gdl_hview_free (h1);
	gdl_phase_em_workspace_free (w1);
	
	param = gdl_phase_em_param_alloc (state->v, m, state->r, state->pmt);
	
	w1 = gdl_phase_em_workspace_alloc (gdl_phase_em_hview, param);
	
	gdl_phase_em_workspace_init_static (w1, kh1);
	
	gdl_mask_free (cur->mask);
	
	cur->mask = m;
    cur->data = w1;
    
    gdl_hview_free (kh2);
		
	return GDL_SUCCESS;
}          

int 
gdl_phase_em_pl_hview_ligation (void * vstate)
{
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	
	return gdl_gpartition_ligation (state->gpart,
	                          state->T,
	                          &gdl_phase_em_pl_hview_state_ligation,
	                          &gdl_phase_em_pl_hview_state_free,
	                          0,
	                          state);
}

double
gdl_phase_em_pl_hview_loglikelihood (void * vstate)
{
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	size_t i, np;
	double log = 0;
	
	np = gdl_gpartition_size (state->gpart);
	
	for (i = 0; i < np; i++)
	{
		gdl_gpartition_state * pstate = gdl_gpartition_get (state->gpart, i);
		gdl_phase_em_workspace * w = (gdl_phase_em_workspace *) pstate->data;
		log += gdl_phase_em_workspace_loglikelihood (w);
	}
	
	return log;
}

double
gdl_phase_em_pl_hview_residual_abs (void * vstate)
{
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	return state->res_abs;
}

double
gdl_phase_em_pl_hview_residual_sq (void * vstate)
{
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	return state->res_sq;
}

void *
gdl_phase_em_pl_hview_result  (void * vstate)
{
	gdl_phase_em_pl_hview_t * state 
		    = (gdl_phase_em_pl_hview_t *) vstate;
	gdl_gpartition_state * pstate;
	gdl_phase_em_workspace * w;
	
	pstate = gdl_gpartition_get (state->gpart, 0);
	
	w = (gdl_phase_em_workspace *) pstate->data;
	
	return gdl_phase_em_workspace_result (w);
}

int
gdl_phase_em_pl_hview_fread  (FILE * stream, void * vstate)
{
}

int
gdl_phase_em_pl_hview_fwrite (FILE * stream, const void * vstate)
{
}

static const gdl_phase_em_pl_workspace_type _em_pl_hview =
{
    "gdl_phase_em_pl_hview",
    sizeof (gdl_phase_em_pl_hview_t),
    &gdl_phase_em_pl_hview_alloc,
    &gdl_phase_em_pl_hview_free,
    &gdl_phase_em_pl_hview_init,
    &gdl_phase_em_pl_hview_psize,
    &gdl_phase_em_pl_hview_iterate,
    &gdl_phase_em_pl_hview_ligation,
    &gdl_phase_em_pl_hview_loglikelihood,
    &gdl_phase_em_pl_hview_residual_abs,
    &gdl_phase_em_pl_hview_residual_sq,
    &gdl_phase_em_pl_hview_result,
    &gdl_phase_em_pl_hview_fread,
    &gdl_phase_em_pl_hview_fwrite
};

const gdl_phase_em_pl_workspace_type * gdl_phase_em_pl_hview = &_em_pl_hview;
