/*  
 *  ppca/ppca.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:42 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_ppca.h>

struct _gdl_ppca_workspace
{
	const gdl_ppca_workspace_type * type;
	const gdl_rng * rng;
	size_t _naxe;
	gdl_vector * tw;
	void * state;
};

gdl_ppca_workspace *
gdl_ppca_workspace_alloc (const gdl_ppca_workspace_type * T, const gdl_rng * rng)
{
	gdl_ppca_workspace * p;
	
	p = GDL_CALLOC (gdl_ppca_workspace, 1);
	
	p->type = T;
	
	p->rng = rng; 
	
	p->tw = 0;
	
	return p;
}

void
gdl_ppca_workspace_free (gdl_ppca_workspace * w)
{
	if (w)
	{
		GDL_FREE (w);
	}
}

static int
_gdl_ppca_workspace_init (gdl_ppca_workspace * w, gdl_matrix * X)
{
	(w->type->free)(w->state);
	
	GDL_FREE (w->state);
	
	w->state = gdl_malloc (w->type->size);
	
	return(w->type->alloc)(w->state, X->size1, X->size2, w->rng);
}

static void
_gdl_ppca_compute_tracy_widom (gdl_ppca_workspace * w, const gdl_matrix * X)
{
	const gdl_vector * l = gdl_ppca_workspace_weights (w);
	const size_t m       =  X->size1;
	const size_t n       =  X->size2;
	size_t i, j;
	double u, v, x, t, s, s2, t2, t3, nn, ep;
	
	if (w->tw)
	{
		gdl_vector_free (w->tw);
	}
	
	w->tw = gdl_vector_calloc (l->size);
	
	t = 0;
	for (i = 0; i < X->size1; i++)
	{
		for (j = 0; j < X->size2; j++)
		{
			x = gdl_matrix_get (X, i, j);
			t += x*x;
		}
	}
	for (i = 0; i < l->size; i++)
	{
		if (X->size1 > X->size2)
		{
			u=sqrt(m-1)+sqrt(n-i-1);
			v=u*pow(1/sqrt(m-1)+1/sqrt(n-i-1),1./3.);
			u*=u;
			x=m*(n-i-1)*gdl_vector_get(l,i)/t;
		}
		else
		{
			u=sqrt(n-1)+sqrt(m-i-1);
			v=u*pow(1/sqrt(n-1)+1/sqrt(m-i-1),1./3.);
			u*=u;
			x=n*(m-i-1)*gdl_vector_get(l,i)/t;
		}
		gdl_vector_set(w->tw, i, (x-u)/v);
		x=gdl_vector_get(l,i);
		t-=x;
	}
}

int
gdl_ppca_workspace_perform (gdl_ppca_workspace * w, gdl_matrix * X, size_t naxe, double eps, size_t maxi)
{
	if (naxe <= 0)
	{
		GDL_ERROR_VAL ("Number of PCA axes invalid\n", GDL_EINVAL, GDL_EINVAL);	
	}
	
	_gdl_ppca_workspace_init (w, X);
	
	(w->type->perform)(w->state, X, naxe, eps, maxi);
	
	w->_naxe = naxe;
	
	_gdl_ppca_compute_tracy_widom (w, X);
	
	return GDL_SUCCESS;
}

int
gdl_ppca_workspace_wperform (gdl_ppca_workspace * pca, gdl_matrix * X, const gdl_vector * w, size_t naxe, double eps, size_t maxi)
{
	if (naxe <= 0)
	{
		GDL_ERROR_VAL ("Number of PCA axes invalid\n", GDL_EINVAL, GDL_EINVAL);	
	}
	if (w->size < X->size1)
	{
		GDL_ERROR_VAL ("The vector of weights must have a size which matches the number of rows\n", GDL_EINVAL, GDL_EINVAL);	
	}
	
	_gdl_ppca_workspace_init (pca, X);
	
	(pca->type->wperform)(pca->state, X, w, naxe, eps, maxi);
	
	pca->_naxe = naxe;
	
	_gdl_ppca_compute_tracy_widom (pca, X);
	
	return GDL_SUCCESS;
}

const gdl_matrix *
gdl_ppca_workspace_projection (const gdl_ppca_workspace * w)
{
	return (w->type->projection)(w->state);
}

const gdl_matrix *
gdl_ppca_workspace_rotation (const gdl_ppca_workspace * w)
{
	return (w->type->rotation)(w->state);
}

const gdl_vector *
gdl_ppca_workspace_weights (const gdl_ppca_workspace * w)
{
	return (w->type->weights)(w->state);
}

const gdl_vector *
gdl_ppca_workspace_tracy_widom (const gdl_ppca_workspace * w)
{
	return w->tw;
}
