/*  
 *  ppca/standard.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:42 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_statistics.h>
#include <gdl/gdl_ppca.h>

#include <gdl/gdl_linalg_svd.h>

typedef struct
{
	size_t size1;
	size_t size2;
	size_t naxe;
	gdl_linalg_svd_workspace * svd;
	gdl_matrix * E;
	gdl_matrix * Y;
	gdl_vector * S;
	double stot;
	double env;
	const gdl_rng * rng;
} gdl_ppca_std_t;

static int
gdl_ppca_std_alloc (void * vpca, size_t size1, size_t size2, const gdl_rng * rng)
{
	if (vpca)
	{
		gdl_ppca_std_t * pca = (gdl_ppca_std_t *) vpca;
		
		pca->size1 = size1;
		pca->size2 = size2;
		
		pca->rng   = rng;
		
		pca->E = pca->Y = 0;
		pca->S = 0;
		
		return GDL_SUCCESS;
	}
}

static int
gdl_ppca_std_free (void * vpca)
{
	if (vpca)
	{
		gdl_ppca_std_t * pca = (gdl_ppca_std_t *) vpca;
		gdl_matrix_free (pca->E);
		gdl_matrix_free (pca->Y);
		gdl_vector_free (pca->S);
		return GDL_SUCCESS;
	}
} 

static int
gdl_ppca_std_em (gdl_ppca_std_t * pca, gdl_matrix * Y, size_t naxe, double eps, size_t maxi)
{
	size_t i, j, k, missing = 0;
	double ss, ss_old;
	gdl_vector * w, * s;
	gdl_matrix * C, *CC, * CtC, * gCtC, * X, * XX, * XtX, * gXtX, * U, * V;
	
	C    = gdl_matrix_alloc (Y->size2, naxe);
	CC   = gdl_matrix_alloc (Y->size2, naxe);
	CtC  = gdl_matrix_alloc (naxe, naxe);
	gCtC = gdl_matrix_alloc (naxe, naxe);
	X    = gdl_matrix_alloc (Y->size1, naxe);
	XX   = gdl_matrix_alloc (Y->size1, naxe);
	XtX  = gdl_matrix_alloc (naxe, naxe);
	gXtX = gdl_matrix_alloc (naxe, naxe);
	V    = gdl_matrix_alloc (naxe, naxe);
	w    = gdl_vector_alloc (naxe);
	s    = gdl_vector_alloc (naxe);
	
	// Initialization
	for(i = 0; i < C->size1; i++)
		for (j = 0; j < C->size2; j++)
			gdl_matrix_set (C, i, j, gdl_rng_uniform (pca->rng));
	// EM
	for (i = 0; i < maxi; i++)
	{
		gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1.0, C, C, 0, CtC);
		gdl_linalg_SV_ginv (CtC, V, s, w, gCtC);
		if (missing)
		{
		   // inpute missing data
		}
		// E Step   
      gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, C, CtC, 0, CC);
      gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, Y, CC, 0, X);
      // M Step
      gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1.0, X, X, 0, XtX);
      gdl_linalg_SV_ginv (XtX, V, s, w, gXtX);
      gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1.0, Y, X, 0, CC);
      gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, CC, gXtX, 0, C);
      // Compute residual error
	}
	// Extract an orthogonal base from C
	gdl_linalg_SV_decomp (C, V, s, w);
	gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, Y, C, 0, X);
	gdl_linalg_SV_decomp (X, V, s, w);
	gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, C, V, 0, CC);
	gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, Y, CC, 0, X);
	// scale X to get the eigenvalues...
	for(i = 0; i < X->size2; i++)
	{
		gdl_vector_view v = gdl_matrix_column (X, i);
		gdl_vector_set (s, i, pow (gdl_blas_dnrm2 (&(v.vector)),2));
	}
	//
	pca->S = s;
	pca->E = CC;
	pca->Y = X;
	
	// Clean workspace
	gdl_matrix_free (C);
	gdl_matrix_free (CtC);
	gdl_matrix_free (gCtC);
	gdl_matrix_free (XX);
	gdl_matrix_free (XtX);
	gdl_matrix_free (gXtX);
	gdl_matrix_free (V);
	gdl_vector_free (w);
}

static int
gdl_ppca_std_perform (void * vpca, gdl_matrix * X, size_t naxe, double eps, size_t maxi)
{
	gdl_ppca_std_t * pca = (gdl_ppca_std_t *) vpca;
	
	gdl_ppca_std_em (pca, X, naxe, eps, maxi);
}

static int
gdl_ppca_std_wperform (void * vpca, gdl_matrix * X, const gdl_vector * w, size_t naxe, double eps, size_t maxi)
{
	gdl_ppca_std_t * pca = (gdl_ppca_std_t *) vpca;
	
	gdl_ppca_std_em (pca, X, naxe, eps, maxi);
}

static gdl_matrix *
gdl_ppca_std_rotation (void * vpca)
{
	gdl_ppca_std_t * pca = (gdl_ppca_std_t *) vpca;
	return pca->E;
}

static gdl_vector *
gdl_ppca_std_weights (void * vpca)
{
	gdl_ppca_std_t * pca = (gdl_ppca_std_t *) vpca;
	return pca->S;
}

static gdl_matrix *
gdl_ppca_std_projection (void * vpca)
{
	gdl_ppca_std_t * pca = (gdl_ppca_std_t *) vpca;
	return pca->Y;
}

static const gdl_ppca_workspace_type _gdl_ppca_standard =
{
	"gdl_ppca_standard",
	sizeof (gdl_ppca_std_t),
	&gdl_ppca_std_alloc,
	&gdl_ppca_std_free,
	&gdl_ppca_std_perform,
	&gdl_ppca_std_wperform,
   &gdl_ppca_std_projection,
	&gdl_ppca_std_rotation,
	&gdl_ppca_std_weights
};

const gdl_ppca_workspace_type * gdl_ppca_standard = &_gdl_ppca_standard;
