/*  
 *  pstruct/plot.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_glabels.h>
#include <gdl/gdl_graphics.h>
#include <gdl/gdl_plot.h>
#include <gdl/gdl_pstruct.h>
#include <gdl/gdl_pstruct_result.h>

static const double GDL_PSTRUCT_PLOT_BAR_WIDTH    = 300;
static const double GDL_PSTRUCT_PLOT_LABEL_WIDTH  = 100;
static const double GDL_PSTRUCT_PLOT_BAR_HEIGHT   = 10;

typedef struct
{
	double height;
	double width;
	const gdl_pstruct_result * result;
} gdl_pstruct_plot_t;

static void
gdl_pstruct_plot_default_parameters (gdl_pstruct_plot_t * plot, gdl_plot_parameters * params)
{
	gdl_plot_parameter * par;

	par = gdl_plot_parameters_get (params, "bar-height");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_PSTRUCT_PLOT_BAR_HEIGHT);
		gdl_plot_parameters_set (params, "bar-height", par);
	}
	par = gdl_plot_parameters_get (params, "bar-width");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_PSTRUCT_PLOT_BAR_WIDTH);
		gdl_plot_parameters_set (params, "bar-width", par);
	}
	par = gdl_plot_parameters_get (params, "label-width");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_double, 1);
		gdl_plot_parameter_set_copy (par, 0, &GDL_PSTRUCT_PLOT_LABEL_WIDTH);
		gdl_plot_parameters_set (params, "label-width", par);
	}
	par = gdl_plot_parameters_get (params, "pop-color");
	if (!par || gdl_plot_parameter_size (par) < gdl_pstruct_result_population_size (plot->result))
	{
		size_t i, nk;
		gdl_rgb_color_vector * palette;
		
		nk      = gdl_pstruct_result_population_size (plot->result);
		palette = gdl_rgb_color_palette (nk);
		par     = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, nk);
		
		for (i  = 0; i < nk; i++)
		{
			gdl_plot_parameter_set_copy (par, i, palette->colors[i]);
		}
		
		gdl_plot_parameters_set (params, "pop-color", par);
		
		gdl_rgb_color_vector_free (palette);
	}
	par = gdl_plot_parameters_get (params, "label-color");
	if (!par)
	{
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_rgb, 1);
		gdl_plot_parameter_set (par, 0, gdl_rgb_color_get_black ());
		gdl_plot_parameters_set (params, "label-color", par);
	}
	par = gdl_plot_parameters_get (params, "label-font");
	if (!par)
	{
		gdl_font * font = gdl_font_default ();
		font->size = GDL_PSTRUCT_PLOT_BAR_HEIGHT;
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_font, 1);
		gdl_plot_parameter_set (par, 0, font);
		gdl_plot_parameters_set (params, "label-font", par);
	}
	par = gdl_plot_parameters_get (params, "accession-list");
	if (!par)
	{
		size_t i, na, * index;
		const gdl_glabels * labels = gdl_pstruct_result_labels (plot->result);
		na = gdl_pstruct_result_accession_size (plot->result);
		index = gdl_pstruct_result_qsort (plot->result);
		par = gdl_plot_parameter_alloc (gdl_plot_parameter_string, na);
		for (i = 0; i < na; i++)
		{
			const gdl_string * name = gdl_glabels_accession (labels, index[i]);
			gdl_plot_parameter_set_copy (par, i, name);
		}
		gdl_plot_parameters_set (params, "accession-list", par);
		GDL_FREE (index);
	}	
}

static int
gdl_pstruct_plot_create (gdl_pstruct_plot_t * p, gdl_plot_parameters * params)
{
	const gdl_pstruct_result * r = p->result;
	gdl_plot_parameter * par;
	double * v;
	
	par = gdl_plot_parameters_get (params, "bar-height");
	v   = (double *) gdl_plot_parameter_get (par, 0);
	p->height = (*v)*gdl_pstruct_result_accession_size (r);
	par = gdl_plot_parameters_get (params, "bar-width");
	v   = (double *) gdl_plot_parameter_get (par, 0);
	p->width  = (*v);
	par = gdl_plot_parameters_get (params, "label-width");
	v   = (double *) gdl_plot_parameter_get (par, 0);
	p->width += (*v);
}

static int
gdl_pstruct_plot_alloc (void * plot, gdl_plot_parameters * par, const void * data, double * width, double * height)
{
	gdl_pstruct_plot_t * p;
	
	p = (gdl_pstruct_plot_t *) plot;
	
	p->result = (gdl_pstruct_result *) data;
	
	gdl_pstruct_plot_default_parameters (p, par);
	
	gdl_pstruct_plot_create (p, par);
	
	*width  = p->width;
	*height = p->height;
	
	return GDL_SUCCESS;
}

static int
gdl_pstruct_plot_free (void * plot)
{
	if (plot)
	{
		gdl_pstruct_plot_t * p = (gdl_pstruct_plot_t *) plot;
	}
}

static int
gdl_pstruct_plot_draw (void * plot, const gdl_plot_parameters * par, gdl_graphics2D * graph)
{
	int a;
	size_t i, j, na, nk;
	double x, y, * height, * w, width;
	gdl_rgb_color * c0;
	gdl_font      * f0;
	gdl_plot_parameter * colors;
	gdl_plot_parameter * alist;
	
	gdl_pstruct_plot_t * p       = (gdl_pstruct_plot_t *) plot;
	const gdl_pstruct_result * r = p->result;
	const gdl_glabels * labels   = gdl_pstruct_result_labels (r);
	
	c0 = gdl_graphics2D_get_color (graph);
	f0 = gdl_graphics2D_get_font (graph);
	
	nk = gdl_pstruct_result_population_size (r);
	
	height  = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "bar-height"), 0);
	
	colors = gdl_plot_parameters_get (par, "pop-color");
	alist  = gdl_plot_parameters_get (par, "accession-list");
	
	gdl_graphics2D_set_font  (graph, (gdl_font *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "label-font"), 0));
	gdl_graphics2D_set_color (graph, (gdl_rgb_color *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "label-color"), 0));
	
	na = gdl_plot_parameter_size (alist);
	
	y = 0;
	for (i = 0; i < na; i++)
	{
		const gdl_string * name = (gdl_string *) gdl_plot_parameter_get (alist, i);
		a = gdl_glabels_search_accession (labels, name);
		if (a<0)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Unable to find accession %s", name), GDL_FAILURE, GDL_FAILURE);	
		}		
		x = 0;
		gdl_graphics2D_draw_text(graph, x, y+*height, name);
		w = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "label-width"), 0);
		x = *w;
		for (j = 0; j < nk; j++)
		{
			w = (double *) gdl_plot_parameter_get (gdl_plot_parameters_get (par, "bar-width"), 0);
			width = (*w) * gdl_pstruct_result_get_accession_q (r, j, a);
			gdl_graphics2D_set_paint(graph, (gdl_rgb_color *) gdl_plot_parameter_get (colors, j));
			gdl_graphics2D_fill_rect(graph, x, y, width, *height);
			x += width;
		}
		y += *height;
	}
	
	gdl_graphics2D_set_color (graph, c0);
	gdl_graphics2D_set_font (graph, f0);
	
	return GDL_SUCCESS;
}

static const gdl_plot_type _gdl_pstruct_result_plot =
{
	"gdl_pstruct_result",
	sizeof (gdl_pstruct_plot_t),
	&gdl_pstruct_plot_alloc,
	&gdl_pstruct_plot_free,
	&gdl_pstruct_plot_draw,
	NULL
};

const gdl_plot_type * gdl_pstruct_result_plot = &_gdl_pstruct_result_plot;

