/*  
 *  pstruct/pstruct.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h> 
#include <gdl/gdl_hview.h>
#include <gdl/gdl_view.h>
#include <gdl/gdl_pstruct.h>

gdl_pstruct_workspace *
gdl_pstruct_workspace_galloc (const gdl_pstruct_workspace_type * T, gdl_gview_wrapper * gw, gdl_rng * rng, size_t k)
{
	gdl_pstruct_workspace * f;
	
	f = GDL_CALLOC (gdl_pstruct_workspace, 1);
	
	f->type = T;
	f->k    = k;
	f->rng  = rng;
	f->gwrap = gw;
	
	f->state = gdl_malloc (T->size);
	
	(T->alloc)(f->state, gw, rng, k);
	
	return f;
}

gdl_pstruct_workspace *
gdl_pstruct_workspace_halloc (const gdl_pstruct_workspace_type * T, gdl_hview * h, gdl_rng * rng, size_t k)
{
	gdl_pstruct_workspace * f;
	
	f = GDL_CALLOC (gdl_pstruct_workspace, 1);
	
	f->type  = T;
	f->k     = k;
	f->rng   = rng;
	f->hview = h;
	
	f->state = gdl_malloc (T->size);
	
	(T->alloc)(f->state, h, rng, k);
	
	return f;
}

void
gdl_pstruct_workspace_free (gdl_pstruct_workspace * f)
{
	if (f)
	{
		(f->type->free)(f->state);
		GDL_FREE (f->state);
		GDL_FREE (f->lidx);
		GDL_FREE (f->aidx);
		GDL_FREE (f);
	}
}

int
gdl_pstruct_workspace_init (gdl_pstruct_workspace * f)
{
	return (f->type->init)(f->state);
}

int gdl_pstruct_workspace_init_update (gdl_pstruct_workspace * f, const int * aidx, const int * lidx, const void * db, double (*get_q)(const void * db, size_t k, size_t i), double (*get_f)(const void * db, size_t k, size_t l, size_t a))
{
	f->aidx = GDL_MALLOC (int, gdl_pstruct_workspace_accession_size (f));
	f->lidx = GDL_MALLOC (int, gdl_pstruct_workspace_locus_size (f));
	
	memcpy (f->aidx, aidx, sizeof(int)*gdl_pstruct_workspace_accession_size (f));
	memcpy (f->lidx, lidx, sizeof(int)*gdl_pstruct_workspace_locus_size (f));
		
	return (f->type->init_update)(f->state, f->aidx, f->lidx, db, get_q, get_f);
}

int
gdl_pstruct_workspace_iterate (gdl_pstruct_workspace * f)
{
	return (f->type->iterate)(f->state);
}

int
gdl_pstruct_workspace_iterate_update (gdl_pstruct_workspace * f)
{
	return (f->type->iterate_update)(f->state, f->aidx, f->lidx);
}

int
gdl_pstruct_workspace_imputation (gdl_pstruct_workspace * f)
{
	if (f->type->imputation)
	{
		return (f->type->imputation)(f->state);
	}
	else
	{
		return GDL_SUCCESS;	
	}
}

double
gdl_pstruct_workspace_loglikelihood (const gdl_pstruct_workspace * f)
{
	return (f->type->loglik)(f->state);
}

double
gdl_pstruct_workspace_residual_sq (const gdl_pstruct_workspace * f)
{
	return (f->type->res_sq)(f->state);
}

double
gdl_pstruct_workspace_residual_abs (const gdl_pstruct_workspace * f)
{
	return (f->type->res_abs)(f->state);
}

size_t
gdl_pstruct_workspace_ploidy (const gdl_pstruct_workspace * f)
{
	if (f->gwrap)
	{
		return gdl_gview_wrapper_ploidy (f->gwrap);
	}
	else if (f->hview)
	{
		return gdl_hview_ploidy (f->hview);
	}
	return 0;
}

size_t
gdl_pstruct_workspace_population_size (const gdl_pstruct_workspace * f)
{
	return f->k;
}

size_t
gdl_pstruct_workspace_accession_size (const gdl_pstruct_workspace * f)
{
	if (f->gwrap)
	{
		return gdl_gview_wrapper_accession_size (f->gwrap);
	}
	else if (f->hview)
	{
		return gdl_hview_accession_size (f->hview);
	}
	return 0;
}

size_t
gdl_pstruct_workspace_locus_size (const gdl_pstruct_workspace * f)
{
	if (f->gwrap)
	{
		return gdl_gview_wrapper_locus_size (f->gwrap);
	}
	else if (f->hview)
	{
		return gdl_hview_locus_size (f->hview);
	}
	return 0;
}

double
gdl_pstruct_workspace_get_population_q (const gdl_pstruct_workspace * f, size_t pop)
{
	return (f->type->get_pop_q)(f->state, pop);
}

double
gdl_pstruct_workspace_get_accession_q (const gdl_pstruct_workspace * f, size_t pop, size_t indiv)
{
	return (f->type->get_acc_q)(f->state, pop, indiv);
}

double
gdl_pstruct_workspace_get_locus_f (const gdl_pstruct_workspace * f, size_t pop, size_t loc, size_t allele)
{
	return (f->type->get_loc_f)(f->state, pop, loc, allele);
}

size_t
gdl_pstruct_workspace_get_locus_f_max (const gdl_pstruct_workspace * f, size_t pop, size_t loc)
{
	return (f->type->get_loc_f_max)(f->state, pop, loc);
}

size_t
gdl_pstruct_workspace_get_locus_f_size (const gdl_pstruct_workspace * f, size_t loc)
{
	return (f->type->get_loc_f_size)(f->state, loc);
}

size_t
gdl_pstruct_workspace_get_accession_q_max (const gdl_pstruct_workspace * f, size_t indiv)
{
	size_t k, kmax=0;
	double q,max=0;
	
	for(k=0;k<f->k;k++)
	{
		q=gdl_pstruct_workspace_get_accession_q (f,k,indiv);
		if (q>max)
		{
			max=q;
			kmax=k;
		}
	}
	
	return kmax;
}
