/*  
 *  rho/work.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_nrec.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_rho.h>

gdl_rho *
gdl_rho_alloc (const gdl_chromosome * chrom, gdl_gview_wrapper * data, const gdl_rng * rng)
{
	gdl_rho * m;
	
	m = GDL_CALLOC (gdl_rho, 1);
	
	m->rng   = rng;
	m->data  = gdl_rho_data_alloc (chrom, data);
	m->param = gdl_rho_params_alloc (m->data->L);
	m->util  = gdl_rho_util_alloc (m->data->N, m->data->L);
	
	return m;	
}

void
gdl_rho_free (gdl_rho * m)
{
	if (m)
	{
		gdl_rho_data_free (m->data);
		gdl_rho_params_free (m->param);
		gdl_rho_util_free (m->util);
		GDL_FREE (m);	
	}		
}

double 
gdl_rho_loglikelihood (gdl_rho * m)
{
	size_t i;
	double log = 0;
	
	for (i = 1; i < m->data->N-1; i++)
	{
		gdl_rho_hmm_forward (m, i, i+1);
		log += m->util->log;
	}
	
	return log;
}

double
gdl_rho_maximize_rho (gdl_rho * m)
{
	double ax,bx,cx,fa,fb,fc,xmin,loglik;
	
	double func (double x)
	{
		double y, t;
		t = m->param->rho;
		m->param->rho = exp (x);
		y = -gdl_rho_loglikelihood(m);
		m->param->rho = t;
		return y;
	};
	
	//printf ("<<RHO %g LOG %g\n", m->param->rho, gdl_rho_loglikelihood(m));
	ax = log (m->param->rho);
	bx = ax+gdl_ran_gaussian (m->rng, 1);
	mnbrak (&ax, &bx, &cx, &fa, &fb, &fc, &func);
	loglik = -dgolden (ax, bx, cx, &func, 1.e-8, &xmin);
	m->param->rho = exp (xmin);
	//printf (">>RHO %g LOG %g\n", m->param->rho, loglik);
	
	return loglik;
}

double
gdl_rho_maximize_theta (gdl_rho * m)
{
	double ax,bx,cx,fa,fb,fc,xmin,loglik;
	
	double func (double x)
	{
		double y, t;
		t = m->param->theta;
		m->param->theta = exp(x)/(1+exp(x));
		y = -gdl_rho_loglikelihood(m);
		m->param->theta = t;
		return y;
	};
	//printf ("<<THETA %g LOG %g\n", m->param->theta, gdl_rho_loglikelihood(m));
	ax = m->param->theta;
	ax = log (ax)-log (1-ax);
	bx = ax+gdl_ran_gaussian (m->rng, 1);
	mnbrak (&ax, &bx, &cx, &fa, &fb, &fc, &func);
	loglik = -dgolden (ax, bx, cx, &func, 1.e-8, &xmin);
	m->param->theta = exp(xmin)/(1+exp(xmin));
	//printf (">>THETA %g LOG %g\n", m->param->theta, loglik);
	
	return loglik;
}

double
gdl_rho_maximize_lambda (gdl_rho * m)
{
	size_t l;
	double ax,bx,cx,fa,fb,fc,xmin,loglik;
	
	double func (double x)
	{
		double y, t;
		t = m->param->lambda[l];
		m->param->lambda[l] = exp (x);
		y = -gdl_rho_loglikelihood(m);
		m->param->lambda[l] = t;
		return y;
	};
	
	for (l = 0; l < m->data->L-1; l++)
	{
		//printf ("<<LAMBDA[%d] %g LOG %g\n", l, m->param->lambda[l], gdl_rho_loglikelihood(m));
		ax = log (m->param->lambda[l]);
		bx = ax+gdl_ran_gaussian (m->rng, 1);
		mnbrak (&ax, &bx, &cx, &fa, &fb, &fc, &func);
		loglik = -dgolden (ax, bx, cx, &func, 1.e-5, &xmin);
		m->param->lambda[l] = exp (xmin);
		//printf (">>LAMBDA[%d] %g LOG %g\n", l, m->param->lambda[l], gdl_rho_loglikelihood(m));
	}
	
	return loglik;
}

int
gdl_rho_perform (gdl_rho * m, gdl_boolean rho_cst, double epsilon, size_t max_iter, size_t rep)
{
	size_t i, r;
	double new, old, eps;
	gdl_rho_params ** theta;
	
	theta = GDL_CALLOC (gdl_rho_params *, rep);
	
	for (r = 0; r < rep; r++)
	{
		gdl_rho_data_shuffle (m->data, m->rng);
		
		gdl_rho_params_init (m->param, m->data->N);
		
		if (rho_cst)
		{
			old = gdl_rho_maximize_rho (m);
			if (m->logger)
			{
				printf ("%d\t%.1f\t%e\n", r+1, old, m->param->rho);
			}
//			for(i = 0; i < max_iter; i++)
//			{
//				gdl_rho_maximize_theta (m);
//				new = gdl_rho_maximize_rho (m);
//				eps = (old-new)/old;
//				if (m->logger)
//				{
//					printf ("%d\t%d\t%.1f\t%e\n", r+1, i+1, new, eps);
//				}
//				old=new;
//				if (eps < epsilon) break;
//			}
		}
		else
		{
			m->param->rho = 1;
			old = gdl_rho_maximize_lambda (m);
			for(i = 0; i < max_iter; i++)
			{
				//gdl_rho_maximize_theta (m);
				new = gdl_rho_maximize_lambda (m);
				eps = (old-new)/old;
				if (m->logger)
				{
					printf ("%d\t%d\t%.1f\t%e\n", r+1, i+1, new, eps);
				}
				old=new;
				if (eps < epsilon) break;				
			}
		}
		theta[r] = gdl_rho_params_clone (m->param);
		theta[r]->proba = old;
	}
	
	gdl_rho_summarize (m, theta, rep);
	
	for (r = 0; r < rep; r++)
	{
		gdl_rho_params_free (theta[r]);
	}
	GDL_FREE (theta);
}

void
gdl_rho_summarize (gdl_rho * m, gdl_rho_params ** theta, size_t n)
{
	size_t i, j;
	double w, tot=0;
	gdl_rho_params * p = m->param;
	
	for (i = 0; i < n; i++)
	{
		tot+=theta[i]->proba;
	}
	for (i = 0; i < n; i++)
	{
		w = theta[i]->proba/tot;
		if (i) p->rho += w*theta[i]->rho;
		else p->rho = w*theta[i]->rho;
		if (i) p->theta += w*theta[i]->theta;
		else p->theta = w*theta[i]->theta;
		for (j = 0; j < m->data->L-1; j++)
		{
			if (i) p->lambda[j] += w*theta[i]->lambda[j];	
			else p->lambda[j] = w*theta[i]->lambda[j];
		}
	}
}

FILE *
gdl_rho_set_logger (gdl_rho * m, FILE * logger)
{
	FILE * old = m->logger;
	m->logger = logger;
	return old;
}
