/* rng/gdl_rng.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2004 James Theiler, Brian Gough
 * 
 * This file has been modified by Jean-Baptiste Veyrieras for inclusion in the
 * GDL library.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_RNG_H__
#define __GDL_RNG_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>

__BEGIN_DECLS

typedef struct
  {
    const char *name;
    unsigned long int max;
    unsigned long int min;
    size_t size;
    void (*set) (void *state, unsigned long int seed);
    unsigned long int (*get) (void *state);
    double (*get_double) (void *state);
  }
gdl_rng_type;

typedef struct
  {
    const gdl_rng_type * type;
    void *state;
  }
gdl_rng;


/* These structs also need to appear in default.c so you can select
   them via the environment variable GDL_RNG_TYPE */

GDL_VAR const gdl_rng_type *gdl_rng_borosh13;
GDL_VAR const gdl_rng_type *gdl_rng_coveyou;
GDL_VAR const gdl_rng_type *gdl_rng_cmrg;
GDL_VAR const gdl_rng_type *gdl_rng_fishman18;
GDL_VAR const gdl_rng_type *gdl_rng_fishman20;
GDL_VAR const gdl_rng_type *gdl_rng_fishman2x;
GDL_VAR const gdl_rng_type *gdl_rng_gfsr4;
GDL_VAR const gdl_rng_type *gdl_rng_knuthran;
GDL_VAR const gdl_rng_type *gdl_rng_knuthran2;
GDL_VAR const gdl_rng_type *gdl_rng_lecuyer21;
GDL_VAR const gdl_rng_type *gdl_rng_minstd;
GDL_VAR const gdl_rng_type *gdl_rng_mrg;
GDL_VAR const gdl_rng_type *gdl_rng_mt19937;
GDL_VAR const gdl_rng_type *gdl_rng_mt19937_1999;
GDL_VAR const gdl_rng_type *gdl_rng_mt19937_1998;
GDL_VAR const gdl_rng_type *gdl_rng_r250;
GDL_VAR const gdl_rng_type *gdl_rng_ran0;
GDL_VAR const gdl_rng_type *gdl_rng_ran1;
GDL_VAR const gdl_rng_type *gdl_rng_ran2;
GDL_VAR const gdl_rng_type *gdl_rng_ran3;
GDL_VAR const gdl_rng_type *gdl_rng_rand;
GDL_VAR const gdl_rng_type *gdl_rng_rand48;
GDL_VAR const gdl_rng_type *gdl_rng_random128_bsd;
GDL_VAR const gdl_rng_type *gdl_rng_random128_glibc2;
GDL_VAR const gdl_rng_type *gdl_rng_random128_libc5;
GDL_VAR const gdl_rng_type *gdl_rng_random256_bsd;
GDL_VAR const gdl_rng_type *gdl_rng_random256_glibc2;
GDL_VAR const gdl_rng_type *gdl_rng_random256_libc5;
GDL_VAR const gdl_rng_type *gdl_rng_random32_bsd;
GDL_VAR const gdl_rng_type *gdl_rng_random32_glibc2;
GDL_VAR const gdl_rng_type *gdl_rng_random32_libc5;
GDL_VAR const gdl_rng_type *gdl_rng_random64_bsd;
GDL_VAR const gdl_rng_type *gdl_rng_random64_glibc2;
GDL_VAR const gdl_rng_type *gdl_rng_random64_libc5;
GDL_VAR const gdl_rng_type *gdl_rng_random8_bsd;
GDL_VAR const gdl_rng_type *gdl_rng_random8_glibc2;
GDL_VAR const gdl_rng_type *gdl_rng_random8_libc5;
GDL_VAR const gdl_rng_type *gdl_rng_random_bsd;
GDL_VAR const gdl_rng_type *gdl_rng_random_glibc2;
GDL_VAR const gdl_rng_type *gdl_rng_random_libc5;
GDL_VAR const gdl_rng_type *gdl_rng_randu;
GDL_VAR const gdl_rng_type *gdl_rng_ranf;
GDL_VAR const gdl_rng_type *gdl_rng_ranlux;
GDL_VAR const gdl_rng_type *gdl_rng_ranlux389;
GDL_VAR const gdl_rng_type *gdl_rng_ranlxd1;
GDL_VAR const gdl_rng_type *gdl_rng_ranlxd2;
GDL_VAR const gdl_rng_type *gdl_rng_ranlxs0;
GDL_VAR const gdl_rng_type *gdl_rng_ranlxs1;
GDL_VAR const gdl_rng_type *gdl_rng_ranlxs2;
GDL_VAR const gdl_rng_type *gdl_rng_ranmar;
GDL_VAR const gdl_rng_type *gdl_rng_slatec;
GDL_VAR const gdl_rng_type *gdl_rng_taus;
GDL_VAR const gdl_rng_type *gdl_rng_taus2;
GDL_VAR const gdl_rng_type *gdl_rng_taus113;
GDL_VAR const gdl_rng_type *gdl_rng_transputer;
GDL_VAR const gdl_rng_type *gdl_rng_tt800;
GDL_VAR const gdl_rng_type *gdl_rng_uni;
GDL_VAR const gdl_rng_type *gdl_rng_uni32;
GDL_VAR const gdl_rng_type *gdl_rng_vax;
GDL_VAR const gdl_rng_type *gdl_rng_waterman14;
GDL_VAR const gdl_rng_type *gdl_rng_zuf;

const gdl_rng_type ** gdl_rng_types_setup(void);

GDL_VAR const gdl_rng_type *gdl_rng_default;
GDL_VAR unsigned long int gdl_rng_default_seed;

gdl_rng *gdl_rng_alloc (const gdl_rng_type * T);
int gdl_rng_memcpy (gdl_rng * dest, const gdl_rng * src);
gdl_rng *gdl_rng_clone (const gdl_rng * r);

void gdl_rng_free (gdl_rng * r);

void gdl_rng_set (const gdl_rng * r, unsigned long int seed);
unsigned long int gdl_rng_max (const gdl_rng * r);
unsigned long int gdl_rng_min (const gdl_rng * r);
const char *gdl_rng_name (const gdl_rng * r);

gdl_rng * gdl_rng_fread (FILE * stream);
int gdl_rng_fwrite (FILE * stream, const gdl_rng * r);

size_t gdl_rng_size (const gdl_rng * r);
void * gdl_rng_state (const gdl_rng * r);

void gdl_rng_print_state (const gdl_rng * r);

const gdl_rng_type * gdl_rng_env_setup (void);

unsigned long int gdl_rng_get (const gdl_rng * r);
double gdl_rng_uniform (const gdl_rng * r);
double gdl_rng_uniform_pos (const gdl_rng * r);
unsigned long int gdl_rng_uniform_int (const gdl_rng * r, unsigned long int n);


#ifdef HAVE_INLINE
extern inline unsigned long int gdl_rng_get (const gdl_rng * r);

extern inline unsigned long int
gdl_rng_get (const gdl_rng * r)
{
  return (r->type->get) (r->state);
}

extern inline double gdl_rng_uniform (const gdl_rng * r);

extern inline double
gdl_rng_uniform (const gdl_rng * r)
{
  return (r->type->get_double) (r->state);
}

extern inline double gdl_rng_uniform_pos (const gdl_rng * r);

extern inline double
gdl_rng_uniform_pos (const gdl_rng * r)
{
  double x ;
  do
    {
      x = (r->type->get_double) (r->state) ;
    }
  while (x == 0) ;

  return x ;
}

extern inline unsigned long int gdl_rng_uniform_int (const gdl_rng * r, unsigned long int n);

extern inline unsigned long int
gdl_rng_uniform_int (const gdl_rng * r, unsigned long int n)
{
  unsigned long int offset = r->type->min;
  unsigned long int range = r->type->max - offset;
  unsigned long int scale = range / n;
  unsigned long int k;

  if (n > range) 
    {
      GDL_ERROR_VAL ("n exceeds maximum value of generator",
                        GDL_EINVAL, 0) ;
    }

  do
    {
      k = (((r->type->get) (r->state)) - offset) / scale;
    }
  while (k >= n);

  return k;
}
#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_RNG_H__ */
