/* rng/rng.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 James Theiler, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>

gdl_rng *
gdl_rng_alloc (const gdl_rng_type * T)
{

  gdl_rng *r = (gdl_rng *) gdl_malloc (sizeof (gdl_rng));

  if (r == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for rng struct",
                        GDL_ENOMEM, 0);
    };

  r->state = gdl_malloc (T->size);

  if (r->state == 0)
    {
      free (r);         /* exception in constructor, avoid memory leak */

      GDL_ERROR_VAL ("failed to allocate space for rng state",
                        GDL_ENOMEM, 0);
    };

  r->type = T;

  gdl_rng_set (r, gdl_rng_default_seed);        /* seed the generator */

  return r;
}

int
gdl_rng_memcpy (gdl_rng * dest, const gdl_rng * src)
{
  if (dest->type != src->type)
    {
      GDL_ERROR ("generators must be of the same type", GDL_EINVAL);
    }

  memcpy (dest->state, src->state, src->type->size);

  return GDL_SUCCESS;
}

gdl_rng *
gdl_rng_clone (const gdl_rng * q)
{
  gdl_rng *r = (gdl_rng *) gdl_malloc (sizeof (gdl_rng));

  if (r == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for rng struct",
                        GDL_ENOMEM, 0);
    };

  r->state = gdl_malloc (q->type->size);

  if (r->state == 0)
    {
      free (r);         /* exception in constructor, avoid memory leak */

      GDL_ERROR_VAL ("failed to allocate space for rng state",
                        GDL_ENOMEM, 0);
    };

  r->type = q->type;

  memcpy (r->state, q->state, q->type->size);

  return r;
}

void
gdl_rng_set (const gdl_rng * r, unsigned long int seed)
{
  (r->type->set) (r->state, seed);
}

#ifndef HIDE_INLINE_STATIC
unsigned long int
gdl_rng_get (const gdl_rng * r)
{
  return (r->type->get) (r->state);
}

double
gdl_rng_uniform (const gdl_rng * r)
{
  return (r->type->get_double) (r->state);
}

double
gdl_rng_uniform_pos (const gdl_rng * r)
{
  double x ;
  do
    {
      x = (r->type->get_double) (r->state) ;
    }
  while (x == 0) ;

  return x ;
}

unsigned long int
gdl_rng_uniform_int (const gdl_rng * r, unsigned long int n)
{
  unsigned long int offset = r->type->min;
  unsigned long int range = r->type->max - offset;
  unsigned long int scale = range / n;
  unsigned long int k;

  if (n > range) 
    {
      GDL_ERROR_VAL ("n exceeds maximum value of generator",
                        GDL_EINVAL, 0) ;
    }

  do
    {
      k = (((r->type->get) (r->state)) - offset) / scale;
    }
  while (k >= n);

  return k;
}
#endif

unsigned long int
gdl_rng_max (const gdl_rng * r)
{
  return r->type->max;
}

unsigned long int
gdl_rng_min (const gdl_rng * r)
{
  return r->type->min;
}

const char *
gdl_rng_name (const gdl_rng * r)
{
  return r->type->name;
}

size_t
gdl_rng_size (const gdl_rng * r)
{
  return r->type->size;
}

void *
gdl_rng_state (const gdl_rng * r)
{
  return r->state;
}

void
gdl_rng_print_state (const gdl_rng * r)
{
  size_t i;
  unsigned char *p = (unsigned char *) (r->state);
  const size_t n = r->type->size;

  for (i = 0; i < n; i++)
    {
      /* FIXME: we're assuming that a char is 8 bits */
      printf ("%.2x", *(p + i));
    }

}

void
gdl_rng_free (gdl_rng * r)
{
  free (r->state);
  free (r);
}
