/*
 *  run/runtime.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:41 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_rng.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_runtime.h>

static const gdl_runtime_type *
_gdl_runtime_fread_type (FILE * stream)
{
	gdl_string * name;
	
	name = gdl_string_fread (stream);
	
	if (name)
	{
		if (!strcmp (name, gdl_runtime_default->name))
		{
			gdl_string_free (name);
			return gdl_runtime_default;
		}	
	}
	return NULL;
}

static gdl_rng *
_gdl_runtime_fread_rng (FILE * stream)
{
	int status;
	unsigned long int seed;
	gdl_rng * rng;
		
	status = fread (&seed, sizeof (unsigned long int), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	rng = gdl_rng_fread (stream);
	GDL_FREAD_STATUS (rng!=0, 1);
	
	gdl_rng_set (rng, seed);
	
	return rng;
}

static int
_gdl_runtime_fwrite_type (FILE * stream, const gdl_runtime_type * T)
{
	return gdl_string_fwrite (stream, T->name);
}

static int
_gdl_runtime_fwrite_rng (FILE * stream, const gdl_rng * rng)
{
	int status;
	unsigned long int seed;
	
	seed = gdl_rng_get (rng);
	
	status = fwrite (&seed, sizeof (unsigned long int), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = gdl_rng_fwrite (stream, rng);
	GDL_FREAD_STATUS (status, GDL_SUCCESS);
	
	return GDL_SUCCESS;
}

gdl_runtime *
gdl_runtime_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		const gdl_runtime_type * T;
		gdl_runtime * run;
		
		T = _gdl_runtime_fread_type (stream);
		GDL_FREAD_STATUS (T!=0, 1);
		
		run = gdl_runtime_alloc (T);
		
		run->rng = _gdl_runtime_fread_rng (stream);
		GDL_FREAD_STATUS (run->rng!=0, 1);
		
		return run;
	}
	return NULL;
}

int
gdl_runtime_fwrite (FILE * stream, const gdl_runtime * run)
{
	if (stream && run)
	{
		int status;
		
		status = _gdl_runtime_fwrite_type (stream, run->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = _gdl_runtime_fwrite_rng (stream, run->rng);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS; 
	}
	else
	{
		return GDL_EINVAL;	
	}
}
