/*
 *  snp/annot.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_snp.h>
#include <gdl/gdl_snp_chromosome.h>
#include <gdl/gdl_snp_annotation.h>

struct _gdl_snp_annot_reader
{
	gdl_hashtable * snp_tab;
	gdl_hashtable * gene_tab;
	gdl_hashtable * func_tab;
	gdl_hashtable * geno_tab;
	gdl_hashtable * tran_tab;
	gdl_hashtable * prot_tab;
};

static void
gdl_snp_annot_reader_init (gdl_snp_annot_reader * r, const gdl_snp_chromosome * chrom)
{
	size_t i;
	
	gdl_hashtable_free (r->snp_tab);
	r->snp_tab = gdl_hashtable_alloc (gdl_hash_default, chrom->chrom->size);
	for (i = 0; i < chrom->chrom->size; i++)
	{
		gdl_hashtable_add (r->snp_tab, chrom->chrom->snps[i]->rs, chrom->chrom->snps[i], 0);	
	}
}

static void
gdl_snp_annot_reader_clean (gdl_snp_annot_reader * r)
{
	gdl_hashtable_free (r->snp_tab);
	r->snp_tab=0;
}

gdl_snp_annot_reader *
gdl_snp_annot_reader_alloc (void)
{
	gdl_snp_annot_reader * r;
	
	r = GDL_CALLOC (gdl_snp_annot_reader, 1);
	r->gene_tab = gdl_hashtable_alloc (gdl_hash_default, 0);
	r->geno_tab = gdl_hashtable_alloc (gdl_hash_default, 0);
	r->func_tab = gdl_hashtable_alloc (gdl_hash_default, 0);
	r->tran_tab = gdl_hashtable_alloc (gdl_hash_default, 0);
	r->prot_tab = gdl_hashtable_alloc (gdl_hash_default, 0);
	
	return r;
}

void
gdl_snp_annot_reader_free (gdl_snp_annot_reader * r)
{
	if (r)
	{
		gdl_hashtable_free (r->snp_tab);
		gdl_hashtable_free (r->gene_tab);
		gdl_hashtable_free (r->func_tab);
		gdl_hashtable_free (r->geno_tab);
		gdl_hashtable_free (r->tran_tab);
		gdl_hashtable_free (r->prot_tab);
		GDL_FREE (r);
	}	
}

gdl_snp_annot_dico *
gdl_snp_annot_dico_alloc (const gdl_snp_annot_reader * reader)
{
	size_t i;
	gdl_hashtable_itr * itr;
	gdl_snp_annot_dico * d;
	
	d = GDL_CALLOC (gdl_snp_annot_dico, 1);
	
	d->ngene = gdl_hashtable_size (reader->gene_tab);
	d->nfunc = gdl_hashtable_size (reader->func_tab);
	d->ngeno = gdl_hashtable_size (reader->geno_tab); 
	d->ntran = gdl_hashtable_size (reader->tran_tab);
	d->nprot = gdl_hashtable_size (reader->prot_tab);
	
	if (d->ngene)
	{
		d->gene = GDL_MALLOC (gdl_snp_annot_gene_entry *, d->ngene);
		itr = gdl_hashtable_iterator (reader->gene_tab);
		do
		{
			gdl_snp_annot_gene_entry * e = gdl_hashtable_iterator_value (itr);
			d->gene[e->idx]=e;
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	if (d->nfunc)
	{
		d->func = GDL_MALLOC (gdl_snp_annot_entry *, d->nfunc);
		itr = gdl_hashtable_iterator (reader->func_tab);
		do
		{
			gdl_snp_annot_entry * e = (gdl_snp_annot_entry *)gdl_hashtable_iterator_value (itr);
			d->func[e->idx]=e;
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	if (d->ngeno)
	{
		d->geno = GDL_MALLOC (gdl_snp_annot_entry *, d->ngeno);
		itr = gdl_hashtable_iterator (reader->geno_tab);
		do
		{
			gdl_snp_annot_entry * e = (gdl_snp_annot_entry *)gdl_hashtable_iterator_value (itr); 
			d->geno[e->idx]=e;
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	if (d->ntran)
	{
		d->tran = GDL_MALLOC (gdl_snp_annot_entry *, d->ntran);
		itr = gdl_hashtable_iterator (reader->tran_tab);
		do
		{
			gdl_snp_annot_entry * e = (gdl_snp_annot_entry *)gdl_hashtable_iterator_value (itr);
			d->tran[e->idx]=e;
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	if (d->nprot)
	{
		d->prot = GDL_MALLOC (gdl_snp_annot_entry *, d->nprot);
		itr = gdl_hashtable_iterator (reader->prot_tab);
		do
		{
			gdl_snp_annot_entry * e = (gdl_snp_annot_entry *)gdl_hashtable_iterator_value (itr);
			d->prot[e->idx]=e;
		}
		while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	
	d->unknown = gdl_snp_annot_entry_alloc ("unknown");
	
	return d;
}

static int
gdl_snp_annot_reader_push (gdl_snp_annot_reader * reader, gdl_snp_annot * annot, char type, const gdl_string * tok)
{
	if (!strcmp(tok, "-") || !strcmp(tok, ""))
		return;
	
	size_t m, k, g=0;
	gdl_string ** toks;
	gdl_hashtable * table; 
	gdl_snp_annot_level ** level;
	gdl_snp_annot_entry * entry;
	
	switch(type)
	{
		case 'g':
			table = reader->geno_tab;
			level = &annot->geno;
			break;
		case 't':
			table = reader->tran_tab;
			level = &annot->tran;
			break;
		case 'p':
			table = reader->prot_tab;
			level = &annot->prot;
			break;
		case 'f':
			g = 1;
			break;
	}
	
	toks = gdl_string_split (tok, ":", &m);
	
	if (!g)
	{
		*level = gdl_snp_annot_level_alloc (m);
		for (k = 0; k < m; k++)
		{
			entry = gdl_hashtable_lookup (table, toks[k]);
			if (!entry)
			{
				entry = gdl_snp_annot_entry_alloc (toks[k]);
				toks[k]=0;
				entry->idx = gdl_hashtable_size (table);
				gdl_hashtable_add (table, entry->name, entry, 0);
			}
			(*level)->ids[k]=entry->idx;
			//printf (" (%s,%d)", entry->name, (*level)->ids[k]);
		}
	}
	else
	{
		gdl_snp_annot_gene_entry * gene;
		gdl_snp_annot_entry * func;
		
		gene = gdl_hashtable_lookup (reader->gene_tab, toks[1]);
		if (!gene)
		{
			gene = gdl_snp_annot_gene_entry_alloc (toks[1], atol(toks[0]));
			toks[1]=0;
			gene->idx = gdl_hashtable_size (reader->gene_tab);
			gdl_hashtable_add (reader->gene_tab, gene->name, gene, 0);
		}
		func = gdl_hashtable_lookup (reader->func_tab, toks[2]);
		if (!func)
		{
			func = gdl_snp_annot_entry_alloc (toks[2]);
			toks[2]=0;
			func->idx = gdl_hashtable_size (reader->func_tab);
			gdl_hashtable_add (reader->func_tab, func->name, func, 0);
		}
		annot->gene = gdl_snp_annot_gene_alloc (gene->idx, func->idx);
		//printf (" (%s,%d,%s,%d)", gene->name, gene->id, func->name, func->idx);
	}
	GDL_MATRIX_FREE (toks, m);
	
	return GDL_SUCCESS;
}
/**
 * Create the annotation dictionary from the input stream and matching
 * the SNPs of the chromosome
 */
int
gdl_snp_annot_reader_parse (gdl_snp_annot_reader * reader, gdl_snp_chromosome * chrom, FILE * stream)
{
	if (stream && chrom)
	{
		size_t i,j,n;
		gdl_string * tok, * line = 0;
		gdl_snp * snp;
		gdl_snp_annot        * annot;
		gdl_snp_annot_dico * dico = 0; 
		
		gdl_snp_annot_reader_init (reader, chrom);
		
		while (gdl_getline (&line, &n, stream) != -1)
		{
			i=j=0;
			// First the SNP id
			tok = gdl_string_next_token (line, n, &i, &j);
			if ((snp=(gdl_snp *)gdl_hashtable_lookup (reader->snp_tab, tok))!=0)
			{
				//printf ("SNP %s", snp->rs);
				annot = snp->annot = gdl_snp_annot_alloc ();
				gdl_string_free (tok);
				// Then the genome level
				tok  = gdl_string_next_token (line, n, &i, &j);
				gdl_snp_annot_reader_push (reader, annot, 'g', tok);
				gdl_string_free (tok);
				// Then the gene level
				tok  = gdl_string_next_token (line, n, &i, &j);
				gdl_snp_annot_reader_push (reader, annot, 'f', tok);
				gdl_string_free (tok);
				// Then the transcript level
				tok = gdl_string_next_token (line, n, &i, &j);
				gdl_snp_annot_reader_push (reader, annot, 't', tok);
				gdl_string_free (tok);
				// Then the protein level
				tok = gdl_string_next_token (line, n, &i, &j);
				gdl_snp_annot_reader_push (reader, annot, 'p', tok);
				gdl_string_free (tok);
				//printf ("\n");
			}
			else
			{
				gdl_string_free (tok);
			}
			GDL_FREE (line);
			line=0;
		}
		
		gdl_snp_annot_reader_clean (reader);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

void
gdl_snp_annot_dico_free (gdl_snp_annot_dico * d)
{
	if (d)
	{
		size_t i;
		for (i = 0; i < d->ngene; i++) gdl_snp_annot_gene_entry_free (d->gene[i]);
		for (i = 0; i < d->nfunc; i++) gdl_snp_annot_entry_free (d->func[i]);
		for (i = 0; i < d->ngeno; i++) gdl_snp_annot_entry_free (d->geno[i]);
		for (i = 0; i < d->ntran; i++) gdl_snp_annot_entry_free (d->tran[i]);
		for (i = 0; i < d->nprot; i++) gdl_snp_annot_entry_free (d->prot[i]);
		GDL_FREE (d->gene);
		GDL_FREE (d->func);
		GDL_FREE (d->geno);
		GDL_FREE (d->tran);
		GDL_FREE (d->prot);
		GDL_FREE (d);
	}	
}

gdl_snp_annot_dico *
gdl_snp_annot_dico_clone (const gdl_snp_annot_dico * d)
{
	if (d)
	{
		size_t i;
		gdl_snp_annot_dico * c;
		
		c = GDL_CALLOC (gdl_snp_annot_dico, 1);
		
		c->ngene = d->ngene;
		c->gene  = GDL_MALLOC (gdl_snp_annot_gene_entry *, c->ngene); 
		for (i = 0; i < c->ngene; i++) c->gene[i] = gdl_snp_annot_gene_entry_clone (d->gene[i]);
		c->nfunc = d->nfunc;
		c->func  = GDL_MALLOC (gdl_snp_annot_entry *, c->nfunc); 
		for (i = 0; i < c->nfunc; i++) c->func[i] = gdl_snp_annot_entry_clone (d->func[i]);
		c->ngeno = d->ngeno;
		c->geno  = GDL_MALLOC (gdl_snp_annot_entry *, c->ngeno); 
		for (i = 0; i < c->ngeno; i++) c->geno[i] = gdl_snp_annot_entry_clone (d->geno[i]);
		c->ntran = d->ntran;
		c->tran  = GDL_MALLOC (gdl_snp_annot_entry *, c->ntran); 
		for (i = 0; i < c->ntran; i++) c->tran[i] = gdl_snp_annot_entry_clone (d->tran[i]);
		c->nprot = d->nprot;
		c->prot  = GDL_MALLOC (gdl_snp_annot_entry *, c->nprot); 
		for (i = 0; i < c->nprot; i++) c->prot[i] = gdl_snp_annot_entry_clone (d->prot[i]);
		c->unknown = gdl_snp_annot_entry_clone (d->unknown);
		c->count = d->count;
		
		return c;
	}
	return 0;	
}

void
gdl_snp_annot_dico_reset_count (gdl_snp_annot_dico * d)
{
	size_t i;
	for (i = 0; i < d->nfunc; i++) d->func[i]->count=0;
	for (i = 0; i < d->ngeno; i++) d->geno[i]->count=0;
	for (i = 0; i < d->ntran; i++) d->tran[i]->count=0;
	for (i = 0; i < d->nprot; i++) d->prot[i]->count=0; 
	d->unknown->count=0;
	d->count=0;
}

int
gdl_snp_annot_dico_fprintf_count (const gdl_snp_annot_dico * d, const gdl_string * chrom, FILE * stream)
{
	size_t l;
	
	for (l = 0; l < d->nfunc; l++)
	{
		if (chrom) fprintf (stream, "%s\t", chrom);
		fprintf (stream, "Functional:%s\t%d\t%1.10f\n", d->func[l]->name, d->func[l]->count, ((double)d->func[l]->count)/d->count);
	}	 
	for (l = 0; l < d->ngeno; l++)
	{
		if (chrom) fprintf (stream, "%s\t", chrom);
		fprintf (stream, "Genome:%s\t%d\t%1.10f\n", d->geno[l]->name, d->geno[l]->count, ((double)d->geno[l]->count)/d->count);
	}
	for (l = 0; l < d->ntran; l++)
	{
		if (chrom) fprintf (stream, "%s\t", chrom);
		fprintf (stream, "Transcript:%s\t%d\t%1.10f\n", d->tran[l]->name, d->tran[l]->count, ((double)d->tran[l]->count)/d->count);
	}
	for (l = 0; l < d->nprot; l++)
	{
		if (chrom) fprintf (stream, "%s\t", chrom);
		fprintf (stream, "Protein:%s\t%d\t%1.10f\n", d->prot[l]->name, d->prot[l]->count, ((double)d->prot[l]->count)/d->count);
	}
	if (chrom) fprintf (stream, "%s\t", chrom);
	fprintf (stream, "Unknown\t%d\t%1.10f\n",  d->unknown->count, ((double)d->unknown->count)/d->count);
}

void
gdl_snp_annot_dico_copy_count (gdl_snp_annot_dico * d1, const gdl_snp_annot_dico * d2)
{
	size_t l;
	for (l = 0; l < d1->nfunc; l++) d1->func[l]->count+=d2->func[l]->count;
	for (l = 0; l < d1->ngeno; l++) d1->geno[l]->count+=d2->geno[l]->count;
	for (l = 0; l < d1->ntran; l++) d1->tran[l]->count+=d2->tran[l]->count;
	for (l = 0; l < d1->nprot; l++) d1->prot[l]->count+=d2->prot[l]->count;
	d1->unknown->count+=d2->unknown->count;
	d1->count+=d2->count;
}

void
gdl_snp_annot_dico_add_count (gdl_snp_annot_dico * dico, const gdl_snp * snp)
{
	size_t l;
	gdl_snp_annot * annot = snp->annot;
	
	if (annot)
	{
		if (annot->gene)
		{
			dico->func[annot->gene->func_id]->count++;
			dico->count++;
		}
		if (annot->geno)
		{
			for (l = 0; l < annot->geno->size; l++)
			{
				dico->geno[annot->geno->ids[l]]->count++;
				dico->count++;
			}
		}
		if (annot->tran)
		{
			for (l = 0; l < annot->tran->size; l++)
			{
				dico->tran[annot->tran->ids[l]]->count++;
				dico->count++;
			}
		}
		if (annot->prot)
		{
			for (l = 0; l < annot->prot->size; l++)
			{
				dico->prot[annot->prot->ids[l]]->count++;
				dico->count++;
			}
		}
	}
	else
	{
		dico->unknown->count++;
		dico->count++;
	}
}

gdl_snp_annot_dico *
gdl_snp_annot_dico_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_snp_annot_dico * d;
		
		d = GDL_CALLOC (gdl_snp_annot_dico, 1);
		
		status = fread (&d->ngene, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&d->nfunc, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&d->ngeno, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&d->ntran, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&d->nprot, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (d->ngene)
		{
			d->gene = GDL_MALLOC (gdl_snp_annot_gene_entry *, d->ngene);
			for (i = 0; i < d->ngene; i++)
			{
				d->gene[i] = gdl_snp_annot_gene_entry_fread (stream);
				GDL_FREAD_STATUS (d->gene[i]!=0, 1); 	
			}
		}
		if (d->nfunc)
		{
			d->func = GDL_MALLOC (gdl_snp_annot_entry *, d->ngene);
			for (i = 0; i < d->nfunc; i++)
			{
				d->func[i] = gdl_snp_annot_entry_fread (stream);
				GDL_FREAD_STATUS (d->func[i]!=0, 1);
				d->count += d->func[i]->count;
			}
		}
		if (d->ngeno)
		{
			d->geno = GDL_MALLOC (gdl_snp_annot_entry *, d->ngene);
			for (i = 0; i < d->ngeno; i++)
			{
				d->geno[i] = gdl_snp_annot_entry_fread (stream);
				GDL_FREAD_STATUS (d->geno[i]!=0, 1);
				d->count += d->geno[i]->count;
			}
		}
		if (d->ntran)
		{
			d->tran = GDL_MALLOC (gdl_snp_annot_entry *, d->ngene);
			for (i = 0; i < d->ntran; i++)
			{
				d->tran[i] = gdl_snp_annot_entry_fread (stream);
				GDL_FREAD_STATUS (d->tran[i]!=0, 1);
				d->count += d->tran[i]->count;
			}
		}
		if (d->nprot)
		{
			d->prot = GDL_MALLOC (gdl_snp_annot_entry *, d->ngene);
			for (i = 0; i < d->nprot; i++)
			{
				d->prot[i] = gdl_snp_annot_entry_fread (stream);
				GDL_FREAD_STATUS (d->prot[i]!=0, 1);
				d->count += d->prot[i]->count;
			}
		}
		d->unknown = gdl_snp_annot_entry_fread (stream);
		GDL_FREAD_STATUS (d->unknown!=0, 1);
		d->count += d->unknown->count;
		
		return d;
	}
	return 0;	
}

int
gdl_snp_annot_dico_fwrite (FILE * stream, const gdl_snp_annot_dico * d)
{
	if (stream && d)
	{
		int status;
		size_t i;
		
		status = fwrite (&d->ngene, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&d->nfunc, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&d->ngeno, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&d->ntran, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&d->nprot, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (d->ngene)
		{
			for (i = 0; i < d->ngene; i++)
			{
				status = gdl_snp_annot_gene_entry_fwrite (stream, d->gene[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS); 	
			}
		}
		if (d->nfunc)
		{
			for (i = 0; i < d->nfunc; i++)
			{
				status = gdl_snp_annot_entry_fwrite (stream, d->func[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS); 
			}
		}
		if (d->ngeno)
		{
			for (i = 0; i < d->ngeno; i++)
			{
				status = gdl_snp_annot_entry_fwrite (stream, d->geno[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS); 
			}
		}
		if (d->ntran)
		{
			for (i = 0; i < d->ntran; i++)
			{
				status = gdl_snp_annot_entry_fwrite (stream, d->tran[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
		}
		if (d->nprot)
		{
			for (i = 0; i < d->nprot; i++)
			{
				status = gdl_snp_annot_entry_fwrite (stream, d->prot[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
		}
		status = gdl_snp_annot_entry_fwrite (stream, d->unknown);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

gdl_snp_annot *
gdl_snp_annot_alloc (void)
{
	gdl_snp_annot * a;
	
	a = GDL_CALLOC (gdl_snp_annot, 1);
	
	return a;
}

gdl_snp_annot *
gdl_snp_annot_clone (const gdl_snp_annot * a)
{
	if (a)
	{
		gdl_snp_annot * c;
		
		c = gdl_snp_annot_alloc ();
		
		c->gene = gdl_snp_annot_gene_clone (a->gene);
		c->geno = gdl_snp_annot_level_clone (a->geno);
		c->tran = gdl_snp_annot_level_clone (a->tran);
		c->prot = gdl_snp_annot_level_clone (a->prot);
		
		return c;
	}
	return 0;
}	

void
gdl_snp_annot_free (gdl_snp_annot * a)
{
	if (a)
	{
		gdl_snp_annot_gene_free (a->gene);
		gdl_snp_annot_level_free (a->geno);
		gdl_snp_annot_level_free (a->tran);
		gdl_snp_annot_level_free (a->prot);
		GDL_FREE (a);
	}	
}

gdl_snp_annot *
gdl_snp_annot_fread (FILE * stream)
{
	if (stream)
	{
		int status;
	   unsigned char has; 
		gdl_snp_annot * a;
		
		a = gdl_snp_annot_alloc ();
		
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has=='1')
		{
			a->gene = gdl_snp_annot_gene_fread (stream);
			GDL_FREAD_STATUS (a->gene!=0, 1);
		}
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has=='1')
		{
			a->geno = gdl_snp_annot_level_fread (stream);
			GDL_FREAD_STATUS (a->geno!=0, 1);
		}
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has=='1')
		{
			a->tran = gdl_snp_annot_level_fread (stream);
			GDL_FREAD_STATUS (a->tran!=0, 1);
		}
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has=='1')
		{
			a->prot = gdl_snp_annot_level_fread (stream);
			GDL_FREAD_STATUS (a->prot!=0, 1);
		}
		
		return a;
	}
	return 0;	
}

int
gdl_snp_annot_fwrite (FILE * stream, const gdl_snp_annot * a)
{
	if (stream && a)
	{
		int status;
	   unsigned char has; 
	   
	   has = (a->gene) ? '1' : '0';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has=='1')
		{
			status = gdl_snp_annot_gene_fwrite (stream, a->gene);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		has = (a->geno) ? '1' : '0';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has=='1')
		{
			status = gdl_snp_annot_level_fwrite (stream, a->geno);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		has = (a->tran) ? '1' : '0';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has=='1')
		{
			status = gdl_snp_annot_level_fwrite (stream, a->tran);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		has = (a->prot) ? '1' : '0';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has=='1')
		{
			status = gdl_snp_annot_level_fwrite (stream, a->prot);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

gdl_snp_annot_level *
gdl_snp_annot_level_alloc (const size_t size)
{
	gdl_snp_annot_level * l;
	
	l = GDL_MALLOC (gdl_snp_annot_level, 1);
	
	l->size = size;
	l->ids  = GDL_MALLOC (size_t, l->size);
	
	return l;	
}

gdl_snp_annot_level *
gdl_snp_annot_level_clone (const gdl_snp_annot_level * l)
{
	if (l)
	{
		gdl_snp_annot_level * c;
		
		c = gdl_snp_annot_level_alloc (l->size);
		memcpy (c->ids, l->ids, sizeof(size_t)*l->size);
		
		return c;
	}
	return 0;	
}

void
gdl_snp_annot_level_free (gdl_snp_annot_level * l)
{
	if (l)
	{
		GDL_FREE (l->ids);
		GDL_FREE (l);
	}
}

gdl_snp_annot_level *
gdl_snp_annot_level_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t size;
		gdl_snp_annot_level * l;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		l = gdl_snp_annot_level_alloc (size);
		
		status = fread (l->ids, sizeof(size_t), size, stream);
		GDL_FREAD_STATUS (status, size);
		
		return l;
	}
	return 0;
}

int
gdl_snp_annot_level_fwrite (FILE * stream, const gdl_snp_annot_level * l)
{
	if (stream && l)
	{
		int status;
		
		status = fwrite (&l->size, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (l->ids, sizeof (size_t), l->size, stream);
		GDL_FWRITE_STATUS (status, l->size);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL; 	
}

gdl_snp_annot_gene *
gdl_snp_annot_gene_alloc (size_t g, size_t f)
{
	gdl_snp_annot_gene * p;
	
	p = GDL_MALLOC (gdl_snp_annot_gene, 1);
	
	p->gene_id = g;
	p->func_id = f;
	
	return p;	
}

gdl_snp_annot_gene *
gdl_snp_annot_gene_clone (const gdl_snp_annot_gene * g)
{
	if (g)
	{
		gdl_snp_annot_gene * c;
		
		c = gdl_snp_annot_gene_alloc (g->gene_id, g->func_id);
		
		return c;
	}
	return 0;	
}

void
gdl_snp_annot_gene_free (gdl_snp_annot_gene * g)
{
	if (g)
	{
		GDL_FREE (g);	
	}	
}

gdl_snp_annot_gene *
gdl_snp_annot_gene_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t g, f;
		
		status = fread (&g, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&f, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);		
		
		return gdl_snp_annot_gene_alloc (g, f);
	}
	return 0;
}

int 
gdl_snp_annot_gene_fwrite (FILE * stream, const gdl_snp_annot_gene *g)
{
	if (stream && g)
	{
		int status;
		
		status = fwrite (&g->gene_id, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->func_id, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);		
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

gdl_snp_annot_entry *
gdl_snp_annot_entry_alloc (gdl_string * name)
{
	gdl_snp_annot_entry * e;
	
	e = GDL_CALLOC (gdl_snp_annot_entry, 1);
	
	e->name = name;
	
	return e;
}

void
gdl_snp_annot_entry_free (gdl_snp_annot_entry * e)
{
	if (e)
	{
		GDL_FREE (e->name);
		GDL_FREE (e);	
	}
}

gdl_snp_annot_entry *
gdl_snp_annot_entry_clone (const gdl_snp_annot_entry * e)
{
	if (e)
	{
		gdl_snp_annot_entry * c;
		
		c = gdl_snp_annot_entry_alloc (gdl_string_clone (e->name));
		c->idx = e->idx;
		c->count = e->count;
		c->prior = e->prior;
		
		return c;
	}
	return 0;	
}

gdl_snp_annot_entry *
gdl_snp_annot_entry_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_string * name;
		gdl_snp_annot_entry * e;
		
		name = gdl_string_fread (stream);
		GDL_FREAD_STATUS (name!=0, 1);
			
		e = gdl_snp_annot_entry_alloc (name);
		
		status = fread (&e->count, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return e;
	}
	return 0;
}

int
gdl_snp_annot_entry_fwrite (FILE * stream, gdl_snp_annot_entry * e)
{
	if (stream && e)
	{
		int status;
		
		status = gdl_string_fwrite (stream, e->name);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS); 
		status = fwrite (&e->count, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

gdl_snp_annot_gene_entry *
gdl_snp_annot_gene_entry_alloc (gdl_string * name, long id)
{
	gdl_snp_annot_gene_entry * e;
	
	e = GDL_MALLOC (gdl_snp_annot_gene_entry, 1);
	
	e->name = name;
	e->id   = id;
	
	return e;
}

gdl_snp_annot_gene_entry *
gdl_snp_annot_gene_entry_clone (const gdl_snp_annot_gene_entry * e)
{
	if (e)
	{
		gdl_snp_annot_gene_entry * c;
		
		c = gdl_snp_annot_gene_entry_alloc (gdl_string_clone (e->name), e->id);
		c->idx = e->idx;
		
		return c;
	}
	return 0;
}

void
gdl_snp_annot_gene_entry_free (gdl_snp_annot_gene_entry * g)
{
	if (g)
	{
		GDL_FREE (g->name);
		GDL_FREE (g);	
	}	
}

gdl_snp_annot_gene_entry *
gdl_snp_annot_gene_entry_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_string * name;
		long id;
		
		name = gdl_string_fread (stream);
		GDL_FREAD_STATUS (name!=0, 1);
		status = fread (&id, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return gdl_snp_annot_gene_entry_alloc (name, id);
	}
	return 0;
}

int
gdl_snp_annot_gene_entry_fwrite (FILE * stream, const gdl_snp_annot_gene_entry * g)
{
	if (stream && g)
	{
		int status;
		
		status = gdl_string_fwrite (stream, g->name);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = fwrite (&g->id, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

