/*
 *  snp/view.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_snp_data.h>

gdl_snp_data *
gdl_snp_data_alloc (const gdl_snp_data_type * T, const size_t N, const size_t L, const size_t P)
{
	gdl_snp_data * v;
	
	v = GDL_CALLOC (gdl_snp_data, 1);
	
	v->T = T;
	v->N = N;
	v->L = L;
	v->P = P;
	if (v->T == gdl_snp_data_genotype)
	{
		v->data = GDL_CALLOC (unsigned char, v->N*v->L);
	}
	else
	{
		v->data = GDL_CALLOC (unsigned char, v->N*v->L*v->P);	
	}
	
	return v;	
}

void
gdl_snp_data_free (gdl_snp_data * v)
{
	if (v)
	{
		GDL_FREE (v->data);
		GDL_FREE (v);
	}	
}

gdl_boolean
gdl_snp_data_is_missing (const gdl_snp_data * v, size_t i, size_t j)
{
	return (gdl_snp_data_get (v, i, j)==(unsigned char)(v->P+1)) ? gdl_true : gdl_false;
}

int
gdl_snp_data_get (const gdl_snp_data * v, size_t i, size_t j)
{
	if (v->T == gdl_snp_data_genotype)
	{
		switch (v->data[i*v->L+j])
		{
			case '0':
				return 0;
			case '1':
				return 1;
			case '2':
				return 2;
			default:
				return -1;	
		}
	}
	else
	{
		size_t k,a = 0;
		for(k = 0; k < v->P; k++)
		{
			switch (v->data[i*(v->L*v->P)+j*v->P+k])
			{
				case '1':
					a++;
					break;
				case '2':
					return -1;
			}
		}
		return a;
	}
}

int
gdl_snp_data_hget (const gdl_snp_data * v, size_t i, size_t j, size_t k)
{
	if (v->T == gdl_snp_data_genotype)
	{
		GDL_ERROR_VOID ("Try to get an haplotype in genotype mode", GDL_FAILURE);
	}
	else
	{
		switch (v->data[i*(v->L*v->P)+j*v->P+k])
		{
			case '0':
				return 0;
			case '1':
				return 1;
			default :
				return -1;
		}
	}
}

void
gdl_snp_data_set (const gdl_snp_data * v, size_t i, size_t j, unsigned char u)
{
	if (v->T == gdl_snp_data_genotype)
	{
		v->data[i*v->L+j]=u;
	}
	else
	{
		GDL_ERROR_VOID ("Try to set a genotype in haplotype mode", GDL_FAILURE);
	}
	
}

void
gdl_snp_data_hset (const gdl_snp_data * v, size_t i, size_t j, size_t k, unsigned char u)
{
	if (v->T == gdl_snp_data_genotype)
	{
		GDL_ERROR_VOID ("Try to set an haplotype in genotype mode", GDL_FAILURE);
	}
	else
	{
		v->data[i*(v->L*v->P)+j*v->P+k]=u;
	}
}

const static gdl_snp_data_type _gdl_snp_data_genotype =
{
	"g"
};

const static gdl_snp_data_type _gdl_snp_data_haplotype =
{
	"h"
};

const gdl_snp_data_type * gdl_snp_data_genotype  = &_gdl_snp_data_genotype;
const gdl_snp_data_type * gdl_snp_data_haplotype = &_gdl_snp_data_haplotype;
