/*
 *  snp/table.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_snp.h>

gdl_snp_table *
gdl_snp_table_fscanf (FILE * stream)
{
	size_t i,j,n;
	gdl_string * tok, * line=0;
	gdl_snp_table * table;
	gdl_hashtable * chrom_table;
	gdl_hashtable * feature_table;
	gdl_string * snp;
	
	table = GDL_MALLOC (gdl_snp_table, 1);
	
	table->table = gdl_hashtable_alloc (gdl_hash_default, 0);
	
	// ignore header
	gdl_getline (&line, &n, stream);
	gdl_string_free (line);line=0;
	while(gdl_getline (&line, &n, stream)!=-1)
	{
		i=j=0;
		// chrom name
		tok=gdl_string_next_token (line, n, &i, &j);
		chrom_table = gdl_hashtable_lookup (table->table, tok);
		if (!chrom_table)
		{
			chrom_table = gdl_hashtable_alloc (gdl_hash_default, 0);
			gdl_hashtable_add (table->table, tok, chrom_table, 0);
		}
		gdl_string_free (tok);
		// feature
		tok=gdl_string_next_token (line, n, &i, &j);
		feature_table = gdl_hashtable_lookup (chrom_table, tok);
		if (!feature_table)
		{
			feature_table = gdl_hashtable_alloc (gdl_string_interface, 0);
			gdl_hashtable_add (chrom_table, tok, feature_table, 0);
		}
		gdl_string_free (tok);
		// snp
		tok=gdl_string_next_token (line, n, &i, &j);
		gdl_hashtable_add (feature_table, tok, tok, 1);
		gdl_string_free (line);line=0;
	}
	
	return table;
}

gdl_boolean
gdl_snp_table_is_inside (const gdl_snp_table * table, const gdl_string * chrom, const gdl_string * feature, const gdl_string * snp)
{
	if (!table->table) return gdl_false;
	gdl_hashtable * chrom_table = gdl_hashtable_lookup (table->table, chrom);
	if (!chrom_table) return gdl_false;
	gdl_hashtable * feature_table = gdl_hashtable_lookup (chrom_table, feature);
	if (!feature_table) return gdl_false;
	gdl_string * ssnp = gdl_hashtable_lookup (feature_table, snp);
	if (!ssnp) return gdl_false;
	return gdl_true;
}

gdl_snp_table_entry **
gdl_snp_table_get_entries (const gdl_snp_table * table, size_t * n)
{
	size_t i;
	gdl_snp_table_entry ** e = 0;
	
	*n=0;
	if (gdl_hashtable_size (table->table))
	{
		gdl_hashtable_itr * citr = gdl_hashtable_iterator (table->table);
		do
		{
			gdl_hashtable * chrom_table = (gdl_hashtable *) gdl_hashtable_iterator_value (citr);
			gdl_hashtable_itr * fitr    = gdl_hashtable_iterator (chrom_table);
			do
			{
				gdl_hashtable * feature_table = (gdl_hashtable *) gdl_hashtable_iterator_value (fitr);
				*n += gdl_hashtable_size (feature_table);
			}
			while(gdl_hashtable_iterator_next (fitr));
			gdl_hashtable_iterator_free (fitr);
		}
		while(gdl_hashtable_iterator_next (citr));
		gdl_hashtable_iterator_free (citr);
		
		e = GDL_MALLOC (gdl_snp_table_entry *, *n);
		i = 0;
		
		citr = gdl_hashtable_iterator (table->table);
		do
		{
			gdl_hashtable * chrom_table = (gdl_hashtable *) gdl_hashtable_iterator_value (citr);
			gdl_hashtable_itr * fitr    = gdl_hashtable_iterator (chrom_table);
			do
			{
				gdl_hashtable * feature_table = (gdl_hashtable *) gdl_hashtable_iterator_value (fitr);
				gdl_hashtable_itr * sitr    = gdl_hashtable_iterator (feature_table);
				do
				{
					e[i] = GDL_MALLOC (gdl_snp_table_entry, 1);
					e[i]->chrom   = gdl_hashtable_iterator_key (citr);
					e[i]->feature = gdl_hashtable_iterator_key (fitr);
					e[i]->snp     = gdl_hashtable_iterator_key (sitr);
					i++;
				}
				while(gdl_hashtable_iterator_next (sitr));
				gdl_hashtable_iterator_free (sitr);
			}
			while(gdl_hashtable_iterator_next (fitr));
			gdl_hashtable_iterator_free (fitr);
		}
		while(gdl_hashtable_iterator_next (citr));
		gdl_hashtable_iterator_free (citr);
	}	
	
	return e;	
}

void
gdl_snp_table_free (gdl_snp_table * table)
{
	if (table)
	{
		if (gdl_hashtable_size (table->table))
		{
			gdl_hashtable_itr * citr = gdl_hashtable_iterator (table->table);
			do
			{
				gdl_hashtable * chrom_table = (gdl_hashtable *) gdl_hashtable_iterator_value (citr);
				gdl_hashtable_itr * fitr    = gdl_hashtable_iterator (chrom_table);
				do
				{
					gdl_hashtable * feature_table = (gdl_hashtable *) gdl_hashtable_iterator_value (fitr);
					gdl_hashtable_free (feature_table);
				}
				while(gdl_hashtable_iterator_next (fitr));
				gdl_hashtable_iterator_free (fitr);
				gdl_hashtable_free (chrom_table);
			}
			while(gdl_hashtable_iterator_next (citr));
			gdl_hashtable_iterator_free (citr);
		}
		gdl_hashtable_free (table->table);
		GDL_FREE (table); 
	}	
}

gdl_feature_table *
gdl_feature_table_fscanf (FILE * stream)
{
	size_t i,j,n;
	gdl_string * tok, * line=0;
	gdl_feature_table * table;
	gdl_hashtable * chrom_table;
	gdl_string * feature;
	
	table = GDL_MALLOC (gdl_feature_table, 1);
	
	table->table = gdl_hashtable_alloc (gdl_hash_default, 0);
	
	// ignore header
	gdl_getline (&line, &n, stream);
	gdl_string_free (line);line=0;
	while(gdl_getline (&line, &n, stream)!=-1)
	{
		i=j=0;
		// chrom name
		tok=gdl_string_next_token (line, n, &i, &j);
		chrom_table = gdl_hashtable_lookup (table->table, tok);
		if (!chrom_table)
		{
			chrom_table = gdl_hashtable_alloc (gdl_string_interface, 0);
			gdl_hashtable_add (table->table, tok, chrom_table, 0);
		}
		gdl_string_free (tok);
		// feature
		feature=gdl_string_next_token (line, n, &i, &j);
		gdl_hashtable_add (chrom_table, feature, feature, 1);
		gdl_string_free (line);line=0;
	}
	
	return table;	
}

gdl_boolean
gdl_feature_table_is_inside (const gdl_feature_table * table, const gdl_string * chrom, const gdl_string * feature)
{
	if (!table->table) return gdl_false;
	gdl_hashtable * chrom_table = gdl_hashtable_lookup (table->table, chrom);
	if (!chrom_table) return gdl_false;
	gdl_string * sfeature = gdl_hashtable_lookup (chrom_table, feature);
	if (!sfeature) return gdl_false;
	return gdl_true;	
}

gdl_boolean
gdl_feature_table_has_chromosome (const gdl_feature_table * table, const gdl_string * chrom)
{
	if (!table->table) return gdl_false;
	gdl_hashtable * chrom_table = gdl_hashtable_lookup (table->table, chrom);
	if (!chrom_table) return gdl_false;
	return gdl_true;	
}

void
gdl_feature_table_free (gdl_feature_table * table)
{
	if (table)
	{
		if (gdl_hashtable_size (table->table))
		{
			gdl_hashtable_itr * citr = gdl_hashtable_iterator (table->table);
			do
			{
				gdl_hashtable * chrom_table = (gdl_hashtable *) gdl_hashtable_iterator_value (citr);
				gdl_hashtable_free (chrom_table);
			}
			while(gdl_hashtable_iterator_next (citr));
			gdl_hashtable_iterator_free (citr);
		}
		gdl_hashtable_free (table->table);
		GDL_FREE (table); 
	}	
}
 
