/* specfunc/elementary.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Author:  G. Jungman */

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_machine.h>
#include <gdl/gdl_sf_elementary.h>

#include "error.h"
#include "check.h"

int
gdl_sf_multiply_e(const double x, const double y, gdl_sf_result * result)
{
  const double ax = fabs(x);
  const double ay = fabs(y);

  if(x == 0.0 || y == 0.0) {
    /* It is necessary to eliminate this immediately.
     */
    result->val = 0.0;
    result->err = 0.0;
    return GDL_SUCCESS;
  }
  else if((ax <= 1.0 && ay >= 1.0) || (ay <= 1.0 && ax >= 1.0)) {
    /* Straddling 1.0 is always safe.
     */
    result->val = x*y;
    result->err = 2.0 * GDL_DBL_EPSILON * fabs(result->val);
    return GDL_SUCCESS;
  }
  else {
    const double f = 1.0 - 2.0 * GDL_DBL_EPSILON;
    const double min = GDL_MIN_DBL(fabs(x), fabs(y));
    const double max = GDL_MAX_DBL(fabs(x), fabs(y));
    if(max < 0.9 * GDL_SQRT_DBL_MAX || min < (f * DBL_MAX)/max) {
      result->val = GDL_COERCE_DBL(x*y);
      result->err = 2.0 * GDL_DBL_EPSILON * fabs(result->val);
      CHECK_UNDERFLOW(result);
      return GDL_SUCCESS;
    }
    else {
      OVERFLOW_ERROR(result);
    }
  }
}


int
gdl_sf_multiply_err_e(const double x, const double dx,
                         const double y, const double dy,
                         gdl_sf_result * result)
{
  int status = gdl_sf_multiply_e(x, y, result);
  result->err += fabs(dx*y) + fabs(dy*x);
  return status;
}


/*-*-*-*-*-*-*-*-*-* Functions w/ Natural Prototypes *-*-*-*-*-*-*-*-*-*-*/

#include "eval.h"

double gdl_sf_multiply(const double x, const double y)
{
  EVAL_RESULT(gdl_sf_multiply_e(x, y, &result));
}

