/* specfunc/gdl_sf_log.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2004 Gerard Jungman
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Author:  G. Jungman */

#ifndef __GDL_SF_LOG_H__
#define __GDL_SF_LOG_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_sf_result.h>

__BEGIN_DECLS

/* Provide a logarithm function with GDL semantics.
 *
 * exceptions: GDL_EDOM
 */
int gdl_sf_log_e(const double x, gdl_sf_result * result);
double gdl_sf_log(const double x);


/* Log(|x|)
 *
 * exceptions: GDL_EDOM
 */
int gdl_sf_log_abs_e(const double x, gdl_sf_result * result);
double gdl_sf_log_abs(const double x);


/* Complex Logarithm
 *   exp(lnr + I theta) = zr + I zi
 * Returns argument in [-pi,pi].
 *
 * exceptions: GDL_EDOM
 */
int gdl_sf_complex_log_e(const double zr, const double zi, gdl_sf_result * lnr, gdl_sf_result * theta);


/* Log(1 + x)
 *
 * exceptions: GDL_EDOM
 */
int gdl_sf_log_1plusx_e(const double x, gdl_sf_result * result);
double gdl_sf_log_1plusx(const double x);


/* Log(1 + x) - x
 *
 * exceptions: GDL_EDOM
 */
int gdl_sf_log_1plusx_mx_e(const double x, gdl_sf_result * result);
double gdl_sf_log_1plusx_mx(const double x);


#ifdef HAVE_INLINE
#include <gdl/gdl_math.h>
#include <gdl/gdl_errno.h>

extern inline
int
gdl_sf_log_e(const double x, gdl_sf_result * result)
{
  /* CHECK_POINTER(result) */

  if(x <= 0.0) {
    result->val = GDL_NAN;
    result->err = GDL_NAN;
    GDL_ERROR ("domain error", GDL_EDOM);
  }
  else {
    result->val = log(x);
    result->err = 2.0 * GDL_DBL_EPSILON * fabs(result->val);
    return GDL_SUCCESS;
  }
}
extern inline
int
gdl_sf_log_abs_e(const double x, gdl_sf_result * result)
{
  /* CHECK_POINTER(result) */

  if(x == 0.0) {
    result->val = GDL_NAN;
    result->err = GDL_NAN;
    GDL_ERROR ("domain error", GDL_EDOM);
  }
  else {
    result->val = log(fabs(x));
    result->err = 2.0 * GDL_DBL_EPSILON * fabs(result->val);
    return GDL_SUCCESS;
  }
}
#endif /* HAVE_INLINE */


__END_DECLS

#endif /* __GDL_SF_LOG_H__ */
