/* vector/gdl_vector_float.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_VECTOR_FLOAT_H__
#define __GDL_VECTOR_FLOAT_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_gblock.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size;
  size_t stride;
  float *data;
  gdl_block_float *block;
  int owner;
} 
gdl_vector_float;

typedef struct
{
  gdl_vector_float vector;
} _gdl_vector_float_view;

typedef _gdl_vector_float_view gdl_vector_float_view;

typedef struct
{
  gdl_vector_float vector;
} _gdl_vector_float_const_view;

typedef const _gdl_vector_float_const_view gdl_vector_float_const_view;


/* Allocation */

gdl_vector_float *gdl_vector_float_alloc (const size_t n);
gdl_vector_float *gdl_vector_float_calloc (const size_t n);

gdl_vector_float *gdl_vector_float_alloc_from_block (gdl_block_float * b,
                                                     const size_t offset, 
                                                     const size_t n, 
                                                     const size_t stride);

gdl_vector_float *gdl_vector_float_alloc_from_vector (gdl_vector_float * v,
                                                      const size_t offset, 
                                                      const size_t n, 
                                                      const size_t stride);

void gdl_vector_float_free (gdl_vector_float * v);

/* Views */

_gdl_vector_float_view 
gdl_vector_float_view_array (float *v, size_t n);

_gdl_vector_float_view 
gdl_vector_float_view_array_with_stride (float *base,
                                         size_t stride,
                                         size_t n);

_gdl_vector_float_const_view 
gdl_vector_float_const_view_array (const float *v, size_t n);

_gdl_vector_float_const_view 
gdl_vector_float_const_view_array_with_stride (const float *base,
                                               size_t stride,
                                               size_t n);

_gdl_vector_float_view 
gdl_vector_float_subvector (gdl_vector_float *v, 
                            size_t i, 
                            size_t n);

_gdl_vector_float_view 
gdl_vector_float_subvector_with_stride (gdl_vector_float *v, 
                                        size_t i,
                                        size_t stride,
                                        size_t n);

_gdl_vector_float_const_view 
gdl_vector_float_const_subvector (const gdl_vector_float *v, 
                                  size_t i, 
                                  size_t n);

_gdl_vector_float_const_view 
gdl_vector_float_const_subvector_with_stride (const gdl_vector_float *v, 
                                              size_t i, 
                                              size_t stride,
                                              size_t n);

/* Operations */

float gdl_vector_float_get (const gdl_vector_float * v, const size_t i);
void gdl_vector_float_set (gdl_vector_float * v, const size_t i, float x);

float *gdl_vector_float_ptr (gdl_vector_float * v, const size_t i);
const float *gdl_vector_float_const_ptr (const gdl_vector_float * v, const size_t i);

void gdl_vector_float_set_zero (gdl_vector_float * v);
void gdl_vector_float_set_all (gdl_vector_float * v, float x);
int gdl_vector_float_set_basis (gdl_vector_float * v, size_t i);

gdl_vector_float * gdl_vector_float_fread (FILE * stream);
int gdl_vector_float_fwrite (FILE * stream, const gdl_vector_float * v);
gdl_vector_float * gdl_vector_float_fscanf (FILE * stream);
int gdl_vector_float_fprintf (FILE * stream, const gdl_vector_float * v,
                              const char *format);

int gdl_vector_float_memcpy (gdl_vector_float * dest, const gdl_vector_float * src);

int gdl_vector_float_reverse (gdl_vector_float * v);

int gdl_vector_float_swap (gdl_vector_float * v, gdl_vector_float * w);
int gdl_vector_float_swap_elements (gdl_vector_float * v, const size_t i, const size_t j);

float gdl_vector_float_max (const gdl_vector_float * v);
float gdl_vector_float_min (const gdl_vector_float * v);
void gdl_vector_float_minmax (const gdl_vector_float * v, float * min_out, float * max_out);

size_t gdl_vector_float_max_index (const gdl_vector_float * v);
size_t gdl_vector_float_min_index (const gdl_vector_float * v);
void gdl_vector_float_minmax_index (const gdl_vector_float * v, size_t * imin, size_t * imax);

int gdl_vector_float_add (gdl_vector_float * a, const gdl_vector_float * b);
int gdl_vector_float_sub (gdl_vector_float * a, const gdl_vector_float * b);
int gdl_vector_float_mul (gdl_vector_float * a, const gdl_vector_float * b);
int gdl_vector_float_div (gdl_vector_float * a, const gdl_vector_float * b);
int gdl_vector_float_scale (gdl_vector_float * a, const double x);
int gdl_vector_float_add_constant (gdl_vector_float * a, const double x);

int gdl_vector_float_isnull (const gdl_vector_float * v);

#ifdef HAVE_INLINE

extern inline
float
gdl_vector_float_get (const gdl_vector_float * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
    }
#endif
  return v->data[i * v->stride];
}

extern inline
void
gdl_vector_float_set (gdl_vector_float * v, const size_t i, float x)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
    }
#endif
  v->data[i * v->stride] = x;
}

extern inline
float *
gdl_vector_float_ptr (gdl_vector_float * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (float *) (v->data + i * v->stride);
}

extern inline
const float *
gdl_vector_float_const_ptr (const gdl_vector_float * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (const float *) (v->data + i * v->stride);
}


#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_VECTOR_FLOAT_H__ */


