/* vector/gdl_vector_int.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_VECTOR_INT_H__
#define __GDL_VECTOR_INT_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_gblock.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size;
  size_t stride;
  int *data;
  gdl_block_int *block;
  int owner;
} 
gdl_vector_int;

typedef struct
{
  gdl_vector_int vector;
} _gdl_vector_int_view;

typedef _gdl_vector_int_view gdl_vector_int_view;

typedef struct
{
  gdl_vector_int vector;
} _gdl_vector_int_const_view;

typedef const _gdl_vector_int_const_view gdl_vector_int_const_view;


/* Allocation */

gdl_vector_int *gdl_vector_int_alloc (const size_t n);
gdl_vector_int *gdl_vector_int_calloc (const size_t n);

gdl_vector_int *gdl_vector_int_alloc_from_block (gdl_block_int * b,
                                                     const size_t offset, 
                                                     const size_t n, 
                                                     const size_t stride);

gdl_vector_int *gdl_vector_int_alloc_from_vector (gdl_vector_int * v,
                                                      const size_t offset, 
                                                      const size_t n, 
                                                      const size_t stride);

void gdl_vector_int_free (gdl_vector_int * v);

/* Views */

_gdl_vector_int_view 
gdl_vector_int_view_array (int *v, size_t n);

_gdl_vector_int_view 
gdl_vector_int_view_array_with_stride (int *base,
                                         size_t stride,
                                         size_t n);

_gdl_vector_int_const_view 
gdl_vector_int_const_view_array (const int *v, size_t n);

_gdl_vector_int_const_view 
gdl_vector_int_const_view_array_with_stride (const int *base,
                                               size_t stride,
                                               size_t n);

_gdl_vector_int_view 
gdl_vector_int_subvector (gdl_vector_int *v, 
                            size_t i, 
                            size_t n);

_gdl_vector_int_view 
gdl_vector_int_subvector_with_stride (gdl_vector_int *v, 
                                        size_t i,
                                        size_t stride,
                                        size_t n);

_gdl_vector_int_const_view 
gdl_vector_int_const_subvector (const gdl_vector_int *v, 
                                  size_t i, 
                                  size_t n);

_gdl_vector_int_const_view 
gdl_vector_int_const_subvector_with_stride (const gdl_vector_int *v, 
                                              size_t i, 
                                              size_t stride,
                                              size_t n);

/* Operations */

int gdl_vector_int_get (const gdl_vector_int * v, const size_t i);
void gdl_vector_int_set (gdl_vector_int * v, const size_t i, int x);

int *gdl_vector_int_ptr (gdl_vector_int * v, const size_t i);
const int *gdl_vector_int_const_ptr (const gdl_vector_int * v, const size_t i);

void gdl_vector_int_set_zero (gdl_vector_int * v);
void gdl_vector_int_set_all (gdl_vector_int * v, int x);
int gdl_vector_int_set_basis (gdl_vector_int * v, size_t i);

gdl_vector_int * gdl_vector_int_fread (FILE * stream);
int gdl_vector_int_fwrite (FILE * stream, const gdl_vector_int * v);
gdl_vector_int * gdl_vector_int_fscanf (FILE * stream);
int gdl_vector_int_fprintf (FILE * stream, const gdl_vector_int * v,
                              const char *format);

int gdl_vector_int_memcpy (gdl_vector_int * dest, const gdl_vector_int * src);

int gdl_vector_int_reverse (gdl_vector_int * v);

int gdl_vector_int_swap (gdl_vector_int * v, gdl_vector_int * w);
int gdl_vector_int_swap_elements (gdl_vector_int * v, const size_t i, const size_t j);

int gdl_vector_int_max (const gdl_vector_int * v);
int gdl_vector_int_min (const gdl_vector_int * v);
void gdl_vector_int_minmax (const gdl_vector_int * v, int * min_out, int * max_out);

size_t gdl_vector_int_max_index (const gdl_vector_int * v);
size_t gdl_vector_int_min_index (const gdl_vector_int * v);
void gdl_vector_int_minmax_index (const gdl_vector_int * v, size_t * imin, size_t * imax);

int gdl_vector_int_add (gdl_vector_int * a, const gdl_vector_int * b);
int gdl_vector_int_sub (gdl_vector_int * a, const gdl_vector_int * b);
int gdl_vector_int_mul (gdl_vector_int * a, const gdl_vector_int * b);
int gdl_vector_int_div (gdl_vector_int * a, const gdl_vector_int * b);
int gdl_vector_int_scale (gdl_vector_int * a, const double x);
int gdl_vector_int_add_constant (gdl_vector_int * a, const double x);

int gdl_vector_int_isnull (const gdl_vector_int * v);

#ifdef HAVE_INLINE

extern inline
int
gdl_vector_int_get (const gdl_vector_int * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
    }
#endif
  return v->data[i * v->stride];
}

extern inline
void
gdl_vector_int_set (gdl_vector_int * v, const size_t i, int x)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
    }
#endif
  v->data[i * v->stride] = x;
}

extern inline
int *
gdl_vector_int_ptr (gdl_vector_int * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (int *) (v->data + i * v->stride);
}

extern inline
const int *
gdl_vector_int_const_ptr (const gdl_vector_int * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (const int *) (v->data + i * v->stride);
}


#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_VECTOR_INT_H__ */


