/* vector/gdl_vector_long.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_VECTOR_LONG_H__
#define __GDL_VECTOR_LONG_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_gblock.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size;
  size_t stride;
  long *data;
  gdl_block_long *block;
  int owner;
} 
gdl_vector_long;

typedef struct
{
  gdl_vector_long vector;
} _gdl_vector_long_view;

typedef _gdl_vector_long_view gdl_vector_long_view;

typedef struct
{
  gdl_vector_long vector;
} _gdl_vector_long_const_view;

typedef const _gdl_vector_long_const_view gdl_vector_long_const_view;


/* Allocation */

gdl_vector_long *gdl_vector_long_alloc (const size_t n);
gdl_vector_long *gdl_vector_long_calloc (const size_t n);

gdl_vector_long *gdl_vector_long_alloc_from_block (gdl_block_long * b,
                                                     const size_t offset, 
                                                     const size_t n, 
                                                     const size_t stride);

gdl_vector_long *gdl_vector_long_alloc_from_vector (gdl_vector_long * v,
                                                      const size_t offset, 
                                                      const size_t n, 
                                                      const size_t stride);

void gdl_vector_long_free (gdl_vector_long * v);

/* Views */

_gdl_vector_long_view 
gdl_vector_long_view_array (long *v, size_t n);

_gdl_vector_long_view 
gdl_vector_long_view_array_with_stride (long *base,
                                         size_t stride,
                                         size_t n);

_gdl_vector_long_const_view 
gdl_vector_long_const_view_array (const long *v, size_t n);

_gdl_vector_long_const_view 
gdl_vector_long_const_view_array_with_stride (const long *base,
                                               size_t stride,
                                               size_t n);

_gdl_vector_long_view 
gdl_vector_long_subvector (gdl_vector_long *v, 
                            size_t i, 
                            size_t n);

_gdl_vector_long_view 
gdl_vector_long_subvector_with_stride (gdl_vector_long *v, 
                                        size_t i,
                                        size_t stride,
                                        size_t n);

_gdl_vector_long_const_view 
gdl_vector_long_const_subvector (const gdl_vector_long *v, 
                                  size_t i, 
                                  size_t n);

_gdl_vector_long_const_view 
gdl_vector_long_const_subvector_with_stride (const gdl_vector_long *v, 
                                              size_t i, 
                                              size_t stride,
                                              size_t n);

/* Operations */

long gdl_vector_long_get (const gdl_vector_long * v, const size_t i);
void gdl_vector_long_set (gdl_vector_long * v, const size_t i, long x);

long *gdl_vector_long_ptr (gdl_vector_long * v, const size_t i);
const long *gdl_vector_long_const_ptr (const gdl_vector_long * v, const size_t i);

void gdl_vector_long_set_zero (gdl_vector_long * v);
void gdl_vector_long_set_all (gdl_vector_long * v, long x);
int gdl_vector_long_set_basis (gdl_vector_long * v, size_t i);

gdl_vector_long * gdl_vector_long_fread (FILE * stream);
int gdl_vector_long_fwrite (FILE * stream, const gdl_vector_long * v);
gdl_vector_long * gdl_vector_long_fscanf (FILE * stream);
int gdl_vector_long_fprintf (FILE * stream, const gdl_vector_long * v,
                              const char *format);

int gdl_vector_long_memcpy (gdl_vector_long * dest, const gdl_vector_long * src);

int gdl_vector_long_reverse (gdl_vector_long * v);

int gdl_vector_long_swap (gdl_vector_long * v, gdl_vector_long * w);
int gdl_vector_long_swap_elements (gdl_vector_long * v, const size_t i, const size_t j);

long gdl_vector_long_max (const gdl_vector_long * v);
long gdl_vector_long_min (const gdl_vector_long * v);
void gdl_vector_long_minmax (const gdl_vector_long * v, long * min_out, long * max_out);

size_t gdl_vector_long_max_index (const gdl_vector_long * v);
size_t gdl_vector_long_min_index (const gdl_vector_long * v);
void gdl_vector_long_minmax_index (const gdl_vector_long * v, size_t * imin, size_t * imax);

int gdl_vector_long_add (gdl_vector_long * a, const gdl_vector_long * b);
int gdl_vector_long_sub (gdl_vector_long * a, const gdl_vector_long * b);
int gdl_vector_long_mul (gdl_vector_long * a, const gdl_vector_long * b);
int gdl_vector_long_div (gdl_vector_long * a, const gdl_vector_long * b);
int gdl_vector_long_scale (gdl_vector_long * a, const double x);
int gdl_vector_long_add_constant (gdl_vector_long * a, const double x);

int gdl_vector_long_isnull (const gdl_vector_long * v);

#ifdef HAVE_INLINE

extern inline
long
gdl_vector_long_get (const gdl_vector_long * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
    }
#endif
  return v->data[i * v->stride];
}

extern inline
void
gdl_vector_long_set (gdl_vector_long * v, const size_t i, long x)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
    }
#endif
  v->data[i * v->stride] = x;
}

extern inline
long *
gdl_vector_long_ptr (gdl_vector_long * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (long *) (v->data + i * v->stride);
}

extern inline
const long *
gdl_vector_long_const_ptr (const gdl_vector_long * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (const long *) (v->data + i * v->stride);
}


#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_VECTOR_LONG_H__ */


