/* vector/gdl_vector_short.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_VECTOR_SHORT_H__
#define __GDL_VECTOR_SHORT_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_gblock.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size;
  size_t stride;
  short *data;
  gdl_block_short *block;
  int owner;
} 
gdl_vector_short;

typedef struct
{
  gdl_vector_short vector;
} _gdl_vector_short_view;

typedef _gdl_vector_short_view gdl_vector_short_view;

typedef struct
{
  gdl_vector_short vector;
} _gdl_vector_short_const_view;

typedef const _gdl_vector_short_const_view gdl_vector_short_const_view;


/* Allocation */

gdl_vector_short *gdl_vector_short_alloc (const size_t n);
gdl_vector_short *gdl_vector_short_calloc (const size_t n);

gdl_vector_short *gdl_vector_short_alloc_from_block (gdl_block_short * b,
                                                     const size_t offset, 
                                                     const size_t n, 
                                                     const size_t stride);

gdl_vector_short *gdl_vector_short_alloc_from_vector (gdl_vector_short * v,
                                                      const size_t offset, 
                                                      const size_t n, 
                                                      const size_t stride);

void gdl_vector_short_free (gdl_vector_short * v);

/* Views */

_gdl_vector_short_view 
gdl_vector_short_view_array (short *v, size_t n);

_gdl_vector_short_view 
gdl_vector_short_view_array_with_stride (short *base,
                                         size_t stride,
                                         size_t n);

_gdl_vector_short_const_view 
gdl_vector_short_const_view_array (const short *v, size_t n);

_gdl_vector_short_const_view 
gdl_vector_short_const_view_array_with_stride (const short *base,
                                               size_t stride,
                                               size_t n);

_gdl_vector_short_view 
gdl_vector_short_subvector (gdl_vector_short *v, 
                            size_t i, 
                            size_t n);

_gdl_vector_short_view 
gdl_vector_short_subvector_with_stride (gdl_vector_short *v, 
                                        size_t i,
                                        size_t stride,
                                        size_t n);

_gdl_vector_short_const_view 
gdl_vector_short_const_subvector (const gdl_vector_short *v, 
                                  size_t i, 
                                  size_t n);

_gdl_vector_short_const_view 
gdl_vector_short_const_subvector_with_stride (const gdl_vector_short *v, 
                                              size_t i, 
                                              size_t stride,
                                              size_t n);

/* Operations */

short gdl_vector_short_get (const gdl_vector_short * v, const size_t i);
void gdl_vector_short_set (gdl_vector_short * v, const size_t i, short x);

short *gdl_vector_short_ptr (gdl_vector_short * v, const size_t i);
const short *gdl_vector_short_const_ptr (const gdl_vector_short * v, const size_t i);

void gdl_vector_short_set_zero (gdl_vector_short * v);
void gdl_vector_short_set_all (gdl_vector_short * v, short x);
int gdl_vector_short_set_basis (gdl_vector_short * v, size_t i);

gdl_vector_short * gdl_vector_short_fread (FILE * stream);
int gdl_vector_short_fwrite (FILE * stream, const gdl_vector_short * v);
gdl_vector_short * gdl_vector_short_fscanf (FILE * stream);
int gdl_vector_short_fprintf (FILE * stream, const gdl_vector_short * v,
                              const char *format);

int gdl_vector_short_memcpy (gdl_vector_short * dest, const gdl_vector_short * src);

int gdl_vector_short_reverse (gdl_vector_short * v);

int gdl_vector_short_swap (gdl_vector_short * v, gdl_vector_short * w);
int gdl_vector_short_swap_elements (gdl_vector_short * v, const size_t i, const size_t j);

short gdl_vector_short_max (const gdl_vector_short * v);
short gdl_vector_short_min (const gdl_vector_short * v);
void gdl_vector_short_minmax (const gdl_vector_short * v, short * min_out, short * max_out);

size_t gdl_vector_short_max_index (const gdl_vector_short * v);
size_t gdl_vector_short_min_index (const gdl_vector_short * v);
void gdl_vector_short_minmax_index (const gdl_vector_short * v, size_t * imin, size_t * imax);

int gdl_vector_short_add (gdl_vector_short * a, const gdl_vector_short * b);
int gdl_vector_short_sub (gdl_vector_short * a, const gdl_vector_short * b);
int gdl_vector_short_mul (gdl_vector_short * a, const gdl_vector_short * b);
int gdl_vector_short_div (gdl_vector_short * a, const gdl_vector_short * b);
int gdl_vector_short_scale (gdl_vector_short * a, const double x);
int gdl_vector_short_add_constant (gdl_vector_short * a, const double x);

int gdl_vector_short_isnull (const gdl_vector_short * v);

#ifdef HAVE_INLINE

extern inline
short
gdl_vector_short_get (const gdl_vector_short * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
    }
#endif
  return v->data[i * v->stride];
}

extern inline
void
gdl_vector_short_set (gdl_vector_short * v, const size_t i, short x)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
    }
#endif
  v->data[i * v->stride] = x;
}

extern inline
short *
gdl_vector_short_ptr (gdl_vector_short * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (short *) (v->data + i * v->stride);
}

extern inline
const short *
gdl_vector_short_const_ptr (const gdl_vector_short * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (const short *) (v->data + i * v->stride);
}


#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_VECTOR_SHORT_H__ */


