/* vector/gdl_vector_uchar.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_VECTOR_UCHAR_H__
#define __GDL_VECTOR_UCHAR_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_gblock.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size;
  size_t stride;
  unsigned char *data;
  gdl_block_uchar *block;
  int owner;
} 
gdl_vector_uchar;

typedef struct
{
  gdl_vector_uchar vector;
} _gdl_vector_uchar_view;

typedef _gdl_vector_uchar_view gdl_vector_uchar_view;

typedef struct
{
  gdl_vector_uchar vector;
} _gdl_vector_uchar_const_view;

typedef const _gdl_vector_uchar_const_view gdl_vector_uchar_const_view;


/* Allocation */

gdl_vector_uchar *gdl_vector_uchar_alloc (const size_t n);
gdl_vector_uchar *gdl_vector_uchar_calloc (const size_t n);

gdl_vector_uchar *gdl_vector_uchar_alloc_from_block (gdl_block_uchar * b,
                                                     const size_t offset, 
                                                     const size_t n, 
                                                     const size_t stride);

gdl_vector_uchar *gdl_vector_uchar_alloc_from_vector (gdl_vector_uchar * v,
                                                      const size_t offset, 
                                                      const size_t n, 
                                                      const size_t stride);

void gdl_vector_uchar_free (gdl_vector_uchar * v);

/* Views */

_gdl_vector_uchar_view 
gdl_vector_uchar_view_array (unsigned char *v, size_t n);

_gdl_vector_uchar_view 
gdl_vector_uchar_view_array_with_stride (unsigned char *base,
                                         size_t stride,
                                         size_t n);

_gdl_vector_uchar_const_view 
gdl_vector_uchar_const_view_array (const unsigned char *v, size_t n);

_gdl_vector_uchar_const_view 
gdl_vector_uchar_const_view_array_with_stride (const unsigned char *base,
                                               size_t stride,
                                               size_t n);

_gdl_vector_uchar_view 
gdl_vector_uchar_subvector (gdl_vector_uchar *v, 
                            size_t i, 
                            size_t n);

_gdl_vector_uchar_view 
gdl_vector_uchar_subvector_with_stride (gdl_vector_uchar *v, 
                                        size_t i,
                                        size_t stride,
                                        size_t n);

_gdl_vector_uchar_const_view 
gdl_vector_uchar_const_subvector (const gdl_vector_uchar *v, 
                                  size_t i, 
                                  size_t n);

_gdl_vector_uchar_const_view 
gdl_vector_uchar_const_subvector_with_stride (const gdl_vector_uchar *v, 
                                              size_t i, 
                                              size_t stride,
                                              size_t n);

/* Operations */

unsigned char gdl_vector_uchar_get (const gdl_vector_uchar * v, const size_t i);
void gdl_vector_uchar_set (gdl_vector_uchar * v, const size_t i, unsigned char x);

unsigned char *gdl_vector_uchar_ptr (gdl_vector_uchar * v, const size_t i);
const unsigned char *gdl_vector_uchar_const_ptr (const gdl_vector_uchar * v, const size_t i);

void gdl_vector_uchar_set_zero (gdl_vector_uchar * v);
void gdl_vector_uchar_set_all (gdl_vector_uchar * v, unsigned char x);
int gdl_vector_uchar_set_basis (gdl_vector_uchar * v, size_t i);

gdl_vector_uchar * gdl_vector_uchar_fread (FILE * stream);
int gdl_vector_uchar_fwrite (FILE * stream, const gdl_vector_uchar * v);
gdl_vector_uchar * gdl_vector_uchar_fscanf (FILE * stream);
int gdl_vector_uchar_fprintf (FILE * stream, const gdl_vector_uchar * v,
                              const char *format);

int gdl_vector_uchar_memcpy (gdl_vector_uchar * dest, const gdl_vector_uchar * src);

int gdl_vector_uchar_reverse (gdl_vector_uchar * v);

int gdl_vector_uchar_swap (gdl_vector_uchar * v, gdl_vector_uchar * w);
int gdl_vector_uchar_swap_elements (gdl_vector_uchar * v, const size_t i, const size_t j);

unsigned char gdl_vector_uchar_max (const gdl_vector_uchar * v);
unsigned char gdl_vector_uchar_min (const gdl_vector_uchar * v);
void gdl_vector_uchar_minmax (const gdl_vector_uchar * v, unsigned char * min_out, unsigned char * max_out);

size_t gdl_vector_uchar_max_index (const gdl_vector_uchar * v);
size_t gdl_vector_uchar_min_index (const gdl_vector_uchar * v);
void gdl_vector_uchar_minmax_index (const gdl_vector_uchar * v, size_t * imin, size_t * imax);

int gdl_vector_uchar_add (gdl_vector_uchar * a, const gdl_vector_uchar * b);
int gdl_vector_uchar_sub (gdl_vector_uchar * a, const gdl_vector_uchar * b);
int gdl_vector_uchar_mul (gdl_vector_uchar * a, const gdl_vector_uchar * b);
int gdl_vector_uchar_div (gdl_vector_uchar * a, const gdl_vector_uchar * b);
int gdl_vector_uchar_scale (gdl_vector_uchar * a, const double x);
int gdl_vector_uchar_add_constant (gdl_vector_uchar * a, const double x);

int gdl_vector_uchar_isnull (const gdl_vector_uchar * v);

#ifdef HAVE_INLINE

extern inline
unsigned char
gdl_vector_uchar_get (const gdl_vector_uchar * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
    }
#endif
  return v->data[i * v->stride];
}

extern inline
void
gdl_vector_uchar_set (gdl_vector_uchar * v, const size_t i, unsigned char x)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
    }
#endif
  v->data[i * v->stride] = x;
}

extern inline
unsigned char *
gdl_vector_uchar_ptr (gdl_vector_uchar * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (unsigned char *) (v->data + i * v->stride);
}

extern inline
const unsigned char *
gdl_vector_uchar_const_ptr (const gdl_vector_uchar * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (const unsigned char *) (v->data + i * v->stride);
}


#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_VECTOR_UCHAR_H__ */


