/* vector/init_source.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

TYPE (gdl_vector) *
FUNCTION (gdl_vector, alloc) (const size_t n)
{
  TYPE (gdl_block) * block;
  TYPE (gdl_vector) * v;

  if (n == 0)
    {
      GDL_ERROR_VAL ("vector length n must be positive integer",
                        GDL_EINVAL, 0);
    }

  v = (TYPE (gdl_vector) *) malloc (sizeof (TYPE (gdl_vector)));

  if (v == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for vector struct",
                        GDL_ENOMEM, 0);
    }

  block = FUNCTION (gdl_block,alloc2) (1, n);

  if (block == 0)
    {
      GDL_FREE (v) ;

      GDL_ERROR_VAL ("failed to allocate space for block",
                        GDL_ENOMEM, 0);
    }
      
  v->data = block->data ;
  v->size = n;
  v->stride = 1;
  v->block = block;
  v->owner = 1;

  return v;
}

TYPE (gdl_vector) *
FUNCTION (gdl_vector, calloc) (const size_t n)
{
  size_t i;

  TYPE (gdl_vector) * v = FUNCTION (gdl_vector, alloc) (n);

  if (v == 0)
    return 0;

  /* initialize vector to zero */

  for (i = 0; i < MULTIPLICITY * n; i++)
    {
      v->data[i] = 0;
    }

  return v;
}

TYPE (gdl_vector) *
FUNCTION (gdl_vector, alloc_from_block) (TYPE(gdl_block) * block, 
                                         const size_t offset, 
                                         const size_t n, 
                                         const size_t stride)
{
  TYPE (gdl_vector) * v;

  if (n == 0)
    {
      GDL_ERROR_VAL ("vector length n must be positive integer",
                        GDL_EINVAL, 0);
    }

  if (stride == 0)
    {
      GDL_ERROR_VAL ("stride must be positive integer", GDL_EINVAL, 0);
    }

  if (block->size <= offset + (n - 1) * stride)
    {
      GDL_ERROR_VAL ("vector would extend past end of block", GDL_EINVAL, 0);
    }

  v = (TYPE (gdl_vector) *) malloc (sizeof (TYPE (gdl_vector)));

  if (v == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for vector struct",
                        GDL_ENOMEM, 0);
    }

  v->data   = block->data + MULTIPLICITY * offset ;
  v->size   = n;
  v->stride = stride;
  v->block  = block;
  v->owner  = 0;

  return v;
}

TYPE (gdl_vector) *
FUNCTION (gdl_vector, alloc_from_vector) (TYPE(gdl_vector) * w, 
                                         const size_t offset, 
                                         const size_t n, 
                                         const size_t stride)
{
  TYPE (gdl_vector) * v;

  if (n == 0)
    {
      GDL_ERROR_VAL ("vector length n must be positive integer",
                        GDL_EINVAL, 0);
    }

  if (stride == 0)
    {
      GDL_ERROR_VAL ("stride must be positive integer", GDL_EINVAL, 0);
    }

  if (offset + (n - 1) * stride >= w->size)
    {
      GDL_ERROR_VAL ("vector would extend past end of block", GDL_EINVAL, 0);
    }

  v = (TYPE (gdl_vector) *) malloc (sizeof (TYPE (gdl_vector)));

  if (v == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for vector struct",
                        GDL_ENOMEM, 0);
    }

  v->data = w->data + MULTIPLICITY * w->stride * offset ;
  v->size = n;
  v->stride = stride * w->stride;
  v->block = w->block;
  v->owner = 0;

  return v;
}


void
FUNCTION (gdl_vector, free) (TYPE (gdl_vector) * v)
{
  if (v && v->owner)
    {
      FUNCTION(gdl_block, free) (v->block) ;
    }
  GDL_FREE (v);
}


void
FUNCTION (gdl_vector, set_all) (TYPE (gdl_vector) * v, BASE x)
{
  ATOMIC * const data = v->data;
  const size_t n = v->size;
  const size_t stride = v->stride;

  size_t i;

  for (i = 0; i < n; i++)
    {
      *(BASE *) (data + MULTIPLICITY * i * stride) = x;
    }
}

void
FUNCTION (gdl_vector, set_zero) (TYPE (gdl_vector) * v)
{
  ATOMIC * const data = v->data;
  const size_t n = v->size;
  const size_t stride = v->stride;
  const BASE zero = ZERO ;

  size_t i;

  for (i = 0; i < n; i++)
    {
      *(BASE *) (data + MULTIPLICITY * i * stride) = zero;
    }
}

int
FUNCTION (gdl_vector, set_basis) (TYPE (gdl_vector) * v, size_t i)
{
  ATOMIC * const data = v->data;
  const size_t n = v->size;
  const size_t stride = v->stride;
  const BASE zero = ZERO ;
  const BASE one = ONE;

  size_t k;

  if (i >= n)
    {
      GDL_ERROR ("index out of range", GDL_EINVAL);
    }

  for (k = 0; k < n; k++)
    {
      *(BASE *) (data + MULTIPLICITY * k * stride) = zero;
    }

  *(BASE *) (data + MULTIPLICITY * i * stride) = one;

  return GDL_SUCCESS;
}
