/*  
 *  view/shared.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_hmap.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_mask.h>
#include <gdl/gdl_fview_wrapper.h>
#include <gdl/gdl_view.h>

size_t
gdl_view_shared_accession (const gdl_gview * gview, gdl_mask ** gmaskptr, const gdl_fview * fview, gdl_mask ** fmaskptr)
{
	size_t i, j, ng, nf, nrm, nc = 0;
	int * fidx;
	gdl_mask * gmask, * fmask;
	
	gmask = *gmaskptr;
	fmask = *fmaskptr;
	
	if (!gdl_mask_size (gmask, GDL_ACCESSION))
	{
		gdl_mask_set_identity (gmask, GDL_ACCESSION, gdl_gview_accession_size (gview));
	}
	
	ng = GDL_GVIEW_ACCESSION_SIZE (gview, gmask);
	nf = GDL_FVIEW_ACCESSION_SIZE (fview, fmask);
	
	//printf ("NG = %d NF = %d\n", ng, nf);
	
	fidx = GDL_MALLOC (int, ng);
	
	for (i = 0; i < ng; i++)
	{
		fidx[i] = -1;	
	}
	
	for (i = 0; i < GDL_GVIEW_ACCESSION_SIZE (gview, gmask); i++)
	{
		gdl_accession * ga = GDL_GVIEW_GET_ACCESSION (gview, gmask, i);
		gdl_accession * fa = gdl_fview_search_accession (fview, gdl_entity_get_name (ga));
		
		if (fa)
		{
			//printf ("SHARED ACCESSION (%s)\n", gdl_entity_get_name (fa));
			for (j = 0; j < nf; j++)
			{
				if (fa == GDL_FVIEW_GET_ACCESSION (fview, fmask, j))
				{
					break;	
				}	
			}
			if (j < nf)
			{
				fidx[i] = gdl_entity_get_idx (fa);
				nc++;
				//printf ("\tFIDX[%d] = %d\n", i, fidx[i]);
			}
			else
			{
				//printf ("\tBUT ACCESSION (%s) IS NOT IN MASK\n", gdl_entity_get_name (fa));
				gdl_mask_remove (gmask, GDL_ACCESSION, i);
				i--;
			}
		}
		else
		{
			//printf ("NOT SHARED ACCESSION (%s)\n", gdl_entity_get_name (ga));
			gdl_mask_remove (gmask, GDL_ACCESSION, i);
			i--;
		}
	}
	
	//printf ("COMMON %d (%d)\n", nc, gdl_mask_size (gmask, GDL_ACCESSION));
	
	gdl_mask_reset (fmask, GDL_ACCESSION);
	
	for (j = i = 0; i < ng; i++)
	{
		if (fidx[i]!=-1)
		{
			gdl_mask_add_idx (fmask, GDL_ACCESSION, fidx[i]);
//			printf ("ACCESSION %s\n", gdl_entity_get_name (GDL_GVIEW_GET_ACCESSION (gview, gmask, j)));
//			printf ("\tGMASK %d ==> %d\n", j, gdl_mask_get_idx (gmask, GDL_ACCESSION, j));
//			printf ("\tFMASK %d ==> %d (%s)\n", j, fidx[i], gdl_entity_get_name (gdl_fview_get_accession (fview, fidx[i])));
			j++;
		}
	}	
	
}

size_t
gdl_view_wrapper_shared_accession (const gdl_gview_wrapper * gview, gdl_mask ** gmaskptr, const gdl_fview_wrapper * fview, gdl_mask ** fmaskptr)
{
	size_t i, j, ng, nf, nrm, nc = 0;
	int * fidx;
	gdl_mask * gmask, * fmask;
	
	gmask = *gmaskptr;
	fmask = *fmaskptr;
	
	if (!gdl_mask_size (gmask, GDL_ACCESSION))
	{
		gdl_mask_set_identity (gmask, GDL_ACCESSION, gdl_gview_wrapper_accession_size (gview));
	}
	
	ng = gdl_mask_size (gmask, GDL_ACCESSION);
	nf = (gdl_mask_size (fmask, GDL_ACCESSION)) ? gdl_mask_size (fmask, GDL_ACCESSION): gdl_fview_wrapper_accession_size (fview);
	
	fidx = GDL_MALLOC (int, ng);
	
	for (i = 0; i < ng; i++)
	{
		fidx[i] = -1;	
	}
	
	for (i = 0; i < gdl_mask_size (gmask, GDL_ACCESSION); i++)
	{
		gdl_accession * ga = gdl_gview_wrapper_get_accession (gview, gdl_mask_get_idx (gmask, GDL_ACCESSION, i));
		gdl_accession * fa = gdl_fview_search_accession (gdl_fview_wrapper_fview (fview), gdl_entity_get_name (ga));
		
		fidx[i] = -1;
		
		if (fa)
		{
			gdl_accession * fam;
			
			for (j = 0; j < nf; j++)
			{
				fam = gdl_fview_wrapper_get_accession (fview, gdl_mask_get_idx (fmask, GDL_ACCESSION, j));
				if (!strcmp (gdl_entity_get_name (fam), gdl_entity_get_name (fa)))
				{
					break;
				}
			}
			if (j < nf)
			{
				fidx[i] = gdl_entity_get_idx (fam);
				nc++;
			}
			else
			{
				gdl_mask_remove (gmask, GDL_ACCESSION, i);
				i--;
			}
		}
		else
		{
			gdl_mask_remove (gmask, GDL_ACCESSION, i);
			i--;
		}
	}
	
	gdl_mask_reset (fmask, GDL_ACCESSION);
	
	for (j = i = 0; i < ng; i++)
	{
		if (fidx[i]!=-1)
		{
			gdl_mask_add_idx (fmask, GDL_ACCESSION, fidx[i]);
			j++;
		}
	}
}

size_t
gdl_view_generic_wrapper_shared_accession (const void * wrapper1, const gdl_entity_type * wtype1, gdl_mask ** mask1ptr, const void * wrapper2, const gdl_entity_type * wtype2, gdl_mask ** mask2ptr)
{
	size_t i, j, ng, nf, nrm, nc = 0;
	int * fidx;
	gdl_mask * gmask, * fmask;
	
	gmask = *mask1ptr;
	fmask = *mask2ptr;
	
	if (!gdl_mask_size (gmask, GDL_ACCESSION))
	{
		if (wtype1 == GDL_LOCUS) gdl_mask_set_identity (gmask, GDL_ACCESSION, gdl_gview_wrapper_accession_size (wrapper1));
		if (wtype1 == GDL_FACTOR) gdl_mask_set_identity (gmask, GDL_ACCESSION, gdl_fview_wrapper_accession_size (wrapper1));
	}
	
	ng = gdl_mask_size (gmask, GDL_ACCESSION);
	
	if (wtype2 == GDL_LOCUS)  nf = (gdl_mask_size (fmask, GDL_ACCESSION)) ? gdl_mask_size (fmask, GDL_ACCESSION): gdl_gview_wrapper_accession_size (wrapper2);
	if (wtype2 == GDL_FACTOR) nf = (gdl_mask_size (fmask, GDL_ACCESSION)) ? gdl_mask_size (fmask, GDL_ACCESSION): gdl_fview_wrapper_accession_size (wrapper2);
	
	fidx = GDL_MALLOC (int, ng);
	
	for (i = 0; i < ng; i++)
	{
		fidx[i] = -1;	
	}
	
	for (i = 0; i < gdl_mask_size (gmask, GDL_ACCESSION); i++)
	{
		gdl_accession * ga;
		gdl_accession * fa;
		
		if (wtype1 == GDL_LOCUS)  ga = gdl_gview_wrapper_get_accession (wrapper1, gdl_mask_get_idx (gmask, GDL_ACCESSION, i));
		if (wtype1 == GDL_FACTOR) ga = gdl_fview_wrapper_get_accession (wrapper1, gdl_mask_get_idx (gmask, GDL_ACCESSION, i));
		if (wtype2 == GDL_LOCUS)  fa = gdl_gview_search_accession (gdl_gview_wrapper_gview (wrapper2), gdl_entity_get_name (ga));
		if (wtype2 == GDL_FACTOR) fa = gdl_fview_search_accession (gdl_fview_wrapper_fview (wrapper2), gdl_entity_get_name (ga));
		
		fidx[i] = -1;
		
		if (fa)
		{
			gdl_accession * fam;
			
			for (j = 0; j < nf; j++)
			{
				if (wtype2 == GDL_LOCUS) fam = gdl_gview_wrapper_get_accession (wrapper2, gdl_mask_get_idx (fmask, GDL_ACCESSION, j));
				if (wtype2 == GDL_FACTOR) fam = gdl_fview_wrapper_get_accession (wrapper2, gdl_mask_get_idx (fmask, GDL_ACCESSION, j));
				
				if (!strcmp (gdl_entity_get_name (fam), gdl_entity_get_name (fa)))
				{
					break;
				}
			}
			if (j < nf)
			{
				fidx[i] = gdl_entity_get_idx (fam);
				nc++;
			}
			else
			{
				gdl_mask_remove (gmask, GDL_ACCESSION, i);
				i--;
			}
		}
		else
		{
			gdl_mask_remove (gmask, GDL_ACCESSION, i);
			i--;
		}
	}
	
	gdl_mask_reset (fmask, GDL_ACCESSION);
	
	for (j = i = 0; i < ng; i++)
	{
		if (fidx[i]!=-1)
		{
			gdl_mask_add_idx (fmask, GDL_ACCESSION, fidx[i]);
			j++;
		}
	}
}
