/*
  This file contains the implementation of the BIOSEG PostgreSQL type.  The
  implementation is a copy of the SEG type from the contrib directory in the
  PostgreSQL source, which has been changed to represent ranges of integers.
  The anticipated use of this type is to represent contiguous intervals of a
  biological sequence.

  SEG code written Gene Selkov, Jr.

  Changes for BIOSEG written by Kim Rutherford <kmr@xenu.org.uk>.
*/

#include "postgres.h"

#include <stdlib.h>
#include <errno.h>
#include <limits.h>

#include "access/gist.h"
#include "access/skey.h"
#include "utils/builtins.h"

#include "biosegdata.h"

#ifdef INTERBASE_COORDS
#define BIOSEG bioseg0
#define BIOSEG_PREFIX(x) bioseg0 ## x
#else
#define BIOSEG bioseg
#define BIOSEG_PREFIX(x) bioseg ## x
#endif

PG_MODULE_MAGIC;

extern int  BIOSEG_PREFIX(_yyparse)();
extern void BIOSEG_PREFIX(_yyerror)(const char *message);
extern void BIOSEG_PREFIX(_scanner_init)(const char *str);
extern void BIOSEG_PREFIX(_scanner_finish)(void);

/*
** Input/Output routines
*/
SEG   *BIOSEG_PREFIX(_in)(char *str);
char  *BIOSEG_PREFIX(_out)(SEG * seg);
Datum  BIOSEG_PREFIX(_create)(PG_FUNCTION_ARGS);
int32  BIOSEG_PREFIX(_lower)(SEG * seg);
int32  BIOSEG_PREFIX(_upper)(SEG * seg);
int32  BIOSEG_PREFIX(_center)(SEG * seg);

/*
** GiST support methods
*/
Datum          BIOSEG_PREFIX(_gist_consistent)(PG_FUNCTION_ARGS);
Datum          BIOSEG_PREFIX(_gist_compress)(PG_FUNCTION_ARGS);
Datum          BIOSEG_PREFIX(_gist_decompress)(PG_FUNCTION_ARGS);
Datum          BIOSEG_PREFIX(_gist_penalty)(PG_FUNCTION_ARGS);
Datum          BIOSEG_PREFIX(_gist_picksplit)(PG_FUNCTION_ARGS);
Datum          BIOSEG_PREFIX(_gist_union)(PG_FUNCTION_ARGS);
Datum          BIOSEG_PREFIX(_gist_same)(PG_FUNCTION_ARGS);


/*
** R-tree support functions
*/
Datum  BIOSEG_PREFIX(_same)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_contains_int)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_contains)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_contained)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_overlap)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_left)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_over_left)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_right)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_over_right)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_rt_size)(PG_FUNCTION_ARGS);
Datum  BIOSEG_PREFIX(_size)(PG_FUNCTION_ARGS);

static SEG *BIOSEG_PREFIX(_union)(SEG * a, SEG * b);

/*
** Various operators
*/
Datum BIOSEG_PREFIX(_cmp)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_lt)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_le)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_gt)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_ge)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_different)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_joinsel)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_sel)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_contsel)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_contjoinsel)(PG_FUNCTION_ARGS);


static int get_dots(char **str);
static int get_int(char **str, int32 *result);


/*****************************************************************************
 * Input/Output functions
 *****************************************************************************/

SEG *
  BIOSEG_PREFIX(_in)(char *str)
{
  int32 lower;
  int32 upper;

  if (!get_int(&str, &lower)) {
    return NULL;
  }

  if (str[0] == 0) {
    upper = lower;
  } else {
    if (!get_dots(&str)) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("number followed by something other than ..: %s", str)));
      return NULL;
    }

    if (!get_int(&str, &upper)) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("number\"..\" followed by something other than a number: %s", str)));
      return NULL;
    }

    if (lower > upper) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("lower limit of range greater than upper")));
      return NULL;
    }

    if (str[0] != 0) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("garbage at end of string: %s", str)));
      return NULL;
    }
  }

  {
    SEG *result = palloc(sizeof(SEG));

    result->lower = lower;
    result->upper = upper;
    return result;
  }
}

#ifdef INTERBASE_COORDS
#define MIN_LOWER INT_MIN
#else
#define MIN_LOWER 1l
#endif

int get_int(char **strp, int32 *result) {
  char *return_pos;
  long int long_result = -1;

  if (!(*strp)[0]) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("end of string found when expecting an integer")));
    return 0;
  }

  errno = 0;

  long_result = strtol(*strp, &return_pos, 0);

  if (errno == ERANGE && (long_result == LONG_MAX || long_result == LONG_MIN)) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("integer at: %s is out of range", *strp)));
    return 0;
  }

  if (errno != 0 && long_result == 0) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("unable to read an integer: %s", strerror(errno))));
    return 0;
  }

  if (*strp == return_pos) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("no integer found at: %s", *strp)));
    return 0;
  }

  if (long_result < MIN_LOWER) {
    // when INTERBASE_COORDS is set, on machines where
    // sizeof(long int) == sizeof(int32) this block won't be executed because
    // strtol() will return ERANGE
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("integer %ld at: %s is out of range - must be >= %ld",
                       long_result, *strp, MIN_LOWER)));
    return 0;
  }

  if (long_result > INT_MAX) {
    // on machines where sizeof(long int) == sizeof(int32) this won't happen
    // because strtol() will return ERANGE
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("integer %ld at: %s is out of range - must be <= %d",
                       long_result, *strp, INT_MAX)));
    return 0;
  }

  *strp = return_pos;
  *result = long_result;

  return 1;
}

int
  get_dots(char **strp)
{
  if ((*strp)[0] == '.') {
    (*strp)++;
    if ((*strp)[0] == '.') {
      (*strp)++;
      if ((*strp)[0] == '.') {
        // allow for "10...20"
        (*strp)++;
      }
      return 1;
    } else {
      return 0;
    }
  } else {
    return 0;
  }
}

char *
  BIOSEG_PREFIX(_out)(SEG * bioseg)
{
  char *result;
  char *p;

  if (bioseg == NULL)
    return (NULL);

  p = result = (char *) palloc(40);

  if (bioseg->lower == bioseg->upper)
    {
      /*
       * indicates that this interval was built by bioseg_in off a single point
       */
      sprintf(p, "%d", bioseg->lower);
    }
  else
    {
      sprintf(p, "%d..%d", bioseg->lower, bioseg->upper);
    }

  return (result);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_create));
Datum BIOSEG_PREFIX(_create)(PG_FUNCTION_ARGS)
{
  SEG *result = palloc(sizeof(SEG));
  int32 lower = PG_GETARG_INT32(0);
  int32 upper = PG_GETARG_INT32(1);

  if (lower > upper) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("lower limit of range greater than upper")));
    PG_RETURN_POINTER(0);
  }

  if (lower < MIN_LOWER) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("lower bound (%ld) must be >= %ld",
                       (long int) lower, MIN_LOWER)));
    PG_RETURN_POINTER(0);
  }

  result->lower = lower;
  result->upper = upper;

  PG_RETURN_POINTER(result);
}



int32
  BIOSEG_PREFIX(_center)(SEG * bioseg)
{
  return ((int32) bioseg->lower + (int32) bioseg->upper) / 2;
}

int32
  BIOSEG_PREFIX(_lower)(SEG * bioseg)
{
  return bioseg->lower;
}

int32
  BIOSEG_PREFIX(_upper)(SEG * bioseg)
{
  return bioseg->upper;
}


/*****************************************************************************
 * GiST functions
 *****************************************************************************/

/*
** The GiST Consistent method for biosegments
** Should return false if for all data items x below entry,
** the predicate x op query == FALSE, where op is the oper
** corresponding to strategy in the pg_amop table.
*/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gist_consistent));
Datum BIOSEG_PREFIX(_gist_consistent)(PG_FUNCTION_ARGS)
{
  GISTENTRY *entry = (GISTENTRY *) PG_GETARG_POINTER(0);
  SEG *query = (SEG *) PG_GETARG_POINTER(1);
  StrategyNumber strategy = (StrategyNumber) PG_GETARG_UINT16(2);
  SEG *key = (SEG *) DatumGetPointer(entry->key);

  if (GIST_LEAF(entry)) {
    switch (strategy)
    {
      case RTLeftStrategyNumber:
        PG_RETURN_BOOL(key->upper < query->lower);
        break;
      case RTOverLeftStrategyNumber:
        PG_RETURN_BOOL(key->upper <= query->upper);
        break;
      case RTOverlapStrategyNumber:
#ifdef INTERBASE_COORDS
        PG_RETURN_BOOL((key->upper > query->lower) && (key->lower < query->upper));
#else
        PG_RETURN_BOOL((key->upper >= query->lower) && (key->lower <= query->upper));
#endif
        break;
      case RTOverRightStrategyNumber:
        PG_RETURN_BOOL(key->lower >= query->lower);
        break;
      case RTRightStrategyNumber:
        PG_RETURN_BOOL(key->lower > query->upper);
        break;
      case RTSameStrategyNumber:
        PG_RETURN_BOOL((key->lower == query->lower) && (key->upper == query->upper));
        break;
      case RTContainsStrategyNumber:
      case RTOldContainsStrategyNumber:
        PG_RETURN_BOOL((key->lower <= query->lower) && (key->upper >= query->upper));
        break;
      case RTContainedByStrategyNumber:
      case RTOldContainedByStrategyNumber:
        PG_RETURN_BOOL((key->lower >= query->lower) && (key->upper <= query->upper));
        break;
    }
  } else {
    switch (strategy)
    {
      case RTLeftStrategyNumber:
        PG_RETURN_BOOL(key->lower < query->lower);
        break;
      case RTOverLeftStrategyNumber:
        PG_RETURN_BOOL(key->lower <= query->upper);
        break;
      case RTOverlapStrategyNumber:
#ifdef INTERBASE_COORDS
        PG_RETURN_BOOL((key->upper > query->lower) && (key->lower < query->upper));
#else
        PG_RETURN_BOOL((key->upper >= query->lower) && (key->lower <= query->upper));
#endif
        break;
      case RTOverRightStrategyNumber:
        PG_RETURN_BOOL(key->upper >= query->lower);
        break;
      case RTRightStrategyNumber:
        PG_RETURN_BOOL(key->upper > query->upper);
        break;
      case RTSameStrategyNumber:
      case RTContainsStrategyNumber:
      case RTOldContainsStrategyNumber:
        PG_RETURN_BOOL((key->lower <= query->lower) && (key->upper >= query->upper));
        break;
      case RTContainedByStrategyNumber:
      case RTOldContainedByStrategyNumber:
#ifdef INTERBASE_COORDS
        PG_RETURN_BOOL((key->upper > query->lower) && (key->lower < query->upper));
#else
        PG_RETURN_BOOL((key->upper >= query->lower) && (key->lower <= query->upper));
#endif
        break;
    }
  }
  PG_RETURN_BOOL(FALSE);
}

/*
** The GiST Union method for biosegments
** returns the minimal bounding bioseg that encloses all the entries in entryvec
*/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gist_union));
Datum BIOSEG_PREFIX(_gist_union)(PG_FUNCTION_ARGS)
{
  GistEntryVector *entryvec = (GistEntryVector *) PG_GETARG_POINTER(0);
  int  numranges;
  int  i;
  SEG *out = (SEG *) NULL;
  SEG *tmp;

#ifdef GIST_DEBUG
  fprintf(stderr, "union\n");
#endif

  out = (SEG *) palloc(sizeof(*out));
  tmp = (SEG *) entryvec->vector[0].key;
  out->lower = tmp->lower;
  out->upper = tmp->upper;
  numranges = entryvec->n;

  for (i = 1; i < numranges; i++) {
    tmp = (SEG *) entryvec->vector[i].key;
    if (tmp->lower < out->lower) {
      out->lower = tmp->lower;
    }
    if (tmp->upper > out->upper) {
      out->upper = tmp->upper;
    }
  }

  PG_RETURN_POINTER(out);
}

/*
** GiST Compress and Decompress methods for biosegments
** do not do anything.
*/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gist_compress));
Datum BIOSEG_PREFIX(_gist_compress)(PG_FUNCTION_ARGS)
{
  GISTENTRY *entry = (GISTENTRY *) PG_GETARG_POINTER(0);
  PG_RETURN_POINTER(entry);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gist_decompress));
Datum BIOSEG_PREFIX(_gist_decompress)(PG_FUNCTION_ARGS)
{
  GISTENTRY *entry = (GISTENTRY *) PG_GETARG_POINTER(0);
  PG_RETURN_POINTER(entry);
}

/*
** The GiST Penalty method for biosegments
** As in the R-tree paper, we use change in area as our penalty metric
*/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gist_penalty));
Datum BIOSEG_PREFIX(_gist_penalty)(PG_FUNCTION_ARGS)
{
  GISTENTRY *origentry = (GISTENTRY *) PG_GETARG_POINTER(0);
  GISTENTRY *newentry = (GISTENTRY *) PG_GETARG_POINTER(1);
  float *result = (float *) PG_GETARG_POINTER(2);
  SEG   *ud;
  SEG   *origrange;
  SEG   *newrange;
  int32 tmp1;
  int32 tmp2;

  origrange = (SEG *) DatumGetPointer(origentry->key);
  newrange = (SEG *) DatumGetPointer(newentry->key);
  ud = BIOSEG_PREFIX(_union)(origrange, newrange);
  tmp1 = ud->upper - ud->lower;
  tmp2 = origrange->upper - origrange->lower;
  if (tmp1 == tmp2)
    {
      tmp1 = newrange->upper - newrange->lower;
      *result = ((tmp2 - tmp1) * 100000.0) / tmp2;
    }
  else
    {
      *result = tmp1 - tmp2 + 100000.0;
    }

#ifdef GIST_DEBUG
  fprintf(stderr, "Penalty of putting %d..%d into %d..%d is %g\n", newrange->lower, newrange->upper, origrange->lower, origrange->upper, *result);
#endif

  PG_RETURN_POINTER(result);
}


/*
** The GiST PickSplit method for segments
*/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gist_picksplit));
Datum BIOSEG_PREFIX(_gist_picksplit)(PG_FUNCTION_ARGS)
{
  GistEntryVector *entryvec = (GistEntryVector *) PG_GETARG_POINTER(0);
  GIST_SPLITVEC *v = (GIST_SPLITVEC *) PG_GETARG_POINTER(1);
  OffsetNumber  i;
  SEG          *datum_alpha;
  SEG          *datum_l;
  SEG          *datum_r;
  int32         lowest;
  int32         highest;
  int32         midpoint;
  int32         split;
  int           nbytes;
  int           firsttime;
  long long int midpointsum;
  int           sumcount;
  OffsetNumber *left;
  OffsetNumber *right;
  OffsetNumber  maxoff;

  maxoff = entryvec->n - 1;
  nbytes = (maxoff + 2) * sizeof(OffsetNumber);
  v->spl_left = (OffsetNumber *) palloc(nbytes);
  v->spl_right = (OffsetNumber *) palloc(nbytes);

  midpointsum = 0;
  sumcount = 0;
  firsttime = true;
  lowest = 0;
  highest = 0;

  for (i = FirstOffsetNumber; i <= maxoff; i = OffsetNumberNext(i))
    {
      datum_alpha = (SEG *) DatumGetPointer(entryvec->vector[i].key);
      midpoint = datum_alpha->lower + ((datum_alpha->upper - datum_alpha->lower) / 2);
      midpointsum += midpoint;
      sumcount++;
      if (firsttime || (midpoint < lowest))
        {
          lowest = midpoint;
        }
      if (firsttime || (midpoint > highest))
        {
          highest = midpoint;
        }
      firsttime = false;
    }

  split = midpointsum / sumcount;
  left = v->spl_left;
  v->spl_nleft = 0;
  right = v->spl_right;
  v->spl_nright = 0;

  datum_l = (SEG *) palloc(sizeof(*datum_l));
  datum_l->lower = lowest;
  datum_l->upper = lowest;
  datum_r = (SEG *) palloc(sizeof(*datum_r));
  datum_r->lower = highest;
  datum_r->upper = highest;

  /*
   * Now split up the regions between the two seeds.
   */

  for (i = FirstOffsetNumber; i <= maxoff; i = OffsetNumberNext(i))
    {
      datum_alpha = (SEG *) DatumGetPointer(entryvec->vector[i].key);
      midpoint = datum_alpha->lower + ((datum_alpha->upper - datum_alpha->lower) / 2);

      /* pick which page to add it to */
      if (midpoint <= split)
        {
          datum_l = BIOSEG_PREFIX(_union)(datum_l, datum_alpha);
          *left++ = i;
          v->spl_nleft++;
        }
      else
        {
          datum_r = BIOSEG_PREFIX(_union)(datum_r, datum_alpha);
          *right++ = i;
          v->spl_nright++;
        }
    }
  *left = *right = FirstOffsetNumber; /* sentinel value, see dosplit() */

  v->spl_ldatum = PointerGetDatum(datum_l);
  v->spl_rdatum = PointerGetDatum(datum_r);

#ifdef GIST_DEBUG
  fprintf(stderr, "Picksplit summary to %d..%d (%d locations) and %d..%d (%d locations), with pivot %d\n", datum_l->lower, datum_l->upper, v->spl_nleft,
datum_r->lower, datum_r->upper, v->spl_nright, split);
#endif

  PG_RETURN_POINTER(v);
}

/*
** Equality methods
*/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gist_same));
Datum BIOSEG_PREFIX(_gist_same)(PG_FUNCTION_ARGS)
{
  SEG *b1 = (SEG *) PG_GETARG_POINTER(0);
  SEG *b2 = (SEG *) PG_GETARG_POINTER(1);
  bool *result = (bool *) PG_GETARG_POINTER(2);

  *result = ((b1->lower == b2->lower) && (b1->upper == b2->upper));

#ifdef GIST_DEBUG
  fprintf(stderr, "same: %s\n", (*result ? "TRUE" : "FALSE"));
#endif

  PG_RETURN_POINTER(result);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_contains));
Datum BIOSEG_PREFIX(_contains)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  
  PG_RETURN_BOOL((a->lower <= b->lower) && (a->upper >= b->upper));
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_contained));
Datum BIOSEG_PREFIX(_contained)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  
  PG_RETURN_BOOL((a->lower >= b->lower) && (a->upper <= b->upper));
}

/*****************************************************************************
 * Operator class for R-tree indexing
 *****************************************************************************/

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_same));
Datum BIOSEG_PREFIX(_same)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);

  PG_RETURN_BOOL((a->lower == b->lower) && (a->upper == b->upper));
}

/*      bioseg_overlap -- does a overlap b?
 */
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_overlap));
Datum BIOSEG_PREFIX(_overlap)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
#ifdef INTERBASE_COORDS
  PG_RETURN_BOOL((a->upper > b->lower) && (a->lower < b->upper));
#else
  PG_RETURN_BOOL((a->upper >= b->lower) && (a->lower <= b->upper));
#endif
}

/*      bioseg_overleft -- is the right edge of (a) located at or left of the right edge of (b)?
 */
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_over_left));
Datum BIOSEG_PREFIX(_over_left)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL(a->upper <= b->upper);
}

/*      bioseg_left -- is (a) entirely on the left of (b)?
 */
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_left));
Datum BIOSEG_PREFIX(_left)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL(a->upper < b->lower);
}

/*      bioseg_right -- is (a) entirely on the right of (b)?
 */
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_right));
Datum BIOSEG_PREFIX(_right)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL(a->lower > b->upper);
}

/*      bioseg_overright -- is the left edge of (a) located at or right of the left edge of (b)?
 */
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_over_right));
Datum BIOSEG_PREFIX(_over_right)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL(a->lower >= b->lower);
}


static SEG *
  BIOSEG_PREFIX(_union)(SEG * a, SEG * b)
{
  SEG *n;

  n = (SEG *) palloc(sizeof(*n));

  /* take max of upper endpoints */
  if (a->upper > b->upper)
    {
      n->upper = a->upper;
    }
  else
    {
      n->upper = b->upper;
    }

  /* take min of lower endpoints */
  if (a->lower < b->lower)
    {
      n->lower = a->lower;
    }
  else
    {
      n->lower = b->lower;
    }

  return (n);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_rt_size));
Datum BIOSEG_PREFIX(_rt_size)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  int32 *size = (int32 *) PG_GETARG_POINTER(1);
  if (a == (SEG *) NULL)
    *size = 0;
  else
#ifdef INTERBASE_COORDS
    *size = (int32) (a->upper - a->lower);
#else
    *size = (int32) (a->upper - a->lower + 1);
#endif
  PG_RETURN_NULL();
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_size));
Datum BIOSEG_PREFIX(_size)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
#ifdef INTERBASE_COORDS
  PG_RETURN_INT32(a->upper - a->lower);
#else
  PG_RETURN_INT32(a->upper - a->lower + 1);
#endif
}


/*****************************************************************************
 *                                 Miscellaneous operators
 *****************************************************************************/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_cmp));
Datum BIOSEG_PREFIX(_cmp)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  if (a->lower < b->lower)
    PG_RETURN_INT32(-1);
  if (a->lower > b->lower)
    PG_RETURN_INT32(1);

  if (a->upper < b->upper)
    PG_RETURN_INT32(-1);
  if (a->upper > b->upper)
    PG_RETURN_INT32(1);

  PG_RETURN_INT32(0);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_lt));
Datum BIOSEG_PREFIX(_lt)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL((a->lower < b->lower) || ((a->lower == b->lower) && (a->upper < b->upper)));
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_le));
Datum BIOSEG_PREFIX(_le)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL((a->lower < b->lower) || ((a->lower == b->lower) && (a->upper <= b->upper)));
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_gt));
Datum BIOSEG_PREFIX(_gt)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL((a->lower > b->lower) || ((a->lower == b->lower) && (a->upper > b->upper)));
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_ge));
Datum BIOSEG_PREFIX(_ge)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL((a->lower > b->lower) || ((a->lower == b->lower) && (a->upper >= b->upper)));
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_different));
Datum BIOSEG_PREFIX(_different)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  SEG *b = (SEG *) PG_GETARG_POINTER(1);
  PG_RETURN_BOOL((a->lower != b->lower) || (a->upper != b->upper));
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_contains_int));
Datum BIOSEG_PREFIX(_contains_int)(PG_FUNCTION_ARGS)
{
  SEG *a = (SEG *) PG_GETARG_POINTER(0);
  int *b = (int *) PG_GETARG_POINTER(1);
#ifdef INTERBASE_COORDS
  PG_RETURN_BOOL((a->lower <= *b) && (a->upper > *b));
#else
  PG_RETURN_BOOL((a->lower <= *b) && (a->upper >= *b));
#endif
}

/*
** These are lower than those in geo_selfuncs.c - found by trail and error.
*/
PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_sel));
Datum
  BIOSEG_PREFIX(_sel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(2.0e-4);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_joinsel));
Datum
  BIOSEG_PREFIX(_joinsel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(1.0e-5);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_contsel));
Datum
  BIOSEG_PREFIX(_contsel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(1.0e-4);
}

PG_FUNCTION_INFO_V1(BIOSEG_PREFIX(_contjoinsel));
Datum
  BIOSEG_PREFIX(_contjoinsel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(5e-6);
}
