%
% t-CWT Copyright (C) 2003-2015 Vladimir Bostanov. Type "tcwt_help".
%

% This file is part of t-CWT.
%
% t-CWT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation, either version 3 of the License,
% or (at your option) any later version.
%
% t-CWT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty
% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with t-CWT; see the file COPYING or the file tcwt_gpl.m
% If not, see <http://www.gnu.org/licenses/>.

%: tcwt_f2stats(is1out,DSNL1,DSNL2,statsNAME,Cnd,aPriori) performs
%: LDA classification of single trials from test datasets by LDFs
%: obtained from training datasets <tcwt_help x2ld>.
%:
%: The function iterates over DSNL2 reading and processing data from
%: the INPUT FILES and then saves the results into the OUTPUT FILES.
%:
%: FUNCTION ARGUMENTS
%:
%:   is1out  =  0 or 1; defines whether 1out indexes are used (see below)
%:
%:   DSNL1      Training DataSet Name List (DSN1 List) <tcwt_help dir2dsnl>
%:
%:   DSNL2      Test DataSet Name List (DSN2 List)
%:
%:   statsNAME  Name of the OUTPUT FILES (without the extensions)
%:
%:   Cnd        List of exactly 2 experimental conditions <tcwt_help t2f>
%:
%:   aPriori  = 0 or 1; defines whether the a priori error rate is computed
%:              from the numbers of trials in Cnd(1) and Cnd(2).
%:              If aPriori == 0, the a priori error rate and the a priori
%:              hit rate are taken to be equal (p1=p2=0.5) and log(p2/p1)=0.
%:
%: INPUT FILES              INPUT VARIABLES
%:
%:   ./DSN2.f.mat            Verp,CIerp <tcwt_help t2f>
%:   ./DSN2.ri0.mat          RIerp, DSNLpool <tcwt_help f2pool>
%:
%:  If is1out == 0
%:
%:   ./DSN1.ld.mat           Nxtr,sdTerp,ldTerp,ldMean,CIerp <tcwt_help x2ld>
%:
%:  If is1out == 1, and DSNL1 has only one row, DSN1
%:
%:   ./DSN2.out.DSN1.ld.mat  Nxtr,sdTerp,ldTerp,ldMean,CIerp <tcwt_help x2ld>
%:
%: OUTPUT FILES                 OUTPUT VARIABLES
%:
%:   ./statsNAME.stats0.mat      dsnl = DSNL2,
%:      (if aPriori == 0)        NtCWTextr,NsdPC,Photelling
%:   ./statsNAME.statsA.mat      Ntrials,Nerrors,ErrP100,Pbinomi
%:      (if aPriori == 1)        NtrialsSum,NerrorsSum,ErrP100Sum,PbinomiSum
%:
%:   ./statsNAME.stats0.txt      Text output
%:   ./statsNAME.statsA.txt
%:
%:  If DSNL1 == DSNL2
%:
%:   ./statsNAME.biased.stats0.mat
%:   ./statsNAME.biased.stats0.txt
%:
%:   ./statsNAME.biased.statsA.mat
%:   ./statsNAME.biased.statsA.txt
%:
%:  If is1out == 1, and DSNL1 has only one row
%:
%:   ./statsNAME.1dataset.out.stats0.mat
%:   ./statsNAME.1dataset.out.stats0.txt
%:
%:   ./statsNAME.1dataset.out.statsA.mat
%:   ./statsNAME.1dataset.out.statsA.txt
%:
%: OUTPUT VARIABLES
%:
%:   NtCWTextr    Number of t-CWT features for each training dataset
%:                (DSN1.ld.mat or DSN2.out.DSN1.ld.mat, see INPUT FILES)
%:
%:   NsdPC        Number of step-down-selected components
%:                in each training dataset
%:
%:   Photelling   P-Value of Hotelling's test performed on each test
%:                dataset DSN2 using the step-down-selected components
%:                from the corresponding training dataset
%:
%:   Ntrials      Nubmer of trials, for each condition and total,
%:                in each DSN2. Ntrials(:,1) corresponds to Cnd(1)
%:                Ntrials(:,2) corresponds to Cnd(2), and Ntrials(:,3)
%:                is the total nubmer of trials
%:
%:   Nerrors      Number of errors of LDA classification, for Cnd(1), Cnd(2),
%:                and total, for each DSN2. Cnd(1) --> Nerrors(:,1), etc.
%:
%:   ErrP100  =   100 * Nerrors / Ntrials    (error percentages)
%:
%:   Pbinomi      P-Values of the binomial distribution testing the hypotheses
%:                that the LDA classifications were better than chance,
%:                i.e., that ErrP100 < 50%
%:
%:   NtrialsSum = sum(Ntrials)
%:
%:   NerrorsSum = sum(Nerrors)
%:
%:   ErrP100Sum = 100 * NerrorsSum / NtrialsSum   (average error percentages)
%:
%:   PbinomiSum   P-Values for testing whether ErrP100Sum < 50%

%       10        20        30        40        50        60        70        80

function tcwt_f2stats(is1out,dsnl_ld,dsnl,statsName,Cnd,aPriori);

rin='0';

Alpha=0.05;

if (is1out && size(dsnl_ld,1)==1)
 dsnl_ld = tcwt_dsnl2xdsnl('',dsnl,['.out.' dsnl_ld]);
 statsName = [statsName '.1dataset.out.stats'];
elseif (all(size(dsnl_ld)==size(dsnl)) && all(all(dsnl_ld==dsnl)))
 statsName = [statsName '.biased.stats'];
else
 statsName = [statsName '.stats'];
end

Nds = size(dsnl,1);

Ntrials = zeros(Nds,3);
Nerrors = zeros(Nds,3);
ErrP100 = zeros(Nds,3);
Pbinomi = zeros(Nds,1);

NtCWTextr = zeros(Nds,1);
NsdPC = zeros(Nds,1);
Photelling = zeros(Nds,1);

statSig = ' *';

spaceS = repmat(' ',1,size(dsnl,2));
tildeS = repmat('~',1,size(dsnl,2));
minusS = repmat('-',1,size(dsnl,2));
equalS = repmat('=',1,size(dsnl,2));
ulineS = repmat('_',1,size(dsnl,2));

if aPriori
 aPrioriS = 'A priori error rate computed from the number of trials in each condition';
 statsName = [statsName 'A'];
else
 aPrioriS = 'A priori error rate and a priori hit rate taken to be equal: p1=p2=0.5';
 statsName = [statsName '0'];
end

O=tcwt_txt('\n');
O=[O,tcwt_txt('%s~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n',tildeS)];
O=[O,tcwt_txt('f2stats: Hotelling & LDA error rates: %s\n',statsName)];
O=[O,tcwt_txt('%s----------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt('%s\n',aPrioriS)];
O=[O,tcwt_txt('%s----------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt('%s tCWT  HOTELLING   COND.1  COND.2  TOTAL = COND.1 + COND.2\n',spaceS)];
O=[O,tcwt_txt('%s extr  PC  p-      Error   Error   Error   A      binomial\n',spaceS)];
O=[O,tcwt_txt('%s    N   N  value   Rate(%%) Rate(%%) Rate(%%) Priori  p-value\n',spaceS)];
%              ?? 1234  12  123.1*  123.1   123.1   123.1   12.1    1.1234*
formatS =     '%s %4u  %2u  %6.4f%1s %5.1f   %5.1f   %5.1f   %4.1f%%   %6.4f%1s\n';
O=[O,tcwt_txt('%s----------------------------------------------------------------------\n',minusS)];

for n=1:Nds
 dsn_ld=dsnl_ld(n,:);
 dsn=dsnl(n,:);
 load([dsn_ld '.ld.mat']);
 load([dsn '.f.mat']);
 load([dsn '.ri' rin '.mat']);

 Sout = logical(RIerp(:,1));
 Scnd1 = RIerp(:,2)==Cnd(1);
 Scnd2 = RIerp(:,2)==Cnd(2);

 if aPriori
  Nt1 = sum(Scnd1); Nt2 = sum(Scnd2);
  p1 = Nt1/(Nt1+Nt2);
 else
  p1 = 0.5;
 end
 p2 = 1 - p1;
 logRatio = log(p2/p1);
 pApriori = min(p1,p2);

 Error1 = Verp(~Sout & Scnd1,:)*ldTerp - ldMean < logRatio;
 Error2 = Verp(~Sout & Scnd2,:)*ldTerp - ldMean >= logRatio;

 Nt1 = length(Error1); Nt2 = length(Error2);
 Ne1  = sum(Error1); Ne2  = sum(Error2);

 Nx = Nxtr;
 Ns = size(sdTerp,2);
 Ph = tcwt_hotelling(Verp(~Sout & Scnd1,:)*sdTerp,Verp(~Sout & Scnd2,:)*sdTerp);
 Sh = statSig([Ph<Alpha]+1);

 Nt = [Nt1, Nt2, Nt1+Nt2];
 Ne = [Ne1, Ne2, Ne1+Ne2];
 Ep = 100*Ne./Nt;
 Pb = binocdf(Ne(3), Nt(3), pApriori);

 Sb = statSig([Pb<Alpha]+1);

 O=[O,tcwt_txt(formatS, dsn, Nx,Ns,Ph,Sh,Ep(1),Ep(2),Ep(3),100*pApriori,Pb,Sb)];

 NtCWTextr(n) = Nx;
 NsdPC(n) = Ns;
 Photelling(n) = Ph;

 Ntrials(n,:) = Nt;
 Nerrors(n,:) = Ne;
 ErrP100(n,:) = Ep;
 Pbinomi(n) = Pb;
end

dsn = spaceS;

Nx = NaN;
Ns = 0;
Ph = NaN;
Sh = ' ';

if ( Nds > 1 )
 Nt = sum(Ntrials);
 Ne = sum(Nerrors);
else
 Nt = Ntrials;
 Ne = Nerrors;
end
Ep = 100*Ne./Nt;
Pb = binocdf(Ne(3), Nt(3), pApriori);

Sb = statSig([Pb<Alpha]+1);

O=[O,tcwt_txt('%s----------------------------------------------------------------------\n',minusS)];
O=[O,tcwt_txt(formatS, dsn, Nx,Ns,Ph,Sh,Ep(1),Ep(2),Ep(3),100*pApriori,Pb,Sb)];
O=[O,tcwt_txt('%s======================================================================\n',equalS)];

NtrialsSum = Nt;
NerrorsSum = Ne;
ErrP100Sum = Ep;
PbinomiSum = Pb;

save([statsName '.mat'], 'dsnl', ...
 'NtCWTextr','NsdPC','Photelling', ...
 'Ntrials','Nerrors','ErrP100','Pbinomi', ...
 'NtrialsSum','NerrorsSum','ErrP100Sum','PbinomiSum');
tcwt_txtsave([statsName '.txt'],O);

O=[O,tcwt_txt('tcwt_f2stats: Results saved to: %s.mat\n',statsName)];
O=[O,tcwt_txt('tcwt_f2stats:   Text output to: %s.txt\n',statsName)];
O=[O,tcwt_txt('%s______________________________________________________________________\n',ulineS)];
