package edu.hust.go.model;

/**
 * <p>Title: GO4J</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2005</p>
 * <p>Company: </p>
 * @author GQ Zhang
 * @version 1.0
 */
import java.io.File;
import org._3pq.jgrapht.graph.SimpleDirectedGraph;
import java.util.*;

import edu.hust.go.model.GoParser.OntologyDefineType;
import edu.hust.go.term.GO_term;
import edu.hust.go.term.OBOGo_term;

/**
 * A graph model is constructed by the connection in GO terms. It can be used to construct GUI by
 * edu.hust.go.gui.GraphCreator.
 */

public class OntologyGraphModel {
	// private static final String[] DEFAULT_ROOT_KEYS={ "0008150", "0005575", "0003674" };
	private HashSet<GO_term> obsoleteSet;

	private SimpleDirectedGraph graph;

	private HashMap<GO_term, Integer> nodeChildNumberMap;

	/**
	 * @param map
	 *            the key is Go ids, the value is GO terms.
	 * @throws InvalidGoDefinitionException
	 *             there are problems in GO definition file, such as one id is connected to another id, while the latter
	 *             is not exist in fact. Please make sure that the definition file comes from GO official site: <a
	 *             href="http://www.geneontology.org">www.geneontology.org</a> or <a
	 *             href="ftp://ftp.geneontology.org">ftp.geneontology.org</a>.
	 */
	public OntologyGraphModel(HashMap<String, GO_term> map) throws GoException {
		graph = init(map);
		obsoleteSet = removeObsolete();
		nodeChildNumberMap = new HashMap<GO_term, Integer>();
		for (Iterator iter = graph.vertexSet().iterator(); iter.hasNext();) {
			Object vertex = iter.next();
			if(vertex instanceof GO_term)
			nodeChildNumberMap.put((GO_term)vertex, new Integer(GraphPath.getLowerSubNodes(graph, vertex).size()));
		}
	}

	/**
	 * organize all go ids in goMap as a graph
	 * 
	 * @param goMap
	 * @return
	 * @throws InvalidGoDefinitionException
	 */
	private SimpleDirectedGraph init(HashMap<String, GO_term> goMap) throws GoException {
		SimpleDirectedGraph g = new SimpleDirectedGraph();
		GO_term targetTerm = null;
		GO_term sourceTerm = null;
		try {
			GO_term[] nodes = goMap.values().toArray(new GO_term[0]);
			for (int i = 0; i < nodes.length; i++) {
				g.addVertex(nodes[i]);
			}
			for (int i = 0; i < nodes.length; i++) {
				targetTerm = nodes[i];
				Vector<String> parents = targetTerm.getIs_a();
				if (parents != null) {
					for (int j = 0; j < parents.size(); j++) {
						sourceTerm = goMap.get(parents.get(j));
						g.addEdge(new OntologyEdge(sourceTerm, targetTerm, "is_a"));
					}
				}
				parents = targetTerm.getPart_of();
				if (parents != null) {
					for (int j = 0; j < parents.size(); j++) {
						sourceTerm = (GO_term) goMap.get(parents.get(j));
						g.addEdge(new OntologyEdge(sourceTerm, targetTerm, "part_of"));
					}
				}
			}
		} catch (Exception e) {
			System.out.println("The problems exists in!" + targetTerm + "**" + sourceTerm + "\n" + e);
			GoException exc = new GoException(sourceTerm, targetTerm);
			g = null;
			throw exc;
		}
		return g;
	}

	/**
	 * remove obsolete go ids in GoMap
	 * 
	 * @return all obsolte go ids
	 */
	private HashSet<GO_term> removeObsolete() {
		HashSet<GO_term> _obsoleteSet = new HashSet<GO_term>();
		Object[] vertexs = graph.vertexSet().toArray();
		for (int i = 0; i < vertexs.length; i++) {
			if (graph.inDegreeOf(vertexs[i]) == 0 && graph.outDegreeOf(vertexs[i]) == 0) {
				if (vertexs[i] instanceof GO_term)
					_obsoleteSet.add((GO_term) vertexs[i]);
				graph.removeVertex(vertexs[i]);
			}
		}
		return _obsoleteSet;
	}

	/**
	 * extract sub graph from GO graph, the sub graph contains a set of target go terms , and related in the pathway
	 * from the root to its leaf nodes, for example, one pathway is A->B->C->Target->D->E->F, while A is root and F is
	 * leaf, and suppose parentLevel and childLevel are both 1, then the nodes C, Taraget, and D are extracted intio the
	 * subGraph, the the related edges are extracted into the subGraph too.
	 * 
	 * @param goTerms
	 *            query terms
	 * @param parentLevel
	 *            the depth from one target term to root
	 * @param childLevel
	 *            the depth from one target term to leaf
	 * @return the extracted sub graph
	 */
	public SimpleDirectedGraph buildSubGraph(ArrayList<Object> goTerms, int parentLevel, int childLevel) {
		SimpleDirectedGraph subGraph = new SimpleDirectedGraph();
		if (parentLevel == 0) {
			getAllSubGraph(goTerms, subGraph, true);
		} else if (parentLevel > 0) {
			getPartSubGraph(goTerms, subGraph, parentLevel, true);
		}
		if (childLevel == 0) {
			getAllSubGraph(goTerms, subGraph, true);
		} else if (childLevel > 0) {
			getPartSubGraph(goTerms, subGraph, childLevel, false);
		}

		return subGraph;
	}

	/**
	 * get the common parents in the pathway of target go terms. Notice: the common parents in the pathways of one
	 * target node is not common parents of all target nodes.
	 * 
	 * @param goTerms
	 *            target go terms
	 * @return the common parents
	 */
	public HashSet<Object> getCommonParents(ArrayList goTerms) {
		HashMap<Object, Integer> map = new HashMap<Object, Integer>();
		HashSet<Object> commonSet = new HashSet<Object>();
		for (int i = 0; i < goTerms.size(); i++) {
			Object[][] nodePath = GraphPath.getPath2Root(graph, goTerms.get(i));
			GraphPath.getCommonElements(nodePath, map);
		}
		Object[] commonObjs = map.keySet().toArray();
		for (int i = 0; i < commonObjs.length; i++) {
			int value = map.get(commonObjs[i]).intValue();
			if (value > 1) {
				commonSet.add(commonObjs[i]);
			}
		}
		return commonSet;
	}

	/**
	 * @see #evalSimilarity(GO_term node1,GO_term node2, byte type)
	 */
	public double evalSimilarity(GO_term node1, GO_term node2) {
		return evalSimilarity(node1, node2, Similarity.CAO);
	}

	/**
	 * evaluate the similarity of two nodes in directed acylic graph
	 * 
	 * @param node1
	 *            one of the two nodes
	 * @param node2
	 *            one of the two nodes
	 * @param type
	 *            one of the following five values. <br/>Similarity.CAO: basen on information content
	 *            <br/>Similarity.LIN: basen on information content, see <a
	 *            href="http://citeseer.ist.psu.edu/cache/papers/cs/1049/ftp:zSzzSzftp.cs.umanitoba.cazSzpubzSzlindekzSzpaperszSzsim.pdf/an-information-theoretic-definition.pdf">
	 *            Lin D. An information-theoretic definition of similarity. In: Proceedings of 15th International
	 *            Conference on Machine Learning, San Francisco US, 1998, 296-304</a> <br/>Similarity.RESNIK: based on
	 *            information content, see <a
	 *            href="http://citeseer.ist.psu.edu/cache/papers/cs/2124/http:zSzzSzwww.cs.umbc.eduzSz891zSzresnik.pdf/resnik95using.pdf">
	 *            Resnik P. Using information content to evaluate semantic similarity in a taxonomy. In: Proceedings of
	 *            the 14th International Joint Conference on Artificial Intelligence, Montreal Canada, 1995, 448-453</a>
	 *            <br/>Similarity.ZZL: refined ZZL based on distance, the LCA(latest common ancestor) is one with
	 *            maximal information content in the of the same distance LCA <br/>Similarity.oldZZL: based on distance,
	 *            see Zhong <a href="http://apex.sjtu.edu.cn/docs/iccs2002.pdf">JW, Zhu HP, Li JM, Yu Y. Conceptual
	 *            Graph Matching for Semantic Search. In: Proceedings of the 10th International Conference on Conceptual
	 *            Structures: Integration and Interfaces, 2002, 92-196</a>
	 * @return the semanic similarity value
	 */
	public double evalSimilarity(GO_term node1, GO_term node2, byte type) {
		if (node1 == null || node2 == null) {
			return -1;
		}
		if (graph.containsVertex(node1) && graph.containsVertex(node2)) {
			return Similarity.evaluate(this, node1, node2, type);
		} else {
			return -1;
		}
	}

	/**
	 * @param node
	 *            the target node
	 * @return all pathways between the target node and its corresponing root. Each element is a GO term.
	 */
	public Object[][] getPathsOfNode2Root(GO_term node) {
		return GraphPath.getPath2Root(graph, node);
	}

	/**
	 * Tell whether two nodes are in one pathway
	 * 
	 * @param node1
	 *            one of the two nodes
	 * @param node2
	 *            one of the two nodes
	 * @return true means the two nodes are in one pathway, false means no.
	 */
	public boolean inOnePathway(GO_term node1, GO_term node2) {
		return GraphPath.related(graph, node1, node2);
	}

	/**
	 * get pathway between two Go terms
	 * 
	 * @param node1
	 * @param node2
	 * @return there is only one key and one value.<bf/> The key is a HashSet of common ancestors, while the value is a
	 *         Object[][] of common pathways.
	 */
	public HashMap getPathsBetweenNodes(GO_term node1, GO_term node2) {
		return GraphPath.getPath2Node(graph, node1, node2);
	}

	private void getAllSubGraph(ArrayList<Object> goids, SimpleDirectedGraph subGraph, boolean toParent) {
		HashSet<Object> nodeSet = new HashSet<Object>();
		nodeSet.addAll(goids);
		if (toParent) {
			for (int i = 0; i < goids.size(); i++) {
				nodeSet.addAll(GraphPath.getUpperSubNodes(graph, goids.get(i)));
			}
		} else {
			for (int i = 0; i < goids.size(); i++) {
				nodeSet.addAll(GraphPath.getLowerSubNodes(graph, goids.get(i)));
			}
		}
		subGraph.addAllVertices(nodeSet);
		Object[] nodes = nodeSet.toArray();
		if (toParent) {
			for (int i = 0; i < nodes.length; i++) {
				subGraph.addAllEdges(graph.incomingEdgesOf(nodes[i]));
			}
		} else {
			for (int i = 0; i < nodes.length; i++) {
				subGraph.addAllEdges(graph.outgoingEdgesOf(nodes[i]));
			}
		}
	}

	private void getPartSubGraph(ArrayList<Object> goids, SimpleDirectedGraph subGraph, int level, boolean toParent) {
		HashSet<Object> nodeSet = new HashSet<Object>();
		nodeSet.addAll(goids);
		subGraph.addAllVertices(goids);
		for (int i = 0; i < level; i++) {
			Object[] vectexes = nodeSet.toArray();
			nodeSet.clear();
			for (int m = 0; m < vectexes.length; m++) {
				Object[] edges;
				if (toParent) {
					edges = graph.incomingEdgesOf(vectexes[m]).toArray();
				} else {
					edges = graph.outgoingEdgesOf(vectexes[m]).toArray();
				}
				for (int n = 0; n < edges.length; n++) {
					Object obj;
					if (toParent) {
						obj = ((OntologyEdge) edges[n]).getSource();
					} else {
						obj = ((OntologyEdge) edges[n]).getTarget();
					}
					nodeSet.add(obj);
					subGraph.addVertex(obj);
					subGraph.addEdge((OntologyEdge) edges[n]);
				}
			}
		}
	}

	/**
	 * the roots are defined as the nodes which has no incoming edges.
	 * 
	 * @param subGraph
	 * @return
	 */
	public static Object[] getRoots(SimpleDirectedGraph subGraph) {
		Object[] vertexs = subGraph.vertexSet().toArray();
		Vector<Object> subRoot = new Vector<Object>();
		for (int i = 0; i < vertexs.length; i++) {
			if (subGraph.incomingEdgesOf(vertexs[i]).size() == 0) {
				subRoot.add(vertexs[i]);
			}
		}
		return subRoot.toArray();
	}

	/**
	 * the leave are defindes as the nodes which has no outgoing edges.
	 * 
	 * @param subGraph
	 * @return
	 */
	public static Object[] getLeave(SimpleDirectedGraph subGraph) {
		Object[] vertexs = subGraph.vertexSet().toArray();
		Vector<Object> leave = new Vector<Object>();
		for (int i = 0; i < vertexs.length; i++) {
			if (subGraph.outgoingEdgesOf(vertexs[i]).size() == 0) {
				leave.add(vertexs[i]);
			}
		}
		return leave.toArray();
	}

	/**
	 * get the whole GO graph
	 * 
	 * @return
	 */
	public SimpleDirectedGraph getGraph() {
		return graph;
	}

	public HashMap<GO_term, Integer> getNodeChildNumberMap() {
		return nodeChildNumberMap;
	}

	/**
	 * get obsolete terms in the go definition
	 * 
	 * @return
	 */
	public HashSet<GO_term> getObsoleteSet() {
		return obsoleteSet;
	}
	
	//public static HashMap<String, GO_term> loadGoDefinition(File definition, GoParser.OntologyDefineType ontologyType){
		
	//}

	/**
	 * Construct a map between GO id and GO terms
	 * 
	 * @param definition
	 *            GO defineion file
	 * @param isXml
	 *            the defintion file type, XML or OBO.
	 * @return GO ids map, including obsolete GO ids
	 */
	public static HashMap<String, GO_term> loadGoDefinition(File definition,String idPatternStr, String idPrefixStr,String connectStr, GoParser.OntologyDefineType ontologyType)
			throws GoException {
		GoParser parser = null;
		HashMap<String, GO_term> goMap = null;
		try {
			if (ontologyType.equals(OntologyDefineType.OBO))
				parser = new OboGoParser(idPatternStr, idPrefixStr, connectStr,definition);
			else if (ontologyType.equals(OntologyDefineType.OBOXML))
				parser = new OboXmlGoParser(idPatternStr, idPrefixStr, connectStr,definition);
			else if (ontologyType.equals(OntologyDefineType.RDFXML))
				parser = new RdfXmlGoParser(idPatternStr, idPrefixStr, connectStr,definition);
			else if (ontologyType.equals(OntologyDefineType.OWL))
				parser = new OwlGoParser(idPatternStr, idPrefixStr, connectStr,definition);
			goMap = parser.getTermMap();
		} catch (Exception e) {
			goMap = null;
		}
		if (parser == null || goMap == null || goMap.size() == 0) {
			GoException exe = new GoException(definition, ontologyType);
			throw exe;
		}
		if (ontologyType == GoParser.OntologyDefineType.OBO) {
			String[] gos = goMap.keySet().toArray(new String[0]);
			try {
				for (int i = 0; i < gos.length; i++) {
					OBOGo_term term = (OBOGo_term) goMap.get(gos[i]);
					if (term.getAlt_id()!=null && term.getAlt_id().size() > 0) {
						for (int j = 0; j < term.getAlt_id().size(); j++) {
							goMap.put(term.getAlt_id().get(j), term);
						}
					}
				}
			} catch (Exception e) {
				System.out.println(e);
			}
		}
		return goMap;
	}

	/*
	 * public SimpleDirectedGraph getSubGraph(Object root) { SimpleDirectedGraph subGraph = new SimpleDirectedGraph();
	 * subGraph.addVertex(root); List edges = graph.outgoingEdgesOf(root); for (int i = 0; i < edges.size(); i++) {
	 * DirectedEdge edge = (DirectedEdge) edges.get(i); Object target = edge.getTarget(); getSubGraph(target, subGraph); }
	 * subGraph.addAllEdges(edges); return subGraph; } private void getSubGraph(Object currentNode, SimpleDirectedGraph
	 * subGraph) { subGraph.addVertex(currentNode); List edges = graph.outgoingEdgesOf(currentNode); for (int i = 0; i <
	 * edges.size(); i++) { DirectedEdge edge = (DirectedEdge) edges.get(i); Object target = edge.getTarget();
	 * getSubGraph(target, subGraph); } subGraph.addAllEdges(edges); }
	 */
}
