#!/usr/bin/env perl
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::
#             Codon.pm odyssey
#
#     Copyright (C) 2001 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Codon.pm,v 1.11 2001/09/11 17:52:49 s98982km Exp $

package G::Seq::Codon;

use SubOpt;
use GD;
use G::Messenger;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;

@ISA = qw(Exporter AutoLoader);
# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.
@EXPORT = qw(
	     codon_counter
	     amino_counter
	     codon_usage
	     _codon_table
	     _codon_amino_printer
	     _codon_usage_printer
);
$VERSION = '0.01';

#::::::::::::::::::::::::::::::
#        Methods Start
#::::::::::::::::::::::::::::::
sub new{
    my $pkg = shift;
    my $filename = shift;
    my $option = shift;
    my $this;

    return $this;
}


#codon_cut ver.20010721-01
#scripting by Koya Mori(mory@g-language.org)
#This program counts codons in CDS.
#options::
#-CDSid    CDS ID number(default:'')
#-filename output filename(default:'codon.csv')
#-option   "f" for file output, "stdout" for STDOUT output(default:'stdout')
#(pointer result)=&codon_counter(pointer Genome,  string key,  boolean debug);
sub codon_counter{
    &opt::default(CDSid=>"",filename=>'codon.csv',output=>'stdout');
    my @args=opt::get(@_);
    my $gb=shift @args;
    my $key=opt::val("CDSid");
    my $filename=opt::val("filename");
    my $output=opt::val("output");
    my %result;
    my $seq;
    my $i=1;
    my $q;

    if($key){
	if(defined(%{$gb->{"$key"}})){
	    $seq=$gb->get_geneseq("$key");
	    for($q=0;$q<length($seq);$q+=3){
		$result{substr($seq,$q,3)}++;
	    }
	}
    }
    else{
	foreach($gb->cds()){
	    $seq=$gb->get_geneseq("CDS$i");
	    for($q=0;$q<length($seq);$q+=3){
		$result{substr($seq,$q,3)}++;
	    }  
	    $i++;
	}
    }
    
    if($output eq "f"){
	_codon_amino_printer(\%result,-output=>"f",-filename=>"$filename");
    }
    if($output eq "stdout"){
	_codon_amino_printer(\%result,-output=>"stdout");
    }
    return \%result;
}


#amino_counter ver.20010721-01
#scripting by Koya Mori(mory@g-language.org)
#This program counts amino acid in CDS.
#options::
#-CDSid    CDS ID number(default:'')
#-filename output filename(default:'amino.csv')
#-output   "f" for file output, "stdout" for STDOUT output(default:'stdout')
#(pointer result)=&amino_counter(pointer Genome,  string key,  boolean debug);
sub amino_counter{
    &opt::default(CDSid=>"",filename=>'amino.csv',output=>'stdout');
    my @args=opt::get(@_);
    my $gb=shift @args;
    my $key=opt::val("CDSid");
    my $filename=opt::val("filename");
    my $output=opt::val("output");
    my %amino;
    my @tmp;
    my $num;
    my $i=1;
    
    if($key){
	if(defined(%{$gb->{"$key"}})){
	    $num=$gb->{"$key"}->{feature};
	    @tmp=split(//,$gb->{"FEATURE$num"}->{translation});
	    foreach(@tmp){
		$amino{$_}++;
	    }
	}
    }
    else{
	foreach($gb->feature()){
	    if($gb->{"FEATURE$i"}->{type} eq 'CDS'){
		@tmp=split(//,$gb->{"FEATURE$i"}->{translation});
		foreach(@tmp){
		    $amino{$_}++;
		}
	    }
	    $i++;
	}
    }
    
    if($output eq "f"){
	_codon_amino_printer(\%amino,-output=>"f",-filename=>"$filename");
    }
    if($output eq "stdout"){
	_codon_amino_printer(\%amino,-output=>"stdout");
    }

    return \%amino;
}


sub codon_amino_counter{
    my $gb=shift;
    my $key=shift;
    my %result;
    my $seq;
    my $amino;
    my $codon;
    my $num;
    my $q;
    my @tmp;
    
    if(defined(%{$gb->{"$key"}})){
	$seq=lc($gb->get_geneseq("$key"));
	$num=$gb->{"$key"}->{feature};
    }
    
    @tmp=split(//,$gb->{"FEATURE$num"}->{translation});
    for($q=0;$q<length($seq);$q+=3){
	$amino=shift @tmp;
	$amino="/" if($amino eq "");
	$codon=substr($seq,$q,3);
	$result{$amino}{$codon}++;
    }
    
    return %result;
}

    
#codon_usage ver.20010721-01
#scripting by Koya Mori(mory@g-language.org)
#This program calculates codon usage in one CDS.
#options::
#-CDSid    CDS ID number(default:'')
#-filename output filename(default:'amino.csv')
#-output   "f" for file output, "stdout" for STDOUT output(default:'stdout')
#(pointer result)=&codon_usage(pointer Genome,  string key,  boolean debug);
sub codon_usage{
    &opt::default(CDSid=>"",filename=>'codon_usage.csv',output=>'show');
    my @args=opt::get(@_);

    my $gb=shift @args;
    my $key=opt::val("CDSid");
    my $filename=opt::val("filename");
    my $output=opt::val("output");
    my $codon;
    my $amino;
    my %result;
    my %usage;
    
    if($key){
	%usage=codon_amino_counter($gb,$key);
    }
    else{
	foreach($gb->cds()){
	    %result=();
	    %result=codon_amino_counter($gb,$_);
	    foreach $amino (keys(%result)){
		foreach $codon (keys(%{$result{$amino}})){
		    $usage{$amino}{$codon}+=$result{$amino}{$codon};
		}
	    }
	}
    }
    
    if($output eq "f"){
	_codon_usage_printer(\%usage,-output=>"f",-filename=>"$filename");
    }
    if($output!~/[fn]/){
	_codon_usage_printer(\%usage,-output=>"stdout");
    }

    _codon_table(\%usage,-output=>"$output") if($output=~/(g|show)/ && $key eq "");

    return \%usage;
}


sub _codon_table{
    &opt::default(output=>"show",filename=>"codon_table.gif");
    my @args=opt::get(@_);
    
    my $result=shift @args;
    my $filename=opt::val("filename");
    my $output=opt::val("output");
    my $x;
    my $y;
    my %amino;
    my %data;
    my %per;
    my $amino_total;
    my $codon;
    my $amino;
    my $v;
    my $h;
    my %exception;
    my $CoDoN;
    my %color;

    my $im = new GD::Image(500,550);
    my $white = $im->colorAllocate(255,255,255);
    my $black = $im->colorAllocate(0,0,0);
    my $red = $im->colorAllocate(255,0,0);
    my $yellow = $im->colorAllocate(200,200,0);
    my $green = $im->colorAllocate(0,150,0);
    my $blue = $im->colorAllocate(0,0,255);

    $color{D}=$yellow;
    $color{E}=$yellow;
    $color{R}=$red;
    $color{K}=$red;
    $color{H}=$red;
    $color{N}=$blue;
    $color{Q}=$blue;
    $color{S}=$blue;
    $color{T}=$blue;
    $color{Y}=$blue;
    $color{A}=$green;
    $color{G}=$green;
    $color{V}=$green;
    $color{L}=$green;
    $color{I}=$green;
    $color{P}=$green;
    $color{F}=$green;
    $color{M}=$green;
    $color{W}=$green;
    $color{C}=$green;
    $color{'/'}=$black;

    foreach((10,50,450,490)){
	$x=$_;
	for($y=10;$y<450;$y++){
	    $im->setPixel($x,$y,$black);
	} 
    }
    foreach((150,250,350)){
	$x=$_;
	for($y=30;$y<450;$y++){
	    $im->setPixel($x,$y,$black);
	} 
    }
    $y=30;
    for($x=50;$x<450;$x++){
	$im->setPixel($x,$y,$black);
    }
    foreach((10,50,150,250,350,450)){
	$y=$_;
	for($x=10;$x<490;$x++){
	    $im->setPixel($x,$y,$black);
	} 
    }

    $im->string(gdSmallFont,15,25,"first",$red);
    $im->string(gdSmallFont,233,15,"second",$green);
    $im->string(gdSmallFont,455,25,"third",$blue);
    $im->string(gdSmallFont,30,95,"T",$red);
    $im->string(gdSmallFont,30,195,"C",$red);
    $im->string(gdSmallFont,30,295,"A",$red);
    $im->string(gdSmallFont,30,395,"G",$red);
    $im->string(gdSmallFont,100,35,"T",$green);
    $im->string(gdSmallFont,200,35,"C",$green);
    $im->string(gdSmallFont,300,35,"A",$green);
    $im->string(gdSmallFont,400,35,"G",$green);
    $im->string(gdSmallFont,470,65,"T",$blue);
    $im->string(gdSmallFont,470,85,"C",$blue);
    $im->string(gdSmallFont,470,105,"A",$blue);
    $im->string(gdSmallFont,470,125,"G",$blue);
    $im->string(gdSmallFont,470,165,"T",$blue);
    $im->string(gdSmallFont,470,185,"C",$blue);
    $im->string(gdSmallFont,470,205,"A",$blue);
    $im->string(gdSmallFont,470,225,"G",$blue);
    $im->string(gdSmallFont,470,265,"T",$blue);
    $im->string(gdSmallFont,470,285,"C",$blue);
    $im->string(gdSmallFont,470,305,"A",$blue);
    $im->string(gdSmallFont,470,325,"G",$blue);
    $im->string(gdSmallFont,470,365,"T",$blue);
    $im->string(gdSmallFont,470,385,"C",$blue);
    $im->string(gdSmallFont,470,405,"A",$blue);
    $im->string(gdSmallFont,470,425,"G",$blue);

    foreach $amino (keys(%{$result})){
	$amino_total=0;
	foreach $codon (keys(%{$$result{$amino}})){
	    $amino_total+=$$result{$amino}{$codon};
	}
	foreach $codon (keys(%{$$result{$amino}})){
	    if($$result{$amino}{$codon} > $data{$codon}){
		if($data{$codon}!=""){
		    $exception{$codon}{amino}=$amino{$codon};
		    $exception{$codon}{per}=$per{$codon};
		}
		$data{$codon}=$$result{$amino}{$codon};
		$amino{$codon}=$amino;
		$per{$codon}=sprintf("%.3f",$$result{$amino}{$codon}/$amino_total);
	    }
	    else{
		$exception{$codon}{amino}=$amino;
		$exception{$codon}{per}=sprintf("%.3f",$$result{$amino}{$codon}/$amino_total);
	    }
	}
    }

    $im->string(gdSmallFont,60,65,"TTT  $amino{ttt}  $per{ttt}",$color{$amino{ttt}});
    $im->string(gdSmallFont,60,85,"TTC  $amino{ttc}  $per{ttc}",$color{$amino{ttc}});
    $im->string(gdSmallFont,60,105,"TTA  $amino{tta}  $per{tta}",$color{$amino{tta}});
    $im->string(gdSmallFont,60,125,"TTG  $amino{ttg}  $per{ttg}",$color{$amino{ttg}});
    $im->string(gdSmallFont,60,165,"CTT  $amino{ctt}  $per{ctt}",$color{$amino{ctt}});
    $im->string(gdSmallFont,60,185,"CTC  $amino{ctc}  $per{ctc}",$color{$amino{ctc}});
    $im->string(gdSmallFont,60,205,"CTA  $amino{cta}  $per{cta}",$color{$amino{cta}});
    $im->string(gdSmallFont,60,225,"CTG  $amino{ctg}  $per{ctg}",$color{$amino{ctg}});
    $im->string(gdSmallFont,60,265,"ATT  $amino{att}  $per{att}",$color{$amino{att}});
    $im->string(gdSmallFont,60,285,"ATC  $amino{atc}  $per{atc}",$color{$amino{atc}});
    $im->string(gdSmallFont,60,305,"ATA  $amino{ata}  $per{ata}",$color{$amino{ata}});
    $im->string(gdSmallFont,60,325,"ATG  $amino{atg}  $per{atg}",$color{$amino{atg}});
    $im->string(gdSmallFont,60,365,"GTT  $amino{gtt}  $per{gtt}",$color{$amino{gtt}});
    $im->string(gdSmallFont,60,385,"GTC  $amino{gtc}  $per{gtc}",$color{$amino{gtc}});
    $im->string(gdSmallFont,60,405,"GTA  $amino{gta}  $per{gta}",$color{$amino{gta}});
    $im->string(gdSmallFont,60,425,"GTG  $amino{gtg}  $per{gtg}",$color{$amino{gtg}});

    $im->string(gdSmallFont,160,65,"TCT  $amino{tct}  $per{tct}",$color{$amino{tct}});
    $im->string(gdSmallFont,160,85,"TCC  $amino{tcc}  $per{tcc}",$color{$amino{tcc}});
    $im->string(gdSmallFont,160,105,"TCA  $amino{tca}  $per{tca}",$color{$amino{tca}});
    $im->string(gdSmallFont,160,125,"TCG  $amino{tcg}  $per{tcg}",$color{$amino{tcg}});
    $im->string(gdSmallFont,160,165,"CCT  $amino{cct}  $per{cct}",$color{$amino{cct}});
    $im->string(gdSmallFont,160,185,"CCC  $amino{ccc}  $per{ccc}",$color{$amino{ccc}});
    $im->string(gdSmallFont,160,205,"CCA  $amino{cca}  $per{cca}",$color{$amino{cca}});
    $im->string(gdSmallFont,160,225,"CCG  $amino{ccg}  $per{ccg}",$color{$amino{ccg}});
    $im->string(gdSmallFont,160,265,"ACT  $amino{act}  $per{act}",$color{$amino{act}});
    $im->string(gdSmallFont,160,285,"ACC  $amino{acc}  $per{acc}",$color{$amino{acc}});
    $im->string(gdSmallFont,160,305,"ACA  $amino{aca}  $per{aca}",$color{$amino{aca}});
    $im->string(gdSmallFont,160,325,"ACG  $amino{acg}  $per{acg}",$color{$amino{acg}});
    $im->string(gdSmallFont,160,365,"GCT  $amino{gct}  $per{gct}",$color{$amino{gct}});
    $im->string(gdSmallFont,160,385,"GCC  $amino{gcc}  $per{gcc}",$color{$amino{gcc}});
    $im->string(gdSmallFont,160,405,"GCA  $amino{gca}  $per{gca}",$color{$amino{gca}});
    $im->string(gdSmallFont,160,425,"GCG  $amino{gcg}  $per{gcg}",$color{$amino{gcg}});

    $im->string(gdSmallFont,260,65,"TAT  $amino{tat}  $per{tat}",$color{$amino{tat}});
    $im->string(gdSmallFont,260,85,"TAC  $amino{tac}  $per{tac}",$color{$amino{tac}});
    $im->string(gdSmallFont,260,105,"TAA  $amino{taa}  $per{taa}",$color{$amino{taa}});
    $im->string(gdSmallFont,260,125,"TAG  $amino{tag}  $per{tag}",$color{$amino{tag}});
    $im->string(gdSmallFont,260,165,"CAT  $amino{cat}  $per{cat}",$color{$amino{cat}});
    $im->string(gdSmallFont,260,185,"CAC  $amino{cac}  $per{cac}",$color{$amino{cac}});
    $im->string(gdSmallFont,260,205,"CAA  $amino{caa}  $per{caa}",$color{$amino{caa}});
    $im->string(gdSmallFont,260,225,"CAG  $amino{cag}  $per{cag}",$color{$amino{cag}});
    $im->string(gdSmallFont,260,265,"AAT  $amino{aat}  $per{aat}",$color{$amino{aat}});
    $im->string(gdSmallFont,260,285,"AAC  $amino{aac}  $per{aac}",$color{$amino{aac}});
    $im->string(gdSmallFont,260,305,"AAA  $amino{aaa}  $per{aaa}",$color{$amino{aaa}});
    $im->string(gdSmallFont,260,325,"AAG  $amino{aag}  $per{aag}",$color{$amino{aag}});
    $im->string(gdSmallFont,260,365,"GAT  $amino{gat}  $per{gat}",$color{$amino{gat}});
    $im->string(gdSmallFont,260,385,"GAC  $amino{gac}  $per{gac}",$color{$amino{gac}});
    $im->string(gdSmallFont,260,405,"GAA  $amino{gaa}  $per{gaa}",$color{$amino{gaa}});
    $im->string(gdSmallFont,260,425,"GAG  $amino{gag}  $per{gag}",$color{$amino{gag}});

    $im->string(gdSmallFont,360,65,"TGT  $amino{tgt}  $per{tgt}",$color{$amino{tgt}});
    $im->string(gdSmallFont,360,85,"TGC  $amino{tgc}  $per{tgc}",$color{$amino{tgc}});
    $im->string(gdSmallFont,360,105,"TGA  $amino{tga}  $per{tga}",$color{$amino{tga}});
    $im->string(gdSmallFont,360,125,"TGG  $amino{tgg}  $per{tgg}",$color{$amino{tgg}});
    $im->string(gdSmallFont,360,165,"CGT  $amino{cgt}  $per{cgt}",$color{$amino{cgt}});
    $im->string(gdSmallFont,360,185,"CGC  $amino{cgc}  $per{cgc}",$color{$amino{cgc}});
    $im->string(gdSmallFont,360,205,"CGA  $amino{cga}  $per{cga}",$color{$amino{cga}});
    $im->string(gdSmallFont,360,225,"CGG  $amino{cgg}  $per{cgg}",$color{$amino{cgg}});
    $im->string(gdSmallFont,360,265,"AGT  $amino{agt}  $per{agt}",$color{$amino{agt}});
    $im->string(gdSmallFont,360,285,"AGC  $amino{agc}  $per{agc}",$color{$amino{agc}});
    $im->string(gdSmallFont,360,305,"AGA  $amino{aga}  $per{aga}",$color{$amino{aga}});
    $im->string(gdSmallFont,360,325,"AGG  $amino{agg}  $per{agg}",$color{$amino{agg}});
    $im->string(gdSmallFont,360,365,"GGT  $amino{ggt}  $per{ggt}",$color{$amino{ggt}});
    $im->string(gdSmallFont,360,385,"GGC  $amino{ggc}  $per{ggc}",$color{$amino{ggc}});
    $im->string(gdSmallFont,360,405,"GGA  $amino{gga}  $per{gga}",$color{$amino{gga}});
    $im->string(gdSmallFont,360,425,"GGG  $amino{ggg}  $per{ggg}",$color{$amino{ggg}});

    $im->string(gdSmallFont,15,465,"yellow  minus charge",$yellow);
    $im->string(gdSmallFont,165,465,"red  plus charge",$red);
    $im->string(gdSmallFont,285,465,"blue  noncharge",$blue);
    $im->string(gdSmallFont,400,465,"green  nonpolar",$green);

    $im->string(gdSmallFont,20,485,"exception",$black);
    $v=485;
    $h=100;
    foreach(sort keys(%exception)){
	$color{$exception{$_}{amino}}=$black if($color{$exception{$_}{amino}}=="");	
	$CoDoN=uc $_;
	$im->string(gdSmallFont,$h,$v,"$CoDoN  $exception{$_}{amino}  $exception{$_}{per}",$color{$exception{$_}{amino}});
	$v+=20;
	if($v == 545){
	    $v=485;
	    $h+=100;
	}
    }

    mkdir ("graph",0777);
    open(OUT,'>graph/'."$filename");
    print OUT $im->gif;
    close(OUT);
    
    system("gimv graph/$filename") if($output eq "show");
}


#codon_amino_printer ver.20010722-01
#scripting by Koya Mori(mory@g-language.org)
#This program prints result of codon or amino counter.
#&codon_amino_printer(pointer codon_counter,  boolean debug);
sub _codon_amino_printer{
    &opt::default(output=>"stdout");
    my @args=opt::get(@_);
    my $result=shift @args;
    my $printer=opt::val("output");
    my $filename=opt::val("filename");
    my $total;

    
    if($printer eq "f"){
	open(FILE,">>$filename");
	foreach(sort keys(%$result)){
	    print FILE $_,",",$result->{$_},"\n";
	    $total+=$result->{$_};
	}
	print FILE "total,",$total,"\n";
	print FILE "\n\n";
	close(FILE);
    }
    else{
	foreach(sort keys(%$result)){
	    &msg::send($_," -> ",$result->{$_},"\n");
	    $total+=$result->{$_};
	}
	&msg::send("total -> ",$total,"\n");    
    }
}


#codon_usage_printer ver.20010722-01
#scripting by Koya Mori(mory@g-language.org)
#This program prints result of codon usage.
#&codon_usage_printer(pointer codon_usage,  boolean debug);
sub _codon_usage_printer{
    &opt::default(output=>"stdout");
    my @args=opt::get(@_);
    my $result=shift @args;
    my $printer=opt::val("output");
    my $filename=opt::val("filename");
    my $amino_total;
    my $key;
    my $key2;
    my $total;


    if($printer eq "f"){
	open(FILE,">>$filename");
	foreach $key (sort keys(%$result)){
	    $amino_total=0;
	    foreach $key2 (sort keys(%{$$result{$key}})){
		$amino_total+=$$result{$key}{$key2};
	    }
	    foreach $key2 (sort keys(%{$$result{$key}})){
		print FILE $key,",",$key2,",",$$result{$key}{$key2},",",sprintf("%.3f",$$result{$key}{$key2}/$amino_total),"\n";
		$total+=$$result{$key}{$key2};
	    }
	}
	print FILE "total,$total\n";
	print FILE "\n\n";
	close(FILE);
    }
    else{
	foreach $key (sort keys(%$result)){
	    $amino_total=0;
	    foreach $key2 (sort keys(%{$$result{$key}})){
		$amino_total+=$$result{$key}{$key2};
	    }
	    foreach $key2 (sort keys(%{$$result{$key}})){
		&msg::send($key," -> ",$key2," -> ",$$result{$key}{$key2},"   ",sprintf("%.3f",$$result{$key}{$key2}/$amino_total),"\n");
		$total+=$$result{$key}{$key2};
	    }
	}
	&msg::send("total -> $total\n");
    }
}


sub _translate {
    my $seq = lc(shift);
    my $amino = '';
    my %CodonTable = (
               'gac', 'D', 'caa', 'Q', 'gca', 'A', 'ctg', 'L',
               'gat', 'D', 'cag', 'Q', 'gcc', 'A', 'ctt', 'L',
               'gaa', 'E', 'agc', 'S', 'gcg', 'A', 'ata', 'I',
               'gag', 'E', 'agt', 'S', 'gct', 'A', 'atc', 'I',
               'aga', 'R', 'tca', 'S', 'gga', 'G', 'att', 'I',
               'agg', 'R', 'tcc', 'S', 'ggc', 'G', 'cca', 'P',
               'cga', 'R', 'tcg', 'S', 'ggg', 'G', 'ccc', 'P',
               'cgc', 'R', 'tct', 'S', 'ggt', 'G', 'ccg', 'P',
               'cgg', 'R', 'aca', 'T', 'gta', 'V', 'cct', 'P',
               'cgt', 'R', 'acc', 'T', 'gtc', 'V', 'atg', 'M',
               'aaa', 'K', 'acg', 'T', 'gtg', 'V', 'tgg', 'W',
               'aag', 'K', 'act', 'T', 'gtt', 'V', 'tgc', 'C',
               'cac', 'H', 'tac', 'Y', 'tta', 'L', 'tgt', 'C',
               'cat', 'H', 'tat', 'Y', 'ttg', 'L', 'taa', '/',
               'aac', 'N', 'ttc', 'F', 'cta', 'L', 'tag', '/',
               'aat', 'N', 'ttt', 'F', 'ctc', 'L', 'tga', '/'
                  );

    while(3 <= length($seq)){
        my $codon = substr($seq, 0, 3);
        substr($seq, 0, 3) = '';
        if ($codon =~ /[^atgc]/){
            $amino .= '?';
        }else{
            $amino .= $CodonTable{$codon};
        }
    }
    if(length($seq)){
        &msg::error("Translation: illegal length.\n");
    }

    return $amino;
}


sub DESTROY {
    my $self = shift;
}

1;
__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

G::Seq::Codon - Perl extension for blah blah blah

=head1 SYNOPSIS

  use G::Seq::Codon;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for G::Seq::Codon was created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.

=head1 AUTHOR

A. U. Thor, a.u.thor@a.galaxy.far.far.away

=head1 SEE ALSO

perl(1).

=cut
