#! /usr/bin/env perl

#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::
#             Rcmd.pm Prelude core
#
#     Copyright (C) 2001 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Rcmd.pm,v 1.1.1.1 2001/08/31 14:10:41 t98901ka Exp $
#
# G-language System is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language System is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language System -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@g-language.org> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#


package Rcmd;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw(
	     new
);
$VERSION = '1.0';



######################################
######### Main Methods ###############
######################################

sub new {
    my $this = shift;
    srand(time);
    my $uniq = rand(9999);
    my $cmd = "/tmp/$uniq.cmd";
    my $log = "/tmp/$uniq.log";

    bless { 
	cmd => $cmd, 
	log => $log
    }
}

sub DESTROY{
    my $this = shift;
    unlink $this->{cmd} or die $!;
    unlink $this->{log} or die $!;
}

sub exec{
    my $this = shift;
    my @tmp = @_;
    my $request = join("\n", @_, '');
    my $data = '';

    open(CMD, '>>' . $this->{cmd});
    print CMD $request;
    close(CMD);

    system("/usr/bin/env R --no-save --slave < "
	   . $this->{cmd} . " > " . $this->{log});

    open(DATA, $this->{log});
    while(<DATA>){
	if (/\[(\d+)\] (.*)/){
	    if ($1 > 1){
		$data .= $2;
	    }else{
		$data = $2;
	    }
	}elsif(/Error/){
	    print STDERR $_;
	}
    }
    close(DATA);

    return split(/ /,$data);
}




1;
__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

Rcmd - Perl interface for R language

=head1 SYNOPSIS

  use Rcmd;

  $rcmd = new Rcmd;

  @result = $rcmd->exec("<R commands>","<next R command>");  

=head1 DESCRIPTION

  Rcmd enables Perl manipulation of the R language, simply executing them
  through $rcmd->exec() function. Input is an array of R commands.

  ex:

  print $rcmd->exec(
                  "x_5", 
                  "y_4", 
                  "z_x*y", 
                  "z"
		   );

  Returned values are always an array. Therefore, in case the returned value
  is only one, the value is accessible as:

  @val = $rcmd->exec("y");

  print $val[0];

  All the values are saved in each session. Thus,

    $val1 = $rcmd->exec( "x_5" , "x" );
    $val2 = $rcmd->exec( "x");

  will output "5" for both $val1 and $val2.

  Obviously, it is also possible to use perl variables, as:

    $i = 3;

    print $rcmd->exec("y_y*$i","x");

  The strength of R graphing abilities can be accessed as:

  @array = $rcmd->exec(
		       "postscript(\"/tmp/out.ps\")", 
		       "x_c(1:10)",
		       "y_c(3,6,3,5,8,0,1,9,2,6)",
		       "plot(x,y)",
		       "z_lsfit(x,y)",
		       "abline(z)",
		       "y"
		       );

  system("gs /tmp/out.ps");

  #R language is S-plus clone availabe with GPL at http://www.r-project.org/

=head1 AUTHOR

Kazuharu Gaou Arakawa, gaou@g-language.org

=head1 SEE ALSO

perl(1).

=cut
