#!/usr/bin/env perl

#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2007 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Skyline.pm,v 1.1 2002/07/30 17:44:27 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@g-language.org> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::Skyline;

use strict;

BEGIN{
    eval "use Bio::SeqIO;";
    if($@){ warn "$@" };
    eval "use Bio::DB::GenBank;";
    if($@){ warn "$@" };
    eval "use Bio::DB::RefSeq;";
    if($@){ warn "$@" };
    eval "use Bio::Perl;";
    if($@){ warn "$@" };
}

use G::Messenger;
use G::DB::SDB;
use G::IO::Bioperl;
use G::IO::Annotation;
require G::Prelude;
require G::IO::GenBankI;
require G::IO::GenBankO;
require G::IO::FastaI;
require G::IO::EmblI;

use File::Basename;

use vars qw($VERSION @ISA @INC $AUTOLOAD);

@ISA = qw(G::Prelude G::IO::GenBankI G::IO::GenBankO);

$VERSION = '1.6.4';

#::::::::::::::::::::::::::::::
#          Variables
#::::::::::::::::::::::::::::::

my $outfile = '';
my $loaded = 0;
my $infile = '';

#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::

sub new {
    my $pkg = shift;
    my $filename = shift;
    my @options = @_;
    my $this = {};
    my $tmp = {};
    $infile = $filename;
    $filename =~ s/\~\//sprintf("%s\/",$ENV{HOME})/e;
    
    bless $this;

    my $no_msg = 0;
    my $without_annotation = 0;
    my $multiple_locus = 0;
    my $long_sequence = 0;
    my $bioperl = 0;
    my $netgbk = 0;
    my $longest_ORF_annotation = 0;
    my $glimmer_annotation = 0;
    my $no_cache = 0;
    my $force_cache = 0;
    my $format = '';
    my $locus_msg = '';

    return $this if ($filename eq 'blessed');

    foreach my $opt_tmp (@options){
	my $opt = lc($opt_tmp);

	if ($opt =~ /no msg/){
	    $no_msg = 1;
	}elsif ($opt =~ /without annotation/){
	    $without_annotation = 1;
	    $no_cache = 1;
	}elsif ($opt =~ /multiple locus/){
	    $multiple_locus = 1;
	    $no_cache = 1;
	}elsif ($opt =~ /long sequence/){
	    $long_sequence = 1;
	    $no_cache = 1;
	}elsif ($opt =~ /bioperl/){
	    $bioperl = 1;
	    $format = 'bioperl';
	}elsif ($opt =~ /longest ORF annotation/){
	    $longest_ORF_annotation = 1;
	}elsif ($opt =~ /glimmer annotation/){
	    $glimmer_annotation = 1;
	}elsif ($opt =~ /no cache/){
	    $no_cache = 1;
	}elsif ($opt =~ /force cache/){
	    $force_cache = 1;
	    $no_cache = 1;
	}elsif ($opt =~ /net GenBank/){
	    $netgbk = 1; 
	    $format = 'net genbank';
	}elsif (   lc($opt) eq 'fasta' || lc($opt) eq 'embl' 
		|| lc($opt) eq 'swiss' || lc($opt) eq 'scf' 
		|| lc($opt) eq 'pir'   || lc($opt) eq 'gcg' 
		|| lc($opt) eq 'raw'   || lc($opt) eq 'ace'
		|| lc($opt) eq 'game'  || lc($opt) eq 'phd'
		|| lc($opt) eq 'qual'  || lc($opt) eq 'bsml' 
		|| lc($opt) eq 'genbank'){
	    $format = lc($opt);
	}else{
	    unless (lc($opt) eq 'net genbank'){
		warn("Warning: Unknown Option $opt at \"new G\"\n");
	    }
	}
    }

    $this->loaded_msg() unless ($no_msg);

    $format = interpret_format($filename) unless(length $format);

    if($format eq 'RefSeq'){
	$format = 'usa';
	$filename = "refseq:$filename";
    }elsif($format eq 'net GenBank'){
	$format = 'usa';
	$filename = "genbank:$filename";
    }elsif($format eq 'net GenPept'){
	$format = 'usa';
	$filename = "genpept:$filename";
    }

    my @stat = stat($filename);
    my $basename = basename($filename) if (-e $filename);
    my $cachedFile = sprintf("%s/.glang/cache-%s-%s-%s", $ENV{HOME}, $basename, $stat[7], $stat[9]);

    if (-e $cachedFile && $no_cache == 0){
	$this = sdb_load(sprintf("cache-%s-%s-%s", $basename, $stat[7], $stat[9]));
	$this->seq_info() unless ($no_msg);
	my $now = time();
	utime $now, $now, $cachedFile;
    }elsif ($bioperl || $format eq 'bioperl'){
        $this = G::IO::Bioperl::convert($filename);
	$this->seq_info() unless ($no_msg);

    }elsif ($format eq 'usa'){
	my ($dbname, $entryname) = split(/:/, $filename, 2);

	msg_error("Retrieving sequence from " . uc($dbname) . "...\n");

	my $bp;

	eval {
	    $bp = get_sequence($dbname, $entryname);
	};
	die("$@ Could not retrieve $filename\n") if $@;

        $this = G::IO::Bioperl::convert($bp);
	$this->seq_info() unless($no_msg);
	$this->{BIOPERL} =  $bp;
    }else{
	if ($format eq 'embl'){
	    $this = new G::IO::EmblI;
	}elsif ($format eq 'fasta'){
	    $this = new G::IO::FastaI;
	}elsif ($format ne 'genbank'){

	    $outfile = _generate_tmpfilename();
	    my $in = read_sequence($filename);
	    my $out = write_sequence(">$outfile", "genbank", $in);

	    $filename = $outfile;
	    $locus_msg = 'no msg';
	    $format = 'genbank';
	}

	*GENBANK = $this->open_gb($filename);

	if ($multiple_locus){
	    $tmp = new G::Skyline($filename, "no msg", $format, "no cache");
	    $this->multi_locus($tmp);
	    $this->seq_info() unless($no_msg);
	}else{
	    if ($without_annotation){
		$this->goto_origin();
	    }else{
		$this->read_locus($locus_msg);
		$this->read_header();
		$this->read_features();
	    }
	    
	    if ($long_sequence){
		$this->{origin} = tell *GENBANK;
	    }else{
		$this->getnucs();
		$this->seq_info() unless ($no_msg);

		if($force_cache || $no_cache == 0){
		    sdb_save($this, sprintf("cache-%s-%s-%s", $basename, $stat[7], $stat[9]));
		}
	    }
	}
    }

    if ($longest_ORF_annotation){
	my $new = new G::Skyline("blessed");
	bless $new;
	annotate_with_LORF($new, $this);
	return $new;
    }elsif ($glimmer_annotation){
	my $new = new G::Skyline("blessed");
	bless $new;
	open(FASTA, '>/tmp/out.fasta') || die ($!);
	printf FASTA ">%s\n%s\n", $this->{LOCUS}->{id}, $this->{SEQ};
	close(FASTA);
	run_glimmer($this, '/tmp/out.fasta');
	annotate_with_glimmer($new, '/tmp/out.fasta');
	unlink('/tmp/out.fasta');
	return $new;
    }

    return $this;
}

sub filepath {
    my $this = shift;

    return $infile;
}

sub loaded_msg {
    my $this = shift;

    $loaded ++;
    return if ($loaded > 1);

    my $print =
	qq(
	     __/__/__/__/__/__/__/__/__/__/__/__/__/
                
             G-language Genome Analysis Environment

	      Version: $VERSION
	      License: GPL
 
	      Copyright (C) 2001-2007 
              G-language Project
	      Institute for Advanced Biosciences,
	      Keio University, JAPAN 

          	 http://www.g-language.org/

	     __/__/__/__/__/__/__/__/__/__/__/__/__/
	   \n);

    &msg_error($print);
}

sub interpret_format {
    my $filename = shift;

    if (ref $filename =~ /Bio::Seq/){
	return 'bioperl';
    }elsif ($filename =~ /^(.*?):(.*)/){
	unless(lc($1) =~ /(swiss|genbank|genpept|embl|refseq)/){
	    die("Unsupported database name for USA. Supported databases are\n" .
		"swiss, genbank, genpept, embl, or refseq\n");
	}
	return "usa";
    }elsif (lc($filename) =~ /\.(gb|gbk|gbank|genbank)$/i){
	return 'genbank';
    }elsif (lc($filename) =~ /\.(fasta|fast|seq|fst|fa|fsa|nt|aa)/i){
	return 'fasta';
    }elsif ($filename =~ /^NC_\d+$/i){
	return 'RefSeq';
    }elsif ($filename =~ /^NP_\d+$/i){
	return 'net GenPept';
    }elsif ($filename =~ /^(?:[A-Z]|[A-Z][A-Z]).\d+$/i){
	return 'net GenBank';
    }else{
	my $format = Bio::SeqIO->_guess_format($filename);

	if (length $format){
	    return $format;
	}else{
	    warn("Unknown file format. Interpreting $filename as GenBank...\n");
	    return 'genbank';
	}
    }
}

sub output {
    my $gb = shift;
    my $file = shift;
    my $option = shift;

    $option = interpret_format($file) unless(length $option);

    if (lc($option) eq 'genbank'){
	$gb->make_gb($file);
    }elsif(length $option){
	my $in;

	if(length $gb->{BIOPERL}){
	    $in = $gb->{BIOPERL};
	}else{
	    my $outfile = _generate_tmpfilename();
	    $gb->make_gb($outfile);
	    $in = read_sequence($outfile);
	    unlink($outfile);
	}

	write_sequence(">$file", $option, $in);
    }else{
	&msg_error("G::output - Unknown format to output.");
    }
}

sub bioperl {
    my $this = shift;
    my $outfile = _generate_tmpfilename();
    $this->output($outfile);
    
    my $in = Bio::SeqIO->new(-format => "genbank", -file =>$outfile);
    my $bp = $in->next_seq();
    unlink($outfile);
    return $bp;
}

sub multi_locus{
    my $this = shift;
    my $gb = shift;
    my $lng;
    my $i = 1;
    my $f = 1;
    my $c = 1;

    do{
        my $F = 1;
        my $C = 1;
        $lng = length($this->{"SEQ"});
        $this->{"LOCUS$i"} = $gb->{"LOCUS"};
        $this->{"HEADER$i"} = $gb->{"HEADER"};
        $this->{"COMMENT$i"} = $gb->{"COMMENT"};

        while(defined (%{$gb->{"FEATURE$F"}})){
            $this->{"FEATURE$f"}            = $gb->{"FEATURE$F"};
            $this->{"FEATURE$f"}->{"start"} = $gb->{"FEATURE$F"}->{"start"} + $lng;
            $this->{"FEATURE$f"}->{"end"}   = $gb->{"FEATURE$F"}->{"end"} + $lng;
            $this->{"FEATURE$f"}->{"locus"} = $i;

	    if($gb->{"FEATURE$F"}->{"type"} eq "CDS"){
                $this->{"CDS$c"}              = $gb->{"CDS$C"};
                $this->{"CDS$c"}->{"start"}   = $gb->{"CDS$C"}->{"start"} + $lng;
                $this->{"CDS$c"}->{"end"}     = $gb->{"CDS$C"}->{"end"} + $lng;
                $this->{"CDS$c"}->{"feature"} = $f;
                $this->{"CDS$c"}->{"locus"}   = $i;


                if(defined $gb->{"CDS$C"}->{"join"}){
                    my @join = split(/\,/,$gb->{"CDS$C"}->{"join"});
		    my @num = ();
		    my @new_join = ();

                    foreach(@join){
                        if(tr/c/c/){
                            @num = split(/\.\./,$_);
                            push (@new_join, sprintf ("c%d\.\.%d", $num[0] + $lng, $num[1] + $lng));
                        } else {
                            @num = split(/\.\./,$_);
                            push (@new_join, sprintf ("%d\.\.%d",  $num[0] + $lng, $num[1] + $lng));
                        }
                    }
                    $this->{"CDS$c"}->{join} = join(',', @new_join);
                }
		$this->{"FEATURE$f"}->{"cds"} = $c;
                $c++;
                $C++;
            }
            $f++;
            $F++;
        }
        $this->{"SEQ"} .= $gb->{"SEQ"};
        $i++;

    }while($gb->next_locus("no msg"));

    $this->{"FEATURE0"}->{"type"} = "source";
    $this->{"FEATURE0"}->{"start"} = 1;
    $this->{"FEATURE0"}->{"end"} = length($this->{"SEQ"});
    $this->{"LOCUS"} = $this->{"LOCUS1"};
    $this->{"HEADER"} = $this->{"HEADER1"};
    $this->{"COMMENT"} = $this->{"COMMENT1"};
    $this->{"CDS0"}->{dummy} = 1;
}


sub _generate_tmpfilename{
    return '/tmp/G-INTERNAL-' . $ENV{USER} . time() . rand() .'.gbk';
}

sub DESTROY {
    my $self = shift;

    undef %{$self};
    unlink($outfile);
    $self->close_gb();
}



1;

__END__

=head1 NAME

G::Skyline

=head1 SYNOPSIS

 use G::Skyline;
 @ISA = (G::Skyline);
   
=head1 DESCRIPTION

 Inherits all necessary classes.
 Intended for internal use only.

=back

=head1 AUTHOR

Kazuharu Gaou Arakawa, gaou@g-language.org

=head1 SEE ALSO

perl(1).

=cut



