#!/usr/bin/env perl

#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2007 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Skyline.pm,v 1.1 2002/07/30 17:44:27 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@g-language.org> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::Tools::RCluster;

use strict;

use Rcmd;
use SubOpt;
use G::Messenger;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);


require Exporter;

@ISA = qw(Exporter AutoLoader Rcmd);
@EXPORT = qw(
	     load_rcluster
);



#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::

sub load_rcluster{
    my $self = new Rcmd(@_);
    return bless ($self);
}

sub data{
    my $rcmd = shift;

    my @args = opt_get(@_);
    my $label = opt_val("label");

    my $flag = 0;
    foreach (@args){
	if ($flag == 0){
	    $flag ++;

	    $rcmd->array('rclust', @$_);
	}else{
	    $rcmd->array('tmp', @$_);
	    $rcmd->exec('rclust <- cbind(rclust, tmp)');
	}
    }

    if (length $label){
	$rcmd->sarray('label', @$label);
	$rcmd->exec('dimnames(rclust) <- list(label, NULL)');
    }
}


sub sample_data{
    my $rcmd = shift;

    $rcmd->exec(
		'rclust<-rbind(matrix(rnorm(100,sd=0.3),ncol=2),' .
		'matrix(rnorm(100,mean=1,sd=0.3),ncol=2))' 
		);
}

sub som{
    my $rcmd = shift;
    opt_default(filename=>"som.png", output=>"show");
    my @args = opt_get(@_);
    my $xdim = shift;
    my $ydim = shift;
    my $filename= opt_val("filename");
    my $output = opt_val("output");

    my $message = $rcmd->exec(
			      'require(GeneSOM)'
			      );

    die(
	"Package GeneSOM not installed.\nrun\n\tinstall.packages(\'GeneSOM\')\n" . 
	"in R as root.\n"
	) if ($message =~ /FALSE/);

    $rcmd->exec(
		"rclust.som<-som(rclust, $xdim, $ydim)",
		'rclust.som$visual'
		);

    my @result;
    open(FILE, $rcmd->{log}) || ($!);
    while(<FILE>){
	if(/qerror/){
	    @result = ();
	    while(<FILE>){
		chomp;
		if(/^\d/){
		    my (undef, $x, $y, $qerror) = split(/\s+/, $_, 4);
		    push(@result, [$x, $y, $qerror]);
		}
	    }
	}
    }
    close(FILE);

    if($output =~ /g/ || $output =~ /show/){
	$rcmd->exec(
		    'postscript("/tmp/som.ps")',
		    'plot(rclust.som)'
		    );

	mkdir('graph', 0777);
	system("convert -rotate 90 /tmp/som.ps graph/$filename");
	msg_gimv("graph/$filename") if ($output =~ /show/);
    }
    
    return @result;
}

sub hclust{
    my $rcmd = shift;
    opt_default("dist_method"=>"euclidean", "hclust_method"=>"com", 
		filename=>"hclust.png", output=>"show");
    my @args = opt_get(@_);
    my $dist_method = opt_val("dist_method");
    my $hclust_method = opt_val("hclust_method");
    my $filename= opt_val("filename");
    my $output = opt_val("output");

    $rcmd->exec(
		'require(mva)',
		"rclust.hclust<-hclust(dist(rclust, \"$dist_method\"), \"$hclust_method\")"
		);

    my @result = $rcmd->exec('rclust.hclust$merge');

    if($output =~ /g/ || $output =~ /show/){
	$rcmd->exec(
		    'postscript("/tmp/rclust.ps")',
		    'plot(rclust.hclust,hang=-1)'
		    );

	mkdir('graph', 0777);
	system("convert -rotate 90 /tmp/rclust.ps graph/$filename");
	msg_gimv("graph/$filename") if ($output =~ /show/);
    }

    shift @result;
    return @result;
}


sub kmeans{
    my $rcmd = shift;
    opt_default("iter.max"=>10, filename=>"kmeans.png", output=>"show");
    my @args = opt_get(@_);
    my $centers = shift @args;
    my $iter = opt_val("iter.max");
    my $output = opt_val("output");
    my $filename = opt_val("filename");

    $rcmd->exec(
		'require(mva)',
		"rclust.kmeans<-kmeans(rclust,$centers,$iter)"
		);

    my @result = $rcmd->exec('rclust.kmeans$cluster');

    if($output =~ /g/ || $output =~ /show/){
	$rcmd->exec(
		    'postscript("/tmp/kmeans.ps")',
		    'plot(rclust,col=rclust.kmeans$cluster)',
		    "points(rclust.kmeans\$centers, col=1:$centers,pch=8)"
		    );

	mkdir('graph', 0777);
	system("convert -rotate 90 /tmp/kmeans.ps graph/$filename");
	msg_gimv("graph/$filename") if ($output =~ /show/);
    }
    
    return @result;
}



1;

__END__

=head1 NAME

G::Tools::RCluster

=head1 SYNOPSIS

 use G::Tools::RCluster
 @ISA = (G::Skyline);
   
=head1 DESCRIPTION

 R wrapper for clustering.

=back

=head1 AUTHOR

Kazuharu Gaou Arakawa, gaou@g-language.org

=head1 SEE ALSO

perl(1).

=cut



