/*
    Copyright (C) 2006  Laurent Poirrier

    This file is part of YGL2.

    YGL2 is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    YGL2 is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

*/

#include <string.h>
#include "draw.h"

// ******************************************************
//
// ******************************************************
unsigned int Ytk24ToColor(unsigned int c)
{
	if (ygl.depth == 15)
		return( ((c&0xf80000)>>9) | ((c&0xf800)>>6) | ((c&0xf8)>>3) );
	if (ygl.depth == 16)
		return( ((c&0xf80000)>>8) | ((c&0xfc00)>>5) | ((c&0xf8)>>3) );
	return( c & 0xffffff);
}

unsigned int YtkRGBToColor(int r, int g, int b)
{
	//if (r&0x80000000) r = 0x00; else if (r & 0xffffff00) r = 0xff;
	//if (g&0x80000000) g = 0x00; else if (g & 0xffffff00) g = 0xff;
	//if (b&0x80000000) b = 0x00; else if (b & 0xffffff00) b = 0xff;
	
	if (ygl.depth == 15)
		return( ((r&0xf8)<<7) | ((g&0xf8)<<2) | ((b&0xf8)>>3) );
	if (ygl.depth == 16)
		return( ((r&0xf8)<<8) | ((g&0xfc)<<3) | ((b&0xf8)>>3) );
	return( ((r&0xff)<<16) | ((g&0xff)<<8) | (b&0xff) );
}

void YtkColorToRGB(int *r, int *g, int *b, unsigned int c)
{
	if (ygl.depth == 15) {
		*r = (c & 0x7c00) >> 7;
		*g = (c & 0x03e0) >> 2;
		*b = (c & 0x001f) << 3;
	} else if (ygl.depth == 16) {
		*r = (c & 0xf800) >> 7;
		*g = (c & 0x07e0) >> 3;
		*b = (c & 0x001f) << 3;
	} else {
		*r = (c>>16)&0xff;
		*g = (c>>8)&0xff;
		*b = c&0xff;
	}
}

// ******************************************************
//
// ******************************************************
unsigned int  YtkGetPixel(struct ytk_ws *ws, int x, int y)
{
	void *ptr;
	unsigned int r;
	
	if (!ws->data) return(0);
	if ((x<0)||(y<0)||(x>=ws->width)||(y>=ws->height)) return(0);
	if (ygl.bits_per_pixel == 32) {
		ptr = ws->data + (y*ws->linesize) + (x<<2);
		return(*((unsigned int *)ptr));
	}
	if (ygl.bits_per_pixel == 24) {
		ptr = ws->data + (y*ws->linesize) + x+x+x;
		r =  ((unsigned char *)ptr)[0];
		r |= ((unsigned char *)ptr)[1] << 8;
		r |= ((unsigned char *)ptr)[2] << 16;
		return(r);
	}
	if (ygl.bits_per_pixel == 16) {
		ptr = ws->data + (y*ws->linesize) + (x<<1);
		return(*((unsigned short *)ptr));
	}
	if (ygl.bits_per_pixel == 8) {
		ptr = ws->data + (y*ws->linesize) + x;
		return(*((unsigned char *)ptr));
	}
	return(0);
}

// ******************************************************
//
// ******************************************************
void YtkPutPixel(struct ytk_ws *ws, int x, int y, unsigned int c)
{
	void *ptr;
	
	if (!ws->data) return;
	if ((x<0)||(y<0)||(x>=ws->width)||(y>=ws->height)) return;
	if (ygl.bits_per_pixel == 32) {
		ptr = ws->data + (y*ws->linesize) + (x<<2);
		*((unsigned int *)ptr) = c;
		return;
	}
	if (ygl.bits_per_pixel == 24) {
		ptr = ws->data + (y*ws->linesize) + x+x+x;
		((unsigned char *)ptr)[0] = c;
		((unsigned char *)ptr)[1] = c >> 8;
		((unsigned char *)ptr)[2] = c >> 16;
		return;
	}
	if (ygl.bits_per_pixel == 16) {
		ptr = ws->data + (y*ws->linesize) + (x<<1);
		*((unsigned short *)ptr) = c;
		return;
	}
	if (ygl.bits_per_pixel == 8) {
		ptr = ws->data + (y*ws->linesize) + x;
		*((unsigned char *)ptr) = c;
		return;
	}
}

// ******************************************************
//
// ******************************************************
void YtkVLine(struct ytk_ws *ws, int x, int y1, int y2, unsigned int c)
{
	void *ptr;
	
	if (!ws->data) return;
	if ((x<0)||(x>=ws->width)) return;
	if (y1<0) y1=0; if (y2>=ws->height) y2=ws->height-1;
	if (y1>y2) return;
	if (y1==y2) { YtkPutPixel(ws, x, y1, c); return; }
	y2 = y2 - y1 + 1;
	
	if (ygl.bits_per_pixel == 32) {
		ptr = ws->data + (y1*ws->linesize) + (x<<2);
		while (y2) {
			*((unsigned int *)ptr) = c;
			ptr += ws->linesize;
			y2--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 24) {
		ptr = ws->data + (y1*ws->linesize) + x+x+x;
		while (y2) {
			((unsigned char *)ptr)[0] = c;
			((unsigned char *)ptr)[1] = c >> 8;
			((unsigned char *)ptr)[2] = c >> 16;
			ptr += ws->linesize;
			y2--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 16) {
		ptr = ws->data + (y1*ws->linesize) + (x<<1);
		while (y2) {
			*((unsigned short *)ptr) = c;
			ptr += ws->linesize;
			y2--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 8) {
		ptr = ws->data + (y1*ws->linesize) + x;
		while (y2) {
			*((unsigned char *)ptr) = c;
			ptr += ws->linesize;
			y2--;
		}
		return;
	}
}

void YtkHLine(struct ytk_ws *ws, int x1, int x2, int y, unsigned int c)
{
	void *ptr;
	
	if (!ws->data) return;
	if ((y<0)||(y>=ws->height)) return;
	if (x1<0) x1=0; if (x2>=ws->width) x2=ws->width-1;
	if (x1>x2) return;
	if (x1==x2) { YtkPutPixel(ws, x1, y, c); return; }
	x2 = x2 - x1 + 1;
	
	if (ygl.bits_per_pixel == 32) {
		ptr = ws->data + (y*ws->linesize) + (x1<<2);
		while (x2) {
			*((unsigned int *)ptr) = c;
			ptr += 4;
			x2--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 24) {
		ptr = ws->data + (y*ws->linesize) + x1+x1+x1;
		while (x2) {
			((unsigned char *)ptr)[0] = c;
			((unsigned char *)ptr)[1] = c >> 8;
			((unsigned char *)ptr)[2] = c >> 16;
			ptr += 3;
			x2--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 16) {
		ptr = ws->data + (y*ws->linesize) + (x1<<1);
		while (x2) {
			*((unsigned short *)ptr) = c;
			ptr += 2;
			x2--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 8) {
		ptr = ws->data + (y*ws->linesize) + x1;
		while (x2) {
			*((unsigned char *)ptr) = c;
			ptr++;
			x2--;
		}
		return;
	}
}

// ******************************************************
//
// ******************************************************
void YtkClear(struct ytk_ws *ws, unsigned int c)
{
	void *ptr, *lineptr;
	int lx, ly;
	
	if (!ws->data) return;
	if ((ws->width<=0)||(ws->height<=0)) return;
	lineptr = ws->data;
	ly = ws->height;
	if (ygl.bits_per_pixel == 32) {
		while (ly) {
			ptr = lineptr;
			lx = ws->width;
			while (lx) {
				*((unsigned int *)ptr) = c;
				ptr+=4;
				lx--;
			}
			lineptr += ws->linesize;
			ly--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 24) {
		while (ly) {
			ptr = lineptr;
			lx = ws->width;
			while (lx) {
				((unsigned char *)ptr)[0] = c;
				((unsigned char *)ptr)[1] = c >> 8;
				((unsigned char *)ptr)[2] = c >> 16;
				ptr += 3;
				lx--;
			}
			lineptr += ws->linesize;
			ly--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 16) {
		while (ly) {
			ptr = lineptr;
			lx = ws->width;
			while (lx) {
				*((unsigned short *)ptr) = c;
				ptr += 2;
				lx--;
			}
			lineptr += ws->linesize;
			ly--;
		}
		return;
	}
	if (ygl.bits_per_pixel == 8) {
		memset(lineptr, c, ws->size);
		return;
	}
}


// ******************************************************
//
// ******************************************************
void YtkRectangle(struct ytk_ws *ws, int x1, int y1, int x2, int y2,
	unsigned int c)
{
	if (x1<0) x1=0; if (x2>=ws->width ) x2=ws->width -1;
	if (x1>x2) return;
	if (x1==x2) { YtkVLine(ws, x1, y1, y2, c); return; }
	if (y1<0) y1=0; if (y2>=ws->height) y2=ws->height-1;
	if (y1>y2) return;
	
	while (y1<=y2) { YtkHLine(ws, x1, x2, y1, c); y1++; }
}


void YtkRectangleOutline(struct ytk_ws *ws, int x1, int y1, int x2, int y2,
	unsigned int c)
{
	if (x1>x2) return;
	if (y1>y2) return;
	
	YtkHLine(ws, x1, x2, y1, c);
	YtkHLine(ws, x1, x2, y2, c);
	YtkVLine(ws, x1, y1, y2, c);
	YtkVLine(ws, x2, y1, y2, c);
}


// ******************************************************
//
// ******************************************************
void YtkGradient(struct ytk_ws *ws, int x1, int y1, int x2, int y2,
	unsigned int c1, unsigned int c2, int vertical)
{
	unsigned int c;
	int r1, g1, b1, ra, ga, ba;
	
	if (x1<0) x1=0; if (x2>=ws->width ) x2=ws->width -1;
	if (x1>x2) return;
	if (y1<0) y1=0; if (y2>=ws->height) y2=ws->height-1;
	if (y1>y2) return;
	
	YtkColorToRGB(&r1, &g1, &b1, c1);
	YtkColorToRGB(&ra, &ga, &ba, c2);
	r1 <<= 16; g1 <<= 16; b1 <<= 16;
	if (vertical) {
		if (y2 != y1) {
			ra = ((ra<<16)-r1)/(y2-y1);
			ga = ((ga<<16)-g1)/(y2-y1);
			ba = ((ba<<16)-b1)/(y2-y1);
		}
	
		while (y1<=y2) {
			c = YtkRGBToColor(r1>>16, g1>>16, b1>>16);
			YtkHLine(ws, x1, x2, y1, c);
			r1 += ra; g1 += ga; b1 += ba;
			y1++;
		}
	} else {
		if (x2 != x1) {
			ra = ((ra<<16)-r1)/(x2-x1);
			ga = ((ga<<16)-g1)/(x2-x1);
			ba = ((ba<<16)-b1)/(x2-x1);
		}
	
		while (x1<=x2) {
			c = YtkRGBToColor(r1>>16, g1>>16, b1>>16);
			YtkVLine(ws, x1, y1, y2, c);
			r1 += ra; g1 += ga; b1 += ba;
			x1++;
		}
	}
}

// ******************************************************
// YtkLine
// ******************************************************
void YtkLine(struct ytk_ws *ws, int x1, int y1, int x2, int y2,
	unsigned int c)
{
	int t, dx, dy;
	int thick, nv;

	if (x2>x1) dx = x2-x1+1;
	else dx = x1-x2+1;

	if (y2>y1) dy = y2-y1+1;
	else dy=y1-y2+1;

	if (dx > dy) {
		if (x1 > x2) {
			t = x2; x2 = x1; x1 = t;
			t = y2; y2 = y1; y1 = t;
		}
		t = (y2 > y1) ? 1 : -1;

		thick = (dx << 16) / dy;
		x1 <<= 16;
		while ((x1>>16) < x2) {
			nv = x1 + thick;
			YtkHLine(ws, x1>>16,
				(nv>>16)-1, y1, c);
			y1 += t;
			x1 = nv;
		}
	} else {
		if (y1 > y2) {
			t = x2; x2 = x1; x1 = t;
			t = y2; y2 = y1; y1 = t;
		}
		t = (x2 > x1) ? 1 : -1;

		thick = (dy<<16) / dx;
		y1 <<= 16;
		while((y1>>16) < y2) {
			nv = y1+thick;
			YtkVLine(ws, x1, y1>>16, (nv>>16)-1, c);
			x1 += t;
			y1 = nv;
		}
	}
}
