/*
    Copyright (C) 2006  Laurent Poirrier

    This file is part of YGL2.

    YGL2 is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    YGL2 is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

*/

#include <GL/gl.h>
#include "mesh_base.h"

#define PI	3.14159265358979323846
#define SQRT2	1.41421356237309504880
#define SQRT3	1.73205080756

#define mesh_sincos(r_sin, r_cos, angle) \
	asm("fsincos" : "=t" (r_cos), "=u" (r_sin) : "0" (angle) )
#define mesh_sqrt(r_result, value) \
	asm("fsqrt" : "=t" (r_result) : "0" (value) )

// ********************************************
//
// ********************************************
static void normalize(float v[3])
{
	float n;

	mesh_sqrt(n, v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
	v[0] = v[0] / n;
	v[1] = v[1] / n;
	v[2] = v[2] / n;
}

static int tetra_faces[4][3] =
	{ { 0, 2, 1 }, { 0, 1, 3 }, { 1, 2, 3 }, { 2, 0, 3 } };
static float tetra_vertices[4][3] = {
	{ 2.0*SQRT2/3.0, -1.0/3.0, .0 },
	{ -SQRT2/3.0, -1.0/3.0, -SQRT2*SQRT3/3.0 },
	{ -SQRT2/3.0, -1.0/3.0, SQRT2*SQRT3/3.0 },
	{ .0, 1.0, .0 } };

static void subdivtri(float mul[3], float **v, int divlevel)
{
	float u[3][3];
	float *t[3];
	int i, j, d;

	if (divlevel <= 0) {
		for (i=0; i<3; i++) {
			glNormal3fv(v[i]);
			glVertex3f(v[i][0]*mul[0], v[i][1]*mul[1],
				v[i][2]*mul[2]);
		}
		return;
	}

	for (i=0; i<3; i++) {
		j = (i==2) ? 0 : (i+1);
		for (d=0; d<3; d++)
			u[i][d] = v[i][d] + v[j][d];
		normalize(u[i]);
	}

	for (i=0; i<3; i++) {
		j = (i==0) ? 2 : (i-1);
		t[0] = v[i]; t[1] = u[i]; t[2] = u[j];
		subdivtri(mul, t, divlevel-1);
	}
	t[0] = u[0]; t[1] = u[1]; t[2] = u[2];
	subdivtri(mul, t, divlevel-1);
}

void glYTetra(float mul[3], int divlevel)
{
	int f, i;
	float *v[3];

	glBegin(GL_TRIANGLES);
	for (f=0; f<4; f++) {
		for (i=0; i<3; i++)
			v[i] = tetra_vertices[tetra_faces[f][i]];
		subdivtri(mul, v, divlevel);
	}
	glEnd();
}

// ********************************************
//
// ********************************************
void glYDisk(float xmul, float zmul, float t, int ndiv)
{
	float costh, sinth, th, x, z;
	int i;

	glBegin(GL_TRIANGLE_FAN);
	glNormal3f(.0,t,.0);
	glVertex3f(.0,.0,.0);
	for (i=0; i<=ndiv; i++) {
		th = -2.0*t*PI*i/ndiv;
		mesh_sincos(sinth, costh, th);
		x = xmul * costh;
		z = zmul * sinth;
		
		glVertex3f(x,.0,z);
	}
	glEnd();

}

// ********************************************
//
// ********************************************
void glYCylinderSide(float xmul, float zmul, float t, float height,
	int ndiv)
{
	float th, costh, sinth, x, z;
	int i;

	glBegin(GL_TRIANGLE_STRIP);
	for (i=0; i<=ndiv; i++) {
		th = 2.0*t*PI*i/ndiv;
		mesh_sincos(sinth, costh, th);
		x = xmul * costh;
		z = zmul * sinth;
		glNormal3f(costh,0,sinth);
		glVertex3f(x,0,z);
		glVertex3f(x,height,z);
	}
	glEnd();
}

// ********************************************
//
// ********************************************
void glYCylinder(float xmul, float zmul, float height, int ndiv)
{
	glPushMatrix();
	glTranslatef(.0, height, .0);
	glYDisk(xmul, zmul, MESH_DISK_TOP, ndiv);
	glPopMatrix();
	glYDisk(xmul, zmul, MESH_DISK_BOTTOM, ndiv);
	glYCylinderSide(xmul, zmul, MESH_CYLINDER_EXT, height, ndiv);
}


// ********************************************
//
// ********************************************
#define swap(a, b)	{ t = a; a = b; b = t; }
void glYBrick(float x1, float y1, float z1, float x2, float y2, float z2)
{
	float t;

	if (x1>x2) swap(x1,x2);
	if (y1>y2) swap(y1,y2);
	if (z1>z2) swap(z1,z2);

	glBegin(GL_QUADS);
	glNormal3f(-1.0,.0,.0);
	glVertex3f(x1,y1,z2);
	glVertex3f(x1,y2,z2);
	glVertex3f(x1,y2,z1);
	glVertex3f(x1,y1,z1);

	glNormal3f(.0,.0,-1.0);
	glVertex3f(x1,y1,z1);
	glVertex3f(x1,y2,z1);
	glVertex3f(x2,y2,z1);
	glVertex3f(x2,y1,z1);

	glNormal3f(1.0,.0,.0);
	glVertex3f(x2,y1,z1);
	glVertex3f(x2,y2,z1);
	glVertex3f(x2,y2,z2);
	glVertex3f(x2,y1,z2);

	glNormal3f(.0,.0,1.0);
	glVertex3f(x2,y1,z2);
	glVertex3f(x2,y2,z2);
	glVertex3f(x1,y2,z2);
	glVertex3f(x1,y1,z2);

	glNormal3f(.0,-1.0,.0);
	glVertex3f(x1,y1,z1);
	glVertex3f(x2,y1,z1);
	glVertex3f(x2,y1,z2);
	glVertex3f(x1,y1,z2);

	glNormal3f(.0,1.0,.0);
	glVertex3f(x1,y2,z1);
	glVertex3f(x1,y2,z2);
	glVertex3f(x2,y2,z2);
	glVertex3f(x2,y2,z1);
	glEnd();
}

