/*
    Copyright (C) 2006  Laurent Poirrier

    This file is part of YGL2.

    YGL2 is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    YGL2 is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

*/

#include <stdlib.h>
#include <stdio.h>
#include "internal.h"
#include "softcsp.h"
#include "workspace.h"

#define PE(s...)	fprintf(stderr, s)

#define YTK_SHMMASK	(YTK_SHMPIXMAP|YTK_SHMIMAGE|YTK_XVSHMIMAGE)
#define YTK_XVMASK	(YTK_XVIMAGE|YTK_XVSHMIMAGE)

// ***************************************************************
// Xv and XShm images: colorspace formats
// ***************************************************************
// X11 and v4l2 standards differ
static int common_csp(int csp)
{
	if (csp == YTK_I420) return(YTK_YU12);
	if (csp == YTK_YUY2) return(YTK_YUYV);
	return(csp);
}

// ***************************************************************
// One Xv image per Xv port
// ***************************************************************
#define MAX_XV_PORTS	64
static int xv_one_per_port = 1;
static int xv_owned_ports[MAX_XV_PORTS];

void YtkInit_workspaces()
{
	int t;

	for (t=0; t<MAX_XV_PORTS; t++) xv_owned_ports[t] = -1;
}

void YtkSetXvExclusion(int yes)
{
	xv_one_per_port = (yes) ? -1 : 0;
}

static int self_grab_port(int portid)
{
	int t, n;

	if (!xv_one_per_port) return(0);

	n = -1;
	for (t=0; t<MAX_XV_PORTS; t++) {
		if (xv_owned_ports[t] == -1) n = t;
		else if (xv_owned_ports[t] == portid) return(-1);
	}
	if (n >= 0) xv_owned_ports[n] = portid;
	else PE("Workspace: Warning: too many Xv ports grabbed\n");

	return(0);
}

static void self_ungrab_port(int portid)
{
	int t;

	if (!xv_one_per_port) return;

	for (t=0; t<MAX_XV_PORTS; t++) {
		if (xv_owned_ports[t] == portid)
			xv_owned_ports[t] = -1;
	}
}

// ***************************************************************
// Try a port
// ***************************************************************
static int try_xv_port(struct ytk_ws *ws, int isshm, int portid)
{
	XvImageFormatValues *fmt;
	int nfmt, g, found;

	fmt = XvListImageFormats(ygl.display, portid, &nfmt);
	if (!fmt) {
		PE("Workspace: Unable to list image formats on port %d\n",
			portid);
		return(0);
	}

	found = 0;
	for (g=0; g<nfmt; g++) {
		if (common_csp(fmt[g].id) == ws->csp) found = fmt[g].id;
	}
	XFree(fmt);

	if (!found) return(0);

	if (self_grab_port(portid)) {
		PE("Workspace: Skipping port %d: already grabbed\n", portid);
		return(0);
	}
	if (XvGrabPort(ygl.display, portid, CurrentTime)) {
		self_ungrab_port(portid);
		PE("Workspace: Unable do grab Xv port %d\n", portid);
		return(0);
	}
	
	// Grrrrrr !!! :-(
	if (ygl_xv.ap_atom != None)
		XvSetPortAttribute(ygl.display, portid, ygl_xv.ap_atom, 1);
	
	if (isshm) {
		ws->i.xv.xvimage = XvShmCreateImage(
			ygl.display, portid, found,
			ws->shm.shmaddr, ws->width, ws->height, &ws->shm);
	} else {
		ws->i.xv.xvimage = XvCreateImage(
			ygl.display, portid, found,
			ws->data, ws->width, ws->height);
	}
	if (ws->i.xv.xvimage == None) {
		PE("Workspace: Unable to create Xv%sImage "
		"on port %d\n", isshm?"Shm":"", portid);
		XvUngrabPort(ygl.display, portid, CurrentTime);
		self_ungrab_port(portid);
		return(0);
	}

	ws->i.xv.portid = portid;
	PE("Workspace: Xv%sImage on port %d\n", isshm?"Shm":"", portid);
	if (isshm) {
		ws->data = ws->shm.shmaddr;
		ws->type = YTK_XVSHMIMAGE;
	} else {
		ws->type = YTK_XVIMAGE;
	}

	return(1);
}

// ***************************************************************
// Try Xv
// ***************************************************************
int try_xv(struct ytk_ws *ws, int isshm)
{
	XvAdaptorInfo *adapt;
	unsigned int nadapt, t, p;

	if (XvQueryAdaptors(ygl.display, ygl.root_window, &nadapt, &adapt)) {
		PE("Workspace: can't get xv adaptors info\n");
		return(0);
	}
	
	if (nadapt <= 0) return(0);
		
	for (t=0; t < nadapt; t++) {
	for (p=adapt[t].base_id; p<adapt[t].base_id+adapt[t].num_ports; p++) {
		if (try_xv_port(ws, isshm, p)) {
			XvFreeAdaptorInfo(adapt);
			return(1);
		}
	}
	}
	XvFreeAdaptorInfo(adapt);
	PE("Workspace: Unable to create Xv%sImage on any port\n",
		isshm?"Shm":"");
	return(0);
}

// ***************************************************************
//
// ***************************************************************
int  YtkCreateWorkspace(struct ytk_ws *ws, int type, int w, int h, int csp)
{
	int lsz, sz;
	
	// Colorspace asserts
	if (type & YTK_CSPIMAGE) {
		if (csp == YTK_DISPLAY) return(-1);
		if (type & ~YTK_CSPIMAGE) return(-1);
	} else {
		if (csp != YTK_DISPLAY) return(-1);
	}

	// Image size calculations
	if (csp==YTK_DISPLAY) {
		lsz = w*(ygl.bits_per_pixel>>3);
		lsz = (lsz+(ygl.scanline_pad>>3)-1)
			& (~((ygl.scanline_pad>>3)-1));
		sz = h*lsz;
	} else if ((csp==YTK_YV12)||(csp==YTK_YU12)||(csp==YTK_I420)) {
		lsz = w;
		sz = (w*h) + (((w>>1)*(h>>1))<<1);
	} else if ((csp==YTK_YUY2)||(csp==YTK_YUYV)||(csp==YTK_UYVY)) {
		lsz = w << 1;
		sz = lsz * h;
	} else {
		// Could say: lsz = w << 1; sz = lsz * (h << 1);
		return(-1);
	}

	ws->width = w; ws->height = h;
	ws->linesize = lsz; ws->size = sz;
	ws->csp = common_csp(csp);

	if (!ygl_xshm.zpixmaps) type &= ~YTK_SHMMASK;
	if (!ygl_xv.present) type &= ~YTK_XVMASK;

	if (type & YTK_SHMMASK) {
		if (YShmAlloc(&ws->shm, sz)) {
			PE("Workspace: Shared memory allocation failed\n");
			type &= ~YTK_SHMMASK;
			return(YtkCreateWorkspace(ws, type, w, h, csp));
		}

		// ********** XvShmImage
		if (type & YTK_XVSHMIMAGE) {
			if (try_xv(ws, 1)) return(0);
		}

		// ********** XShmPixmap
		if (type & YTK_SHMPIXMAP) {
			ws->i.pixmap = XShmCreatePixmap(ygl.display,
				ygl.root_window, ws->shm.shmaddr,
				&ws->shm, w, h, ygl.depth);
			if (ws->i.pixmap != None) {
				ws->data = ws->shm.shmaddr;
				ws->type = YTK_SHMPIXMAP;
				PE("Workspace: XShmPixmap\n");
				return(0);
			}
			PE("Workspace: Unable to create XShmPixmap\n");
		}

		// ********** XShmImage
		if (type & YTK_SHMIMAGE) {
			ws->i.image = XShmCreateImage(ygl.display,
				ygl.default_visual, ygl.depth, ZPixmap,
				ws->shm.shmaddr, &ws->shm, w, h);
			if (ws->i.image != NULL) {
				ws->data = ws->shm.shmaddr;
				ws->type = YTK_SHMIMAGE;
				PE("Workspace: XShmImage\n");
				return(0);
			}
			PE("Workspace: Unable to create XShmImage\n");
		}
		YShmFree(&ws->shm);
	}

	// ********** XvImage
	if (type & YTK_XVIMAGE) {
		ws->data = malloc(sz);
		if (ws->data) {
			if (try_xv(ws, 0)) return(0);
			free(ws->data);
		} else PE("Workspace: Unable to allocate memory for "
			"Xv image data\n");
	}

	// ********** XPixmap
	if (type & YTK_PIXMAP) {
		ws->i.pixmap = XCreatePixmap(ygl.display, ygl.root_window,
			w, h, ygl.depth);
		if (ws->i.pixmap != None) {
			ws->data = NULL;
			ws->type = YTK_PIXMAP;
			PE("Workspace: X Pixmap\n");
			return(0);
		}
		PE("Workspace: Unable to create X Pixmap\n");
	}
	
	// ********** XImage
	if (type & YTK_IMAGE) {
		ws->data = malloc(sz);
		if (ws->data) {
			ws->i.image = XCreateImage(ygl.display,
				ygl.default_visual, ygl.depth, ZPixmap, 0,
				ws->data, w, h, 8, lsz);
			if (ws->i.image != NULL) {
				ws->type = YTK_IMAGE;
				PE("Workspace: XImage\n");
				return(0);
			}
			free(ws->data);
			PE("Workspace: Unable to create XImage\n");
		} else PE("Workspace: Unable to allocate memory for "
			"X image data\n");
	}

	// ********** Software colorspace converter
	if ((type & YTK_SOFTCSP)
	&& (YtkSoftcspSupported(YTK_DISPLAY, ws->csp))) {
		ws->i.softcsp = malloc(sizeof(struct ytk_ws) + sz);
		if (ws->i.softcsp) {
			ws->data = (void *)ws->i.softcsp
				+ sizeof(struct ytk_ws);
			if (YtkCreateWorkspace(ws->i.softcsp, YTK_RGBIMAGE,
			w, h, YTK_DISPLAY) == 0) {
				ws->type = YTK_SOFTCSP;
				PE("Workspace: Software colorspace "
					"translator\n");
				return(0);
			}
			free(ws->i.softcsp);
			PE("Workspace: Unable to create rgb workspace\n");
		} else PE("Workspace: Unable to allocate memory for "
			"SoftCsp image data\n");
	}

	// ********** Not found
	return(-1);
}

// ***************************************************************
//
// ***************************************************************
void YtkDestroyWorkspace(struct ytk_ws *ws)
{
	if (ws->type == YTK_SOFTCSP) {
		YtkDestroyWorkspace(ws->i.softcsp);
		free(ws->i.softcsp); // includes data
		return;
	}

	if (ws->type & (YTK_PIXMAP|YTK_SHMPIXMAP))
		XFreePixmap(ygl.display, ws->i.pixmap);
	if (ws->type & (YTK_IMAGE|YTK_SHMIMAGE))
		XFree(ws->i.image);
	if (ws->type & (YTK_XVIMAGE|YTK_XVSHMIMAGE)) {
		XvUngrabPort(ygl.display, ws->i.xv.portid, CurrentTime);
		self_ungrab_port(ws->i.xv.portid);
		XFree(ws->i.xv.xvimage);
	}

	if (ws->type & YTK_SHMMASK)
		YShmFree(&ws->shm);
	if (ws->type & (YTK_IMAGE|YTK_XVIMAGE)) free(ws->data);
}

// ***************************************************************
//
// ***************************************************************
void YtkPutWorkspace(struct ytk_ws *ws, Drawable d, int dx, int dy,
	int sx, int sy, int w, int h)
{
	if (ws->type & YTK_SCALABLE) {
		YtkPutExtWorkspace(ws, d, dx, dy, w, h, sx, sy, w, h);
		return;
	}

	if ((ws->type == YTK_PIXMAP) || (ws->type == YTK_SHMPIXMAP)) {
		XCopyArea(ygl.display, ws->i.pixmap, d,
			ygl.default_gc, sx, sy, w, h, dx, dy);
		return;
	}
	if (ws->type == YTK_SHMIMAGE) {
		XShmPutImage(ygl.display, d, ygl.default_gc,
			ws->i.image, sx, sy, dx, dy, w, h, False);
		return;
	}
	if (ws->type == YTK_IMAGE) {
		XPutImage(ygl.display, d, ygl.default_gc,
			ws->i.image, sx, sy, dx, dy, w, h);
		return;
	}
	if (ws->type == YTK_SOFTCSP) {
		YtkSoftcsp(ws->i.softcsp, ws, sx, sy, w, h);
		YtkPutWorkspace(ws->i.softcsp, d, dx, dy, sx, sy, w, h);
		return;
	}
}

// ***************************************************************
//
// ***************************************************************
void YtkPutExtWorkspace(struct ytk_ws *ws, Drawable d,
	int dx, int dy, int dw, int dh, int sx, int sy, int sw, int sh)
{
	if (ws->type & (~YTK_SCALABLE)) {
		// Assert...
		YtkPutWorkspace(ws, d, dx, dy, sx, sy, sw, sh);
		return;
	}

	if (ws->type == YTK_XVSHMIMAGE) {
		XvShmPutImage(ygl.display, ws->i.xv.portid, d,
			ygl.default_gc, ws->i.xv.xvimage,
			sx, sy, sw, sh, dx, dy, dw, dh, False);
		return;
	}
	if (ws->type == YTK_XVIMAGE) {
		XvPutImage(ygl.display, ws->i.xv.portid, d,
			ygl.default_gc, ws->i.xv.xvimage,
			sx, sy, sw, sh, dx, dy, dw, dh);
		return;
	}
}

