#include <time.h>
#include <stdio.h>
#include <string.h>
#include "rw.h"
#include "timer.h"
#include "driver_main.h"
#include "driver_log.h"
#include "driver_embla_cmd.h"

int embla_sync(int n)
{
	char s;

	while(n > 0) {
		s = 0x05;
		if (bwrite(in_embla_fd, &s, 1)) return(-1);
		log_data(0, 1, &s, 1);
		if (btread(in_embla_fd, &s, 1, 2000)) return(-1);
		log_data(1, 1, &s, 1);
		if (s != 0x06) return(-1);
		n--;
	}
	return(0);
}

static char embla_buff[256];
int embla_send(char *cmd)
{
	int l;
	
	l = strlen(cmd);
	if (l+1 > sizeof(embla_buff)) {
		PE("BUG: Embla command too long (%d/%d)\n",
			l+1, sizeof(embla_buff));
		return(-1);
	}
	if ((l) && (cmd != embla_buff)) memcpy(embla_buff, cmd, l);
	embla_buff[l] = 0x0d;
	if (bwrite(in_embla_fd, embla_buff, l+1)) return(-1);
	log_data(0, 1, embla_buff, l+1);
	return(0);
}
#define embla_sendf(s...)	( \
	snprintf(embla_buff, sizeof(embla_buff), s), \
	embla_buff[sizeof(embla_buff)-1] = 0, \
	embla_send(embla_buff) )

char *embla_recv()
{
	int l, c;
	
	l = 0;
	embla_buff[0] = 0;
	while(1) {
		c = 0;
		if (btread(in_embla_fd, &c, 1, 5000)) return(0);
		if (c == 0x0d) {
			if (l+1 > sizeof(embla_buff))
				PE("WARNING: Embla reply too long\n");
			embla_buff[l] = 0x0d;
			log_data(1, 1, embla_buff, l+1);
			embla_buff[l] = 0;
			return(embla_buff);
		}
		if (l+1 <= sizeof(embla_buff)) {
			embla_buff[l] = c;
			embla_buff[l+1] = 0;
		}
		l++;
	}
}

#define SC_NSYNC_MASK	0x00ff
#define SC_ENDCMD	0x0100
#define SC_DONTWARN	0x0200
#define SC_MANDATORY	0x0400
struct embla_cmd {
	int syncs;
	char *cmd;
	char *std_reply;
};

static struct embla_cmd detect[] = {
	{ 0, "", 0 },
	{ 2, "!V0", "EmblaPC  Version 5.04 (Jan 28 2000)" },
	{ 1, "!V40", "16" },
	{ 1, "!V56", "EmblaDSP=4.10 EmblaMC=6.33 EmblaOS=0.70" },
	{ 2, "!V55", "EmblaPCB=5.00 " },
	{ 1, "!V58", "EG301/1" },
	{ 1, "!V610", "2755" },
	{ 1, "!V600", "50000" },
	{ 1, "!V60 1", "SunDisk" },
	{ 1, "!V201", "AY-5.12" },
	{ 1, "!V2", "AMBULATORY" },
	{ 1, "!V204", "F060043-AY" },
	{ 1|SC_DONTWARN, "!V210", "523" },
	{ 1, "!V208", "2000/11/6" },
	{ 1, "!V16", "0" },
	{ 0|SC_ENDCMD, 0, 0 }
};

int embla_detect()
{
	char *s;
	int t, n, differ;
	
	t=0;
	while(1) {
		n = detect[t].syncs;
		if (n & SC_ENDCMD) {
			logf("Detection successful\n");
			return(0);
		}
		
		if (embla_sync(n & SC_NSYNC_MASK)) return(-1);
		if (detect[t].cmd) {
			if (embla_send(detect[t].cmd)) return(-1);
		}
		if (detect[t].std_reply) {
			s = embla_recv();
			if (!s) return(-1);
			differ = strcmp(detect[t].std_reply, s);
		} else differ = 0;
		
		if ((differ) && (!(n & SC_DONTWARN)))
			logf("hardware differs (%s)\n<%s\n>%s\n",
				detect[t].cmd, detect[t].std_reply, s);
		if ((differ) && (n & SC_MANDATORY)) {
			PE("Hardware error (on mandatory reply)\n");
			return(-1);
		}
		t++;
	}
}

int embla_impedance_start()
{
	if (embla_detect()) return(-1);
	if (embla_send("")) return(-1);
	if (embla_sync(1)) return(-1);
	if (embla_send("!Y RMAX=16.7 1+ 2+ 3+ 4+ 5+ 6+ 7- 8- 9- 10- "
		"11- 12- 13- 14- 15- 16-")) return(-1);
	return(0);
}

int embla_impedance_stop()
{
	char *s;
	int c;
	
	c = 0x15;
	if (bwrite(in_embla_fd, &c, 1)) return(-1);
	log_data(0, 1, &c, 1);
	c = 0;
	while (c < 10) {
		c++;
		s = embla_recv();
		if (!s) return(-1);
		if (!strcmp(s, "DONE")) c = 100;
	}
	if (embla_sync(1)) return(-1);
	if (c!=100) return(-1);
	return(0);
}

int embla_acquire_start()
{
	char *s;
	time_t now;
	struct tm *bd;
	int c;
	
	// Detect & setup
	if (embla_detect()) return(-1);
	if (embla_sync(1)) return(-1);
	if (embla_send("")) return(-1);
	if (embla_sync(1)) return(-1);
	now = time(0);
	bd = localtime(&now);
	if (embla_sendf("!V11 %d/%d/%d-%d:%d:%d",
		bd->tm_year+1900, bd->tm_mon+1, bd->tm_mday,
		bd->tm_hour, bd->tm_min, bd->tm_sec)) return(-1);
	if (embla_send("")) return(-1);
	s = embla_recv();
	if (!s) return(-1);
	if (strcmp(s, "OK")) return(-1);
	
	if (embla_send("!D ALL RESET")) return(-1);
	if (embla_sync(1)) return(-1);
	if (embla_send("!S ALL RESET")) return(-1);
	if (embla_sync(1)) return(-1);
	
	for (c=0; c<=3; c++) {
		if (embla_sendf("!D%d OPEN DCRJ+ PWRJ+ DCDL+", c+1))
			return(-1);
		if (embla_sendf("!D%d SAMP=100000 SCALE=10", c+1))
			return(-1);
		if (embla_sync(1)) return(-1);
	}
	for (c=4; c<=5; c++) {
		if (embla_sendf("!D%d OPEN DCRJ+ PWRJ+ DCDL+", c+1))
			return(-1);
		if (embla_sendf("!D%d SAMP=200000", c+1))
			return(-1);
		if (embla_sync(1)) return(-1);
	}
	
	if (embla_send("!D EVENT CLOSE")) return(-1);
	if (embla_sync(2)) return(-1);
	
	if (embla_send("")) return(-1);
	if (embla_sync(1)) return(-1);
	if (embla_send("!B P160 BAUDRATE=57600")) return(1);
	s = embla_recv();
	if (!s) return(-1);
	if (strcmp(s, "0006")) return(-1);
	s = embla_recv();
	if (!s) return(-1);
	if (strcmp(s, "P160 BAUDRATE=57600")) return(-1);
	logf("Data mode started successfully\n");
	return(0);
}

