#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include "timer.h"
#include "socket_utils.h"
#include "driver_log.h"
#include "driver_sniffer.h"
#include "driver_master.h"
#include "driver_main.h"

#define PO(s...)	printf(s)

// ***********************************************************
// Global variables
// ***********************************************************
#define DRIVER_MASTER	0
#define DRIVER_SNIFFER	1
#define DRIVER_PSEUDO	2
int driver_mode = DRIVER_MASTER;

int out_pseudo_fd=-1, out_cooked_fd=-1;
int in_embla_fd=-1, in_somno_fd=-1;

// ***********************************************************
// Help message
// ***********************************************************
#define DEFAULT_EMBLA	"/dev/ttyS0"
#define DEFAULT_SOMNO	"/dev/ttyS1"
#define DEFAULT_PORT	18050
#define STRINGIFY(s)	#s
#define PORT_STRING	STRINGIFY(DEFAULT_PORT)
static char *help_msg =
"Usage: embla-driver {mode} [options]\n"
"Mode:\n"
"  -m, --master            stand-alone mode (default)\n"
"  -s, --sniffer           let third-party software drive the device\n"
"  -i, --ipseudo <file>    simulate device with given file\n"
"Options:\n"
"  -e, --embla <dev>       device to drive (default " DEFAULT_EMBLA ")\n"
"  -t, --somno <dev>       device running third-party software\n"
"                            (default " DEFAULT_SOMNO ")\n"
"  -o, --opseudo <file>    create simulator file\n"
"  -c, --cooked <addr>     address to use for output & control\n"
"  -p, --port <port>       port number (default " PORT_STRING ")\n"
"Messages:\n"
"      --silent            no verbosity\n"
"      --verbose           normal verbosity (default)\n"
"      --log               extensive log\n"
"      --timelog           timing log\n";

// ***********************************************************
// Command line arguments
// ***********************************************************
#define ARG_MARK(n)	used_args[(n)>>5] |= 1<<((n)&31);
#define ARG_NOT_USED(n)	!(used_args[(n)>>5] & (1<<((n)&31)))
static int used_args[8];
static int arg_count, arg_cur;
static char **arg_v;

static int arg_has(char *n1, char *n2)
{
	int t;
	
	if (n1 == NULL) {
		n1 = n2;
		if (n1 == NULL) return(0);
	} else if (n2 == NULL) n2 = n1;
	
	for (t=1; t<arg_count; t++)
	if ((!strcmp(arg_v[t], n1))||(!strcmp(arg_v[t], n2))) {
		ARG_MARK(t);
		arg_cur = t;
		return(1);
	}
	return(0);
}

static char *arg_value()
{
	if (arg_cur+1 >= arg_count) {
		PO("Value expected after %s\n"
			"Try embla-driver --help\n", arg_v[arg_cur]);
		exit(1);
	}
	ARG_MARK(arg_cur+1);
	return(arg_v[arg_cur+1]);
}

static int arg_port()
{
	char *s;
	char *e;
	int r;

	s = arg_value();
	r = strtol(s, &e, 0);
	if ((e==s)||(*e!=0)||(r<0)||(r>=65536)) {
		PO("Port should be in the range [0;65535]\n");
		exit(1);
	}
	return(r);
}

static int arg_open(char *s, int arw)
{
	int fd;
	
	if (arw == 1) fd = open(s, O_RDONLY);
	else if (arw == 2) fd = open(s, O_CREAT|O_EXCL|O_WRONLY,
		S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
	else fd = open(s, O_RDWR);
	
	if (fd<0) {
		PO("Can't open %s: %s\n", s, strerror(errno));
		exit(1);
	}
	return(fd);
}

// ***********************************************************
// Main
// ***********************************************************
int main(int argc, char **argv)
{
	char *socket_torm = NULL;
	int port = DEFAULT_PORT;
	int t, g;
	
	if (argc >= 256) {
		PO("%s", help_msg);
		return(1);
	}
	for (t=0; t<8; t++) used_args[t] = 0;
	arg_count = argc;
	arg_v = argv;
	arg_cur = 0;
	
	// Help
	if (arg_has("-h", "--help")) {
		PO("%s", help_msg);
		return(0);
	}
	
	// Mode
	if (arg_has("-s", "--sniffer")) driver_mode = DRIVER_SNIFFER;
	else if (arg_has("-m", "--master")) driver_mode = DRIVER_MASTER;
	else if (arg_has("-i", "--ipseudo")) {
		driver_mode = DRIVER_PSEUDO;
		in_embla_fd = arg_open(arg_value(), 1);
	}
	
	// Files
	if ((driver_mode==DRIVER_MASTER)||(driver_mode==DRIVER_SNIFFER)) {
		if (arg_has("-e", "--embla"))
			in_embla_fd = arg_open(arg_value(), 0);
		else in_embla_fd = arg_open(DEFAULT_EMBLA, 0);
	}
	if (driver_mode==DRIVER_SNIFFER) {
		if (arg_has("-t", "--somno"))
			in_somno_fd = arg_open(arg_value(), 0);
		else in_somno_fd = arg_open(DEFAULT_SOMNO, 0);
	}
	if (arg_has("-o", "--opseudo"))
		out_pseudo_fd = arg_open(arg_value(), 2);
	if (arg_has("-c", "--cooked")) socket_torm = arg_value();
	if (arg_has("-p", "--port")) port = arg_port();
	
	// Messages
	if (arg_has(NULL, "--silent")) dump_type = DUMP_NONE;
	else if (arg_has(NULL, "--verbose")) dump_type = DUMP_NORMAL;
	else if (arg_has(NULL, "--log")) dump_type = DUMP_LOG;
	else if (arg_has(NULL, "--timelog")) dump_type = DUMP_TIME;

	// Check arguments
	if (!socket_torm) {
		if (driver_mode==DRIVER_MASTER) {
			PO("Error (master): no socket specified\n");
			return(1);
		}
		PO("Warning: no socket specified\n");
	}
	
	g = 0;
	for (t=1; t<argc; t++) {
		if (ARG_NOT_USED(t)) {
			PO("Wrong argument: %s\n", argv[t]);
			g++;
		}
	}
	if (g) {
		PO("Try embla-driver --help\n");
		return(1);
	}
	
	// Start operation
	timer_init();
	log_init();
	logf("embla-driver: startup OK, mode: %s\n",
		(driver_mode==DRIVER_MASTER)?"master":
		(driver_mode==DRIVER_SNIFFER)?"sniffer":"pseudo");
	
	if (socket_torm) {
		out_cooked_fd = socket_create_accept(socket_torm,port);
		if (out_cooked_fd < 0) return(1);
		logf("embla-driver: connected\n");
	}
	
	if (driver_mode == DRIVER_MASTER) t = driver_master();
	else if (driver_mode == DRIVER_SNIFFER) t = driver_sniffer();
	else if (driver_mode == DRIVER_PSEUDO) t = driver_pseudo();
	
	// Remove control socket
	if (out_cooked_fd >= 0) close(out_cooked_fd);
	if (out_pseudo_fd >= 0) close(out_pseudo_fd);
	if (in_embla_fd >= 0) close(in_embla_fd);
	if (in_somno_fd >= 0) close(in_somno_fd);
	
	return(t);
}
