#include <stdio.h>
#include "pnm.h"

// ********************************************
// PBM/PGM/PPM blanks
// ********************************************
static int pnm_blanks_c(char **src, int *max)
{
	char *s = *src;
	
	if (*max < 1) return(-1);
	if ((*s!=' ')&&(*s!='\n')&&(*s!='\t')&&(*s!='\r')&&(*s!='#'))
		return(-1);
	while (*max > 0) {
		if (*s=='#') {
			do {
				s++; (*max)--;
				if (*max <= 0) return(-1);
			} while (*s != '\n');
		} else if ((*s!=' ')&&(*s!='\n')&&(*s!='\t')&&(*s!='\r')) {
			*src = s;
			return(0);
		}
		s++; (*max)--;
	}
	return(-1);
}

static int pnm_blanks(char **src, int *max)
{
	char *s = *src;

	
	if (*max <= 0) return(-1);
	if ((*s!=' ')&&(*s!='\n')&&(*s!='\t')&&(*s!='\r'))
		return(-1);
	while (*max > 0) {
		if ((*s!=' ')&&(*s!='\n')&&(*s!='\t')&&(*s!='\r')) {
			*src = s;
			return(0);
		}
		s++; (*max)--;
	}
	return(-1);
}

static int pnm_int(unsigned int *r, char **src, int *max)
{
	char *s = *src;

	*r = 0;
	while(*max>0) {
		if ((*s<'0')||(*s>'9')) {
			if (*src==s) return(-1);
			*src = s;
			return(0);
		}
		*r *= 10;
		*r += *s - '0';
		s++; (*max)--;
	}
	return(-1);
}

// ********************************************
// Generic info
// ********************************************
char *pnm_info(char *src, int *w, int *h, int *maxval, int *p, int sz)
{
	unsigned int i;
	int type;
	
	if (sz < 8) return(NULL);
	
	if (src[0] != 'P') return(NULL);
	type = src[1];
	if ((type < '1') && (type > '6')) return(NULL);
	if (p) *p = type - '0';
	src += 2; sz -= 2;

	if (pnm_blanks_c(&src, &sz)) return(NULL);
	
	if (pnm_int(&i, &src, &sz)) return(NULL);
	if (w) *w = i;

	if (pnm_blanks_c(&src, &sz)) return(NULL);
	
	if (pnm_int(&i, &src, &sz)) return(NULL);
	if (h) *h = i;
	
	if ((type == '1') || (type == '4')) {
		if (maxval) *maxval = 1;
	} else {
		if (pnm_blanks_c(&src, &sz)) return(NULL);
		
		if (pnm_int(&i, &src, &sz)) return(NULL);
		if ((i<=0)||(i>=65536)) return(NULL);
		if (maxval) *maxval = i;
	}
	
	if ((*src!=' ')&&(*src!='\n')&&(*src!='\t')&&(*src!='\r'))
		return(NULL);
	return(src+1);
}

// ********************************************
// Helper functions
// ********************************************
int pnm_width(char *src, int sz)
{
	int r;

	if (!pnm_info(src, &r, NULL, NULL, NULL, sz)) return(0);
	return(r);
}

int pnm_height(char *src, int sz)
{
	int r;
	
	if (!pnm_info(src, NULL, &r, NULL, NULL, sz)) return(0);
	return(r);
}

int pnm_native_csp(char *src, int sz)
{
	int m, p;

	if (!pnm_info(src, NULL, NULL, &m, &p, sz)) return(M_NONE);
	if ((p==1)||(p==4)) return(M_BITMAP_BE);
	else if ((p==2)||(p==6)) {
		if (m<256) return(M_GRAYSCALE8);
		else return(M_GRAYSCALE16);
	} else {
		if (m<256) return(M_RGB);
		else return(M_RRGGBB);
	}
	return(M_NONE);

}

// ********************************************
// Bitmap mem alignment
// ********************************************
static int pbm_mask = 7, pnm_mask = 0;
int pnm_byte_align(int bytes)
{
	int b = bytes;
	while (b > 1) { if (b & 1) return(-1); b >>= 1; }
	
	pbm_mask = (bytes <= 0) ? 0 : (bytes << 3)-1;
	pnm_mask = (bytes <= 0) ? 0 : bytes-1;
	return(0);
}

static int pnm_pad(int csp, int w)
{
	int pad=0;
	if ((csp==M_BITMAP_LE)||(csp==M_BITMAP_BE)) {
		pad = ( ((w+pbm_mask)&(~pbm_mask)) - w )>>8;
	} else if (csp == M_GRAYSCALE8)
		pad = ((w+pnm_mask)&(~pnm_mask)) - w;
	else if ((csp==M_GRAYSCALE16)||(csp==M_RGB15)||(csp==M_RGB16))
		pad = (( (w<<1) +pnm_mask)&(~pnm_mask)) - (w<<1);
	else if ((csp == M_RGB) || (csp == M_BGR))
		pad = (( (w*3) +pnm_mask)&(~pnm_mask)) - (w*3);
	else if ((csp==M_RGBA)||(csp==M_XXXA)||(csp==M_BGRA))
		pad = (( (w<<2) +pnm_mask)&(~pnm_mask)) - (w<<2);
	else if (csp == M_RRGGBB)
		pad = (( (w*6) +pnm_mask)&(~pnm_mask)) - (w*6);
	else if ((csp == M_RRGGBBAA) || (csp == M_XXXXXXAA))
		pad = (( (w<<3) +pnm_mask)&(~pnm_mask)) - (w<<3);
	return(pad);
}

// ********************************************
// Reader
// ********************************************
int pnm_read(void *dst, int csp, char *src, int size)
{
	char *s;
	int w, n, mv,c, type, pad;
	unsigned int srcshift, dstshift;
	unsigned int r, g, b;

	s = pnm_info(src, &w, &n, &mv, &type, size);
	if (!s) return(-1);
	
	n *= w; c = w;
	
	size -= s-src;
	if ((n > 0) && (size < 1)) return(-1);
	
	if (type == 4) srcshift = 128;
	if (type <= 3) { s--; size++; }
	
	if (csp == M_BITMAP_LE) dstshift = 1;
	else if (csp == M_BITMAP_BE) dstshift = 128;
	else if (csp == M_XXXA) dst += 3;
	else if (csp == M_XXXXXXAA) dst += 6;
	
	pad = pnm_pad(csp, w);
	
	while (n > 0) {
		c--;
		if (type==4) {
			r = (*s & srcshift);
			if (r) {
				r=g=b=0;
			} else r=g=b=65535;
			srcshift >>= 1;
			if ((srcshift == 0)||(c==0)) {
				srcshift = 128;
				s++;
				size--;
				if (size<=0) return(-1);
			}
		} else if (type==5) {
			if (mv < 256) {
				if (size<1) return(-1);
				r = *((unsigned char *)s);
				s++;
				size--;
			} else {
				if (size<2) return(-1);
				r = *((unsigned short *)s);
				s+=2;
				size -= 2;
			}
			r=g=b= r*65535/mv;
		} else if (type==6) {
			if (mv < 256) {
				if (size<3) return(-1);
				size-=3;
				r = *((unsigned char *)s) *65535/mv; s++;
				g = *((unsigned char *)s) *65535/mv; s++;
				b = *((unsigned char *)s) *65535/mv; s++;
			} else {
				if (size<6) return(-1);
				size-=6;
				r = *((unsigned short *)s) *65535/mv; s+=2;
				g = *((unsigned short *)s) *65535/mv; s+=2;
				b = *((unsigned short *)s) *65535/mv; s+=2;
			}
		} else if (type==1) {
			if (pnm_blanks(&s, &size)) return(-1);
			if (*s == '1') r=g=b=0;
			else if (*s == '0') r=g=b=65535;
			else return(-1);
			s++; size--;
		} else if (type==2) {
			if (pnm_blanks(&s, &size)) return(-1);
			if (pnm_int(&r, &s, &size)) return(-1);
			r=g=b= r*65535/mv;
		} else if (type==3) {
			if (pnm_blanks(&s, &size)) return(-1);
			if (pnm_int(&r, &s, &size)) return(-1);
			r = r * 65535 / mv;
			if (pnm_blanks(&s, &size)) return(-1);
			if (pnm_int(&g, &s, &size)) return(-1);
			g = g * 65535 / mv;
			if (pnm_blanks(&s, &size)) return(-1);
			if (pnm_int(&b, &s, &size)) return(-1);
			b = b * 65535 / mv;
		}

		if (csp==M_BITMAP_LE) {
			if (g & 32768) *((unsigned char *)dst) |= dstshift;
			else *((unsigned char *)dst) &= ~dstshift;
			dstshift <<= 1;
			if (dstshift>128) {
				dstshift = 1;
				dst++;
			}
		} else if (csp==M_BITMAP_BE) {
			if (g & 32768) *((unsigned char *)dst) |= dstshift;
			else *((unsigned char *)dst) &= ~dstshift;
			dstshift >>= 1;
			if (dstshift == 0) {
				dstshift = 128;
				dst++;
			}
		} else if (csp==M_GRAYSCALE8) {
			*((unsigned char *)dst) = g >> 8; dst++;
		} else if (csp==M_GRAYSCALE16) {
			*((unsigned short *)dst) = g; dst += 2;
		} else if (csp==M_XXXA) {
			*((unsigned char *)dst) = g >> 8; dst+=4;
		} else if (csp==M_XXXXXXAA) {
			*((unsigned short *)dst) = g; dst += 8;
		} else if (csp==M_RGB) {
			*((unsigned char *)dst) = r >> 8; dst++;
			*((unsigned char *)dst) = g >> 8; dst++;
			*((unsigned char *)dst) = b >> 8; dst++;
		} else if (csp==M_RGBA) {
			*((unsigned char *)dst) = r >> 8; dst++;
			*((unsigned char *)dst) = g >> 8; dst++;
			*((unsigned char *)dst) = b >> 8; dst++;
			*((unsigned char *)dst) = 255; dst++;
		} else if (csp==M_RRGGBB) {
			*((unsigned short *)dst) = r; dst+=2;
			*((unsigned short *)dst) = g; dst+=2;
			*((unsigned short *)dst) = b; dst+=2;
		} else if (csp==M_RRGGBBAA) {
			*((unsigned short *)dst) = r; dst+=2;
			*((unsigned short *)dst) = g; dst+=2;
			*((unsigned short *)dst) = b; dst+=2;
			*((unsigned short *)dst) = 65535; dst+=2;
		} else if (csp==M_RGB15) {
			*((unsigned short *)dst) =
				((r>>1)&0x7c00)|((g>>6)&0x03e0)|(b>>11);
			dst+=2;
		} else if (csp==M_RGB16) {
			*((unsigned short *)dst) =
				(r&0xf800)|((g>>5)&0x07e0)|(b>>11);
			dst+=2;
		} else if (csp==M_BGR) {
			*((unsigned char *)dst) = b >> 8; dst++;
			*((unsigned char *)dst) = g >> 8; dst++;
			*((unsigned char *)dst) = r >> 8; dst++;
		} else if (csp==M_BGRA) {
			*((unsigned char *)dst) = b >> 8; dst++;
			*((unsigned char *)dst) = g >> 8; dst++;
			*((unsigned char *)dst) = r >> 8; dst++;
			*((unsigned char *)dst) = 255; dst++;
		}
		n--;
		if (c<=0) {
			c = w;
			if (pad) {
				dst += pad;
				if (csp==M_BITMAP_LE) dstshift = 1;
				else if (csp==M_BITMAP_BE) dstshift = 128;
			}
		}
	}
	return((void *)s-(void *)src);
}


// ********************************************
// Writer
// ********************************************
int pnm_write(char *dst, int size, void *src, int w, int h, int csp)
{
	char *d;
	int r, srcshift, dstshift, pad;
	
	if ((csp==M_BITMAP_LE)||(csp==M_BITMAP_BE))
		r = snprintf(dst, size, "P4\n# PBM (bitmap file)\n%d %d\n",
			w, h);
	else if ((csp==M_GRAYSCALE8)||(csp==M_XXXA))
		r = snprintf(dst, size,
			"P5\n# PGM (grayscale file)\n%d %d\n255\n", w, h);
	else if ((csp==M_GRAYSCALE16)||(csp==M_XXXXXXAA))
		r = snprintf(dst, size,
			"P5\n# PGM (grayscale file)\n%d %d\n65535\n", w, h);
	else if ((csp==M_RGB)||(csp==M_RGBA)||(csp==M_RGB15)
	||(csp==M_RGB16)||(csp==M_BGR)||(csp==M_BGRA))
		r = snprintf(dst, size,
			"P6\n# PPM (pixmap file)\n%d %d\n255\n", w, h);
	else if ((csp==M_RRGGBB)||(csp==M_RRGGBBAA))
		r = snprintf(dst, size,
			"P6\n# PPM (pixmap file)\n%d %d\n65535\n", w, h);
	else return(-1);
	
	if ((r<=0)||(r>=size)) return(-1);
	size -= r;
	
	pad = pnm_pad(csp, w);
	d = dst + r;
	h *= w;
	r = w;
	if (csp == M_XXXA) src += 3;
	else if (csp == M_XXXXXXAA) src += 6;
	else if (csp == M_BITMAP_BE) { srcshift = 128; dstshift = 128; }
	else if (csp == M_BITMAP_LE) { srcshift = 1; dstshift = 128; }

	while (h > 0) {
		r--;
		if (csp == M_BITMAP_BE) {
			if (*((unsigned char *)src) & srcshift)
				*((unsigned char *)d) &= ~dstshift;
			else	*((unsigned char *)d) |= dstshift;
			srcshift >>= 1;
			if (srcshift == 0) {
				srcshift = 128;
				src++;
			}
			dstshift >>= 1;
			if ((dstshift == 0) || (r == 0)) {
				dstshift = 128;
				d++;
				size--;
				if (size<=0) return(-1);
			}
		} else if (csp == M_BITMAP_LE) {
			if (*((unsigned char *)src) & srcshift)
				*((unsigned char *)d) &= ~dstshift;
			else	*((unsigned char *)d) |= dstshift;
			srcshift <<= 1;
			if (srcshift > 128) {
				srcshift = 1;
				src++;
			}
			dstshift >>= 1;
			if ((dstshift == 0) || (r == 0)) {
				dstshift = 128;
				d++;
				size--;
				if (size<=0) return(-1);
			}
		} else if (csp == M_GRAYSCALE8) {
			size--; if (size<0) return(-1);
			*((unsigned char *)d) = *((unsigned char *)src);
			src++; d++;
		} else if (csp == M_GRAYSCALE16) {
			size-=2; if (size<0) return(-1);
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=2; d+=2;
		} else if (csp == M_XXXA) {
			size--; if (size<0) return(-1);
			*((unsigned char *)d) = *((unsigned char *)src);
			src+=4; d++;
		} else if (csp == M_XXXXXXAA) {
			size-=2; if (size<0) return(-1);
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=8; d+=2;
		} else if (csp == M_RGB) {
			size-=3; if (size<0) return(-1);
			*((unsigned char *)d) = *((unsigned char *)src);
			src++; d++;
			*((unsigned char *)d) = *((unsigned char *)src);
			src++; d++;
			*((unsigned char *)d) = *((unsigned char *)src);
			src++; d++;
		} else if (csp == M_RGBA) {
			size-=3; if (size<0) return(-1);
			*((unsigned char *)d) = *((unsigned char *)src);
			src++; d++;
			*((unsigned char *)d) = *((unsigned char *)src);
			src++; d++;
			*((unsigned char *)d) = *((unsigned char *)src);
			src+=2; d++;
		} else if (csp == M_RRGGBB) {
			size-=6; if (size<0) return(-1);
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=2; d+=2;
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=2; d+=2;
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=2; d+=2;
		} else if (csp == M_RRGGBBAA) {
			size-=6; if (size<0) return(-1);
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=2; d+=2;
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=2; d+=2;
			*((unsigned short *)d) = *((unsigned short *)src);
			src+=4; d+=2;
		} else if (csp == M_RGB15) {
			size-=3; if (size<0) return(-1);
			r = *((unsigned short *)src); src+=2;
			*((unsigned char *)d) = ((r&0x7c00)>>10)*255/31; d++;
			*((unsigned char *)d) = ((r&0x03e0)>>5)*255/31; d++;
			*((unsigned char *)d) = (r&0x001f)*255/31; d++;
		} else if (csp == M_RGB16) {
			size-=3; if (size<0) return(-1);
			r = *((unsigned short *)src); src+=2;
			*((unsigned char *)d) = (r>>11)*255/31; d++;
			*((unsigned char *)d) = ((r&0x07e0)>>5)*255/63; d++;
			*((unsigned char *)d) = (r&0x001f)*255/31; d++;
		} else if (csp == M_BGR) {
			size-=3; if (size<0) return(-1);
			*((unsigned char *)d)=((unsigned char *)src)[3]; d++;
			*((unsigned char *)d)=((unsigned char *)src)[2]; d++;
			*((unsigned char *)d)=((unsigned char *)src)[1]; d++;
			src += 3;
		} else if (csp == M_BGRA) {
			size-=3; if (size<0) return(-1);
			*((unsigned char *)d)=((unsigned char *)src)[3]; d++;
			*((unsigned char *)d)=((unsigned char *)src)[2]; d++;
			*((unsigned char *)d)=((unsigned char *)src)[1]; d++;
			src += 4;
		}
		h--;
		if (r <= 0) {
			r = w;
			if (pad) {
				src += pad;
				if (csp==M_BITMAP_BE) srcshift = 128;
				else if (csp==M_BITMAP_LE) srcshift = 1;
			}
		}
	}
	return((void *)d - (void *)dst);
}

// ********************************************
// Reverser
// ********************************************
void pnm_fliplines(void *data, int lines, int linesize)
{
	char *d2, t;
	int c;

	d2 = data + ((lines-1) * linesize);
	lines >>= 1;
	while (lines > 0) {
		c = linesize;
		while (c) {
			t = *((char *)data);
			*((char *)data) = *d2;
			*d2 = t;
			data++;
			d2++;
			c--;
		}
		d2 -= linesize<<1;
		lines--;
	}
}
